"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheckType = exports.RoutingPolicy = exports.DnsRecordType = exports.Service = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alias_target_instance_1 = require("./alias-target-instance");
const cname_instance_1 = require("./cname-instance");
const ip_instance_1 = require("./ip-instance");
const namespace_1 = require("./namespace");
const non_ip_instance_1 = require("./non-ip-instance");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
class ServiceBase extends core_1.Resource {
}
/**
 * (experimental) Define a CloudMap Service.
 *
 * @experimental
 */
class Service extends ServiceBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const namespaceType = props.namespace.type;
        // Validations
        if (namespaceType === namespace_1.NamespaceType.HTTP && (props.routingPolicy || props.dnsRecordType)) {
            throw new Error('Cannot specify `routingPolicy` or `dnsRecord` when using an HTTP namespace.');
        }
        if (props.healthCheck && props.customHealthCheck) {
            throw new Error('Cannot specify both `healthCheckConfig` and `healthCheckCustomConfig`.');
        }
        if (namespaceType === namespace_1.NamespaceType.DNS_PRIVATE && props.healthCheck) {
            throw new Error('Cannot specify `healthCheckConfig` for a Private DNS namespace.');
        }
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.dnsRecordType === DnsRecordType.CNAME) {
            throw new Error('Cannot use `CNAME` record when routing policy is `Multivalue`.');
        }
        // Additional validation for eventual attachment of LBs
        // The same validation happens later on during the actual attachment
        // of LBs, but we need the property for the correct configuration of
        // routingPolicy anyway, so might as well do the validation as well.
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.loadBalancer) {
            throw new Error('Cannot register loadbalancers when routing policy is `Multivalue`.');
        }
        if (props.healthCheck
            && props.healthCheck.type === HealthCheckType.TCP
            && props.healthCheck.resourcePath) {
            throw new Error('Cannot specify `resourcePath` when using a `TCP` health check.');
        }
        // Set defaults where necessary
        const routingPolicy = (props.dnsRecordType === DnsRecordType.CNAME) || props.loadBalancer
            ? RoutingPolicy.WEIGHTED
            : RoutingPolicy.MULTIVALUE;
        const dnsRecordType = props.dnsRecordType || DnsRecordType.A;
        if (props.loadBalancer
            && (!(dnsRecordType === DnsRecordType.A
                || dnsRecordType === DnsRecordType.AAAA
                || dnsRecordType === DnsRecordType.A_AAAA))) {
            throw new Error('Must support `A` or `AAAA` records to register loadbalancers.');
        }
        const dnsConfig = props.namespace.type === namespace_1.NamespaceType.HTTP
            ? undefined
            : {
                dnsRecords: renderDnsRecords(dnsRecordType, props.dnsTtl),
                namespaceId: props.namespace.namespaceId,
                routingPolicy,
            };
        const healthCheckConfigDefaults = {
            type: HealthCheckType.HTTP,
            failureThreshold: 1,
            resourcePath: props.healthCheck && props.healthCheck.type !== HealthCheckType.TCP
                ? '/'
                : undefined,
        };
        const healthCheckConfig = props.healthCheck && { ...healthCheckConfigDefaults, ...props.healthCheck };
        const healthCheckCustomConfig = props.customHealthCheck;
        // Create service
        const service = new servicediscovery_generated_1.CfnService(this, 'Resource', {
            name: props.name,
            description: props.description,
            dnsConfig,
            healthCheckConfig,
            healthCheckCustomConfig,
            namespaceId: props.namespace.namespaceId,
        });
        this.serviceName = service.attrName;
        this.serviceArn = service.attrArn;
        this.serviceId = service.attrId;
        this.namespace = props.namespace;
        this.dnsRecordType = dnsRecordType;
        this.routingPolicy = routingPolicy;
    }
    /**
     * @experimental
     */
    static fromServiceAttributes(scope, id, attrs) {
        class Import extends ServiceBase {
            constructor() {
                super(...arguments);
                this.namespace = attrs.namespace;
                this.serviceId = attrs.serviceId;
                this.serviceArn = attrs.serviceArn;
                this.dnsRecordType = attrs.dnsRecordType;
                this.routingPolicy = attrs.routingPolicy;
                this.serviceName = attrs.serviceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Registers an ELB as a new instance with unique name instanceId in this service.
     *
     * @experimental
     */
    registerLoadBalancer(id, loadBalancer, customAttributes) {
        return new alias_target_instance_1.AliasTargetInstance(this, id, {
            service: this,
            dnsName: loadBalancer.loadBalancerDnsName,
            customAttributes,
        });
    }
    /**
     * (experimental) Registers a resource that is accessible using values other than an IP address or a domain name (CNAME).
     *
     * @experimental
     */
    registerNonIpInstance(id, props) {
        return new non_ip_instance_1.NonIpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * (experimental) Registers a resource that is accessible using an IP address.
     *
     * @experimental
     */
    registerIpInstance(id, props) {
        return new ip_instance_1.IpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * (experimental) Registers a resource that is accessible using a CNAME.
     *
     * @experimental
     */
    registerCnameInstance(id, props) {
        return new cname_instance_1.CnameInstance(this, id, {
            service: this,
            ...props,
        });
    }
}
exports.Service = Service;
function renderDnsRecords(dnsRecordType, dnsTtl = core_1.Duration.minutes(1)) {
    const ttl = dnsTtl.toSeconds();
    if (dnsRecordType === DnsRecordType.A_AAAA) {
        return [{
                type: DnsRecordType.A,
                ttl,
            }, {
                type: DnsRecordType.AAAA,
                ttl,
            }];
    }
    else {
        return [{ type: dnsRecordType, ttl }];
    }
}
/**
 * @experimental
 */
var DnsRecordType;
(function (DnsRecordType) {
    DnsRecordType["A"] = "A";
    DnsRecordType["AAAA"] = "AAAA";
    DnsRecordType["A_AAAA"] = "A, AAAA";
    DnsRecordType["SRV"] = "SRV";
    DnsRecordType["CNAME"] = "CNAME";
})(DnsRecordType = exports.DnsRecordType || (exports.DnsRecordType = {}));
/**
 * @experimental
 */
var RoutingPolicy;
(function (RoutingPolicy) {
    RoutingPolicy["WEIGHTED"] = "WEIGHTED";
    RoutingPolicy["MULTIVALUE"] = "MULTIVALUE";
})(RoutingPolicy = exports.RoutingPolicy || (exports.RoutingPolicy = {}));
/**
 * @experimental
 */
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["HTTP"] = "HTTP";
    HealthCheckType["HTTPS"] = "HTTPS";
    HealthCheckType["TCP"] = "TCP";
})(HealthCheckType = exports.HealthCheckType || (exports.HealthCheckType = {}));
//# sourceMappingURL=data:application/json;base64,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