"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Expires = exports.StorageClass = exports.ServerSideEncryption = exports.CacheControl = exports.BucketDeployment = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const handlerCodeBundle = path.join(__dirname, '..', 'lambda', 'bundle.zip');
const handlerSourceDirectory = path.join(__dirname, '..', 'lambda', 'src');
/**
 * @experimental
 */
class BucketDeployment extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (props.distributionPaths && !props.distribution) {
            throw new Error('Distribution must be specified if distribution paths are specified');
        }
        const assetHash = calcSourceHash(handlerSourceDirectory);
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(handlerCodeBundle, { assetHash }),
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
        });
        const handlerRole = handler.role;
        if (!handlerRole) {
            throw new Error('lambda.SingletonFunction should have created a Role');
        }
        const sources = props.sources.map((source) => source.bind(this, { handlerRole }));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                Prune: (_a = props.prune) !== null && _a !== void 0 ? _a : true,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution ? props.distribution.distributionId : undefined,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
/**
 * We need a custom source hash calculation since the bundle.zip file
 * contains python dependencies installed during build and results in a
 * non-deterministic behavior.
 *
 * So we just take the `src/` directory of our custom resoruce code.
 */
function calcSourceHash(srcDir) {
    const sha = crypto.createHash('sha256');
    for (const file of fs.readdirSync(srcDir)) {
        const data = fs.readFileSync(path.join(srcDir, file));
        sha.update(`<file name=${file}>`);
        sha.update(data);
        sha.update('</file>');
    }
    return sha.digest('hex');
}
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase().startsWith('x-amzn-meta-')
        ? key.toLowerCase()
        : `x-amzn-meta-${key.toLowerCase()}`;
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.date.toUTCString();
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * (experimental) Used for HTTP cache-control header, which influences downstream caches.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @experimental
 */
class CacheControl {
    constructor(value) {
        this.value = value;
    }
    /**
     * @experimental
     */
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    /**
     * @experimental
     */
    static noCache() { return new CacheControl('no-cache'); }
    /**
     * @experimental
     */
    static noTransform() { return new CacheControl('no-transform'); }
    /**
     * @experimental
     */
    static setPublic() { return new CacheControl('public'); }
    /**
     * @experimental
     */
    static setPrivate() { return new CacheControl('private'); }
    /**
     * @experimental
     */
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    /**
     * @experimental
     */
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    /**
     * @experimental
     */
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    /**
     * @experimental
     */
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * (experimental) Indicates whether server-side encryption is enabled for the object, and whether that encryption is from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @experimental
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * (experimental) Storage class used for storing the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @experimental
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * (deprecated) Used for HTTP expires header, which influences downstream caches.
 *
 * Does NOT influence deletion of the object.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 * @deprecated use core.Expiration
 */
class Expires {
    constructor(value) {
        this.value = value;
    }
    /**
     * (deprecated) Expire at the specified date.
     *
     * @param d date to expire at.
     * @deprecated
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * (deprecated) Expire at the specified timestamp.
     *
     * @param t timestamp in unix milliseconds.
     * @deprecated
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * (deprecated) Expire once the specified duration has passed since deployment time.
     *
     * @param t the duration to wait before expiring.
     * @deprecated
     */
    static after(t) { return Expires.atDate(new Date(Date.now() + t.toMilliseconds())); }
    /**
     * @deprecated
     */
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,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