"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Asset = void 0;
const fs = require("fs");
const path = require("path");
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const compat_1 = require("./compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * (experimental) An asset represents a local file or directory, which is automatically uploaded to S3 and then can be referenced within a CDK application.
 *
 * @experimental
 */
class Asset extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: compat_1.toSymlinkFollow(props.follow),
            assetHash: (_a = props.assetHash) !== null && _a !== void 0 ? _a : props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        this.assetPath = staging.stagedPath;
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === cdk.FileAssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        const stack = cdk.Stack.of(this);
        const location = stack.synthesizer.addFileAsset({
            packaging,
            sourceHash: this.sourceHash,
            fileName: staging.stagedPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        const kmsKey = location.kmsKeyArn ? kms.Key.fromKeyArn(this, 'Key', location.kmsKeyArn) : undefined;
        this.bucket = s3.Bucket.fromBucketAttributes(this, 'AssetBucket', {
            bucketName: this.s3BucketName,
            encryptionKey: kmsKey,
        });
        for (const reader of ((_b = props.readers) !== null && _b !== void 0 ? _b : [])) {
            this.grantRead(reader);
        }
    }
    /**
     * (experimental) Adds CloudFormation template metadata to the specified resource with information that indicates which resource property is mapped to this local asset.
     *
     * This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface].
     * @param resourceProperty The property name where this asset is referenced (e.g. "Code" for AWS::Lambda::Function).
     * @see https://github.com/aws/aws-cdk/issues/1432
     * @experimental
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * (experimental) Grants read permissions to the principal on the assets bucket.
     *
     * @experimental
     */
    grantRead(grantee) {
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return cdk.FileAssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return cdk.FileAssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
//# sourceMappingURL=data:application/json;base64,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