"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseInstanceEngine = exports.SqlServerEngineVersion = exports.OracleEngineVersion = exports.OracleLegacyEngineVersion = exports.PostgresEngineVersion = exports.MysqlEngineVersion = exports.MariaDbEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const option_group_1 = require("./option-group");
class InstanceEngineBase {
    constructor(props) {
        var _a;
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.engineVersion = props.version;
        this.parameterGroupFamily = (_a = props.parameterGroupFamily) !== null && _a !== void 0 ? _a : (this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined);
        this.engineFamily = props.engineFamily;
    }
    bindToInstance(_scope, options) {
        if (options.timezone && !this.supportsTimezone) {
            throw new Error(`timezone property can not be configured for ${this.engineType}`);
        }
        return {
            features: this.features,
            optionGroup: options.optionGroup,
        };
    }
    /** Defines whether this Instance Engine can support timezone properties. */
    get supportsTimezone() { return false; }
}
/**
 * (experimental) The versions for the MariaDB instance engines (those returned by {@link DatabaseInstanceEngine.mariaDb}).
 *
 * @experimental
 */
class MariaDbEngineVersion {
    constructor(mariaDbFullVersion, mariaDbMajorVersion) {
        this.mariaDbFullVersion = mariaDbFullVersion;
        this.mariaDbMajorVersion = mariaDbMajorVersion;
    }
    /**
     * (experimental) Create a new MariaDbEngineVersion with an arbitrary version.
     *
     * @param mariaDbFullVersion the full version string, for example "10.5.28".
     * @param mariaDbMajorVersion the major version of the engine, for example "10.5".
     * @experimental
     */
    static of(mariaDbFullVersion, mariaDbMajorVersion) {
        return new MariaDbEngineVersion(mariaDbFullVersion, mariaDbMajorVersion);
    }
}
exports.MariaDbEngineVersion = MariaDbEngineVersion;
/**
 * (experimental) Version "10.0" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0 = MariaDbEngineVersion.of('10.0', '10.0');
/**
 * (experimental) Version "10.0.17".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_17 = MariaDbEngineVersion.of('10.0.17', '10.0');
/**
 * (experimental) Version "10.0.24".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_24 = MariaDbEngineVersion.of('10.0.24', '10.0');
/**
 * (experimental) Version "10.0.28".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_28 = MariaDbEngineVersion.of('10.0.28', '10.0');
/**
 * (experimental) Version "10.0.31".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_31 = MariaDbEngineVersion.of('10.0.31', '10.0');
/**
 * (experimental) Version "10.0.32".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_32 = MariaDbEngineVersion.of('10.0.32', '10.0');
/**
 * (experimental) Version "10.0.34".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_34 = MariaDbEngineVersion.of('10.0.34', '10.0');
/**
 * (experimental) Version "10.0.35".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_0_35 = MariaDbEngineVersion.of('10.0.35', '10.0');
/**
 * (experimental) Version "10.1" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1 = MariaDbEngineVersion.of('10.1', '10.1');
/**
 * (experimental) Version "10.1.14".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1_14 = MariaDbEngineVersion.of('10.1.14', '10.1');
/**
 * (experimental) Version "10.1.19".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1_19 = MariaDbEngineVersion.of('10.1.19', '10.1');
/**
 * (experimental) Version "10.1.23".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1_23 = MariaDbEngineVersion.of('10.1.23', '10.1');
/**
 * (experimental) Version "10.1.26".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1_26 = MariaDbEngineVersion.of('10.1.26', '10.1');
/**
 * (experimental) Version "10.1.31".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1_31 = MariaDbEngineVersion.of('10.1.31', '10.1');
/**
 * (experimental) Version "10.1.34".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_1_34 = MariaDbEngineVersion.of('10.1.34', '10.1');
/**
 * (experimental) Version "10.2" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_2 = MariaDbEngineVersion.of('10.2', '10.2');
/**
 * (experimental) Version "10.2.11".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_2_11 = MariaDbEngineVersion.of('10.2.11', '10.2');
/**
 * (experimental) Version "10.2.12".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_2_12 = MariaDbEngineVersion.of('10.2.12', '10.2');
/**
 * (experimental) Version "10.2.15".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_2_15 = MariaDbEngineVersion.of('10.2.15', '10.2');
/**
 * (experimental) Version "10.2.21".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_2_21 = MariaDbEngineVersion.of('10.2.21', '10.2');
/**
 * (experimental) Version "10.3" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_3 = MariaDbEngineVersion.of('10.3', '10.3');
/**
 * (experimental) Version "10.3.8".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_3_8 = MariaDbEngineVersion.of('10.3.8', '10.3');
/**
 * (experimental) Version "10.3.13".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_3_13 = MariaDbEngineVersion.of('10.3.13', '10.3');
/**
 * (experimental) Version "10.3.20".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_3_20 = MariaDbEngineVersion.of('10.3.20', '10.3');
/**
 * (experimental) Version "10.3.23".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_3_23 = MariaDbEngineVersion.of('10.3.23', '10.3');
/**
 * (experimental) Version "10.4" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_4 = MariaDbEngineVersion.of('10.4', '10.4');
/**
 * (experimental) Version "10.4.8".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_4_8 = MariaDbEngineVersion.of('10.4.8', '10.4');
/**
 * (experimental) Version "10.4.13".
 *
 * @experimental
 */
MariaDbEngineVersion.VER_10_4_13 = MariaDbEngineVersion.of('10.4.13', '10.4');
class MariaDbInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mariadb',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mariaDbFullVersion,
                    majorVersion: version.mariaDbMajorVersion,
                }
                : undefined,
        });
    }
    bindToInstance(scope, options) {
        if (options.domain) {
            throw new Error(`domain property cannot be configured for ${this.engineType}`);
        }
        return super.bindToInstance(scope, options);
    }
}
/**
 * (experimental) The versions for the MySQL instance engines (those returned by {@link DatabaseInstanceEngine.mysql}).
 *
 * @experimental
 */
class MysqlEngineVersion {
    constructor(mysqlFullVersion, mysqlMajorVersion) {
        this.mysqlFullVersion = mysqlFullVersion;
        this.mysqlMajorVersion = mysqlMajorVersion;
    }
    /**
     * (experimental) Create a new MysqlEngineVersion with an arbitrary version.
     *
     * @param mysqlFullVersion the full version string, for example "8.1.43".
     * @param mysqlMajorVersion the major version of the engine, for example "8.1".
     * @experimental
     */
    static of(mysqlFullVersion, mysqlMajorVersion) {
        return new MysqlEngineVersion(mysqlFullVersion, mysqlMajorVersion);
    }
}
exports.MysqlEngineVersion = MysqlEngineVersion;
/**
 * (experimental) Version "5.5" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_5 = MysqlEngineVersion.of('5.5', '5.5');
/**
 * (experimental) Version "5.5.46".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_5_46 = MysqlEngineVersion.of('5.5.46', '5.5');
/**
 * (experimental) Version "5.5.53".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_5_53 = MysqlEngineVersion.of('5.5.53', '5.5');
/**
 * (experimental) Version "5.5.57".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_5_57 = MysqlEngineVersion.of('5.5.57', '5.5');
/**
 * (experimental) Version "5.5.59".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_5_59 = MysqlEngineVersion.of('5.5.59', '5.5');
/**
 * (experimental) Version "5.5.61".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_5_61 = MysqlEngineVersion.of('5.5.61', '5.5');
/**
 * (experimental) Version "5.6" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6 = MysqlEngineVersion.of('5.6', '5.6');
/**
 * (experimental) Version "5.6.34".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_34 = MysqlEngineVersion.of('5.6.34', '5.6');
/**
 * (experimental) Version "5.6.35".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_35 = MysqlEngineVersion.of('5.6.35', '5.6');
/**
 * (experimental) Version "5.6.37".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_37 = MysqlEngineVersion.of('5.6.37', '5.6');
/**
 * (experimental) Version "5.6.39".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_39 = MysqlEngineVersion.of('5.6.39', '5.6');
/**
 * (experimental) Version "5.6.40".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_40 = MysqlEngineVersion.of('5.6.40', '5.6');
/**
 * (experimental) Version "5.6.41".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_41 = MysqlEngineVersion.of('5.6.41', '5.6');
/**
 * (experimental) Version "5.6.43".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_43 = MysqlEngineVersion.of('5.6.43', '5.6');
/**
 * (experimental) Version "5.6.44".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_44 = MysqlEngineVersion.of('5.6.44', '5.6');
/**
 * (experimental) Version "5.6.46".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_46 = MysqlEngineVersion.of('5.6.46', '5.6');
/**
 * (experimental) Version "5.6.48".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_6_48 = MysqlEngineVersion.of('5.6.48', '5.6');
/**
 * (experimental) Version "5.7" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7 = MysqlEngineVersion.of('5.7', '5.7');
/**
 * (experimental) Version "5.7.16".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_16 = MysqlEngineVersion.of('5.7.16', '5.7');
/**
 * (experimental) Version "5.7.17".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_17 = MysqlEngineVersion.of('5.7.17', '5.7');
/**
 * (experimental) Version "5.7.19".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_19 = MysqlEngineVersion.of('5.7.19', '5.7');
/**
 * (experimental) Version "5.7.21".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_21 = MysqlEngineVersion.of('5.7.21', '5.7');
/**
 * (experimental) Version "5.7.22".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_22 = MysqlEngineVersion.of('5.7.22', '5.7');
/**
 * (experimental) Version "5.7.23".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_23 = MysqlEngineVersion.of('5.7.23', '5.7');
/**
 * (experimental) Version "5.7.24".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_24 = MysqlEngineVersion.of('5.7.24', '5.7');
/**
 * (experimental) Version "5.7.25".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_25 = MysqlEngineVersion.of('5.7.25', '5.7');
/**
 * (experimental) Version "5.7.26".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_26 = MysqlEngineVersion.of('5.7.26', '5.7');
/**
 * (experimental) Version "5.7.28".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_28 = MysqlEngineVersion.of('5.7.28', '5.7');
/**
 * (experimental) Version "5.7.30".
 *
 * @experimental
 */
MysqlEngineVersion.VER_5_7_30 = MysqlEngineVersion.of('5.7.30', '5.7');
/**
 * (experimental) Version "8.0" (only a major version, without a specific minor version).
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0 = MysqlEngineVersion.of('8.0', '8.0');
/**
 * (experimental) Version "8.0.11".
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0_11 = MysqlEngineVersion.of('8.0.11', '8.0');
/**
 * (experimental) Version "8.0.13".
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0_13 = MysqlEngineVersion.of('8.0.13', '8.0');
/**
 * (experimental) Version "8.0.15".
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0_15 = MysqlEngineVersion.of('8.0.15', '8.0');
/**
 * (experimental) Version "8.0.16".
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0_16 = MysqlEngineVersion.of('8.0.16', '8.0');
/**
 * (experimental) Version "8.0.17".
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0_17 = MysqlEngineVersion.of('8.0.17', '8.0');
/**
 * (experimental) Version "8.0.19".
 *
 * @experimental
 */
MysqlEngineVersion.VER_8_0_19 = MysqlEngineVersion.of('8.0.19', '8.0');
class MySqlInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'mysql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.mysqlFullVersion,
                    majorVersion: version.mysqlMajorVersion,
                }
                : undefined,
            engineFamily: 'MYSQL',
        });
    }
}
/**
 * (experimental) The versions for the PostgreSQL instance engines (those returned by {@link DatabaseInstanceEngine.postgres}).
 *
 * @experimental
 */
class PostgresEngineVersion {
    constructor(postgresFullVersion, postgresMajorVersion, postgresFeatures) {
        this.postgresFullVersion = postgresFullVersion;
        this.postgresMajorVersion = postgresMajorVersion;
        this._features = {
            s3Import: (postgresFeatures === null || postgresFeatures === void 0 ? void 0 : postgresFeatures.s3Import) ? 's3Import' : undefined,
        };
    }
    /**
     * (experimental) Create a new PostgresEngineVersion with an arbitrary version.
     *
     * @param postgresFullVersion the full version string, for example "13.11".
     * @param postgresMajorVersion the major version of the engine, for example "13".
     * @experimental
     */
    static of(postgresFullVersion, postgresMajorVersion, postgresFeatures) {
        return new PostgresEngineVersion(postgresFullVersion, postgresMajorVersion, postgresFeatures);
    }
}
exports.PostgresEngineVersion = PostgresEngineVersion;
/**
 * (experimental) Version "9.5" (only a major version, without a specific minor version).
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5 = PostgresEngineVersion.of('9.5', '9.5');
/**
 * (experimental) Version "9.5.2".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_2 = PostgresEngineVersion.of('9.5.2', '9.5');
/**
 * (experimental) Version "9.5.4".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_4 = PostgresEngineVersion.of('9.5.4', '9.5');
/**
 * (experimental) Version "9.5.6".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_6 = PostgresEngineVersion.of('9.5.6', '9.5');
/**
 * (experimental) Version "9.5.7".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_7 = PostgresEngineVersion.of('9.5.7', '9.5');
/**
 * (experimental) Version "9.5.9".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_9 = PostgresEngineVersion.of('9.5.9', '9.5');
/**
 * (experimental) Version "9.5.10".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_10 = PostgresEngineVersion.of('9.5.10', '9.5');
/**
 * (experimental) Version "9.5.12".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_12 = PostgresEngineVersion.of('9.5.12', '9.5');
/**
 * (experimental) Version "9.5.13".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_13 = PostgresEngineVersion.of('9.5.13', '9.5');
/**
 * (experimental) Version "9.5.14".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_14 = PostgresEngineVersion.of('9.5.14', '9.5');
/**
 * (experimental) Version "9.5.15".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_15 = PostgresEngineVersion.of('9.5.15', '9.5');
/**
 * (experimental) Version "9.5.16".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_16 = PostgresEngineVersion.of('9.5.16', '9.5');
/**
 * (experimental) Version "9.5.18".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_18 = PostgresEngineVersion.of('9.5.18', '9.5');
/**
 * (experimental) Version "9.5.19".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_19 = PostgresEngineVersion.of('9.5.19', '9.5');
/**
 * (experimental) Version "9.5.20".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_20 = PostgresEngineVersion.of('9.5.20', '9.5');
/**
 * (experimental) Version "9.5.21".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_21 = PostgresEngineVersion.of('9.5.21', '9.5');
/**
 * (experimental) Version "9.5.22".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_5_22 = PostgresEngineVersion.of('9.5.22', '9.5');
/**
 * (experimental) Version "9.6" (only a major version, without a specific minor version).
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6 = PostgresEngineVersion.of('9.6', '9.6');
/**
 * (experimental) Version "9.6.1".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_1 = PostgresEngineVersion.of('9.6.1', '9.6');
/**
 * (experimental) Version "9.6.2".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_2 = PostgresEngineVersion.of('9.6.2', '9.6');
/**
 * (experimental) Version "9.6.3".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_3 = PostgresEngineVersion.of('9.6.3', '9.6');
/**
 * (experimental) Version "9.6.5".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_5 = PostgresEngineVersion.of('9.6.5', '9.6');
/**
 * (experimental) Version "9.6.6".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_6 = PostgresEngineVersion.of('9.6.6', '9.6');
/**
 * (experimental) Version "9.6.8".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_8 = PostgresEngineVersion.of('9.6.8', '9.6');
/**
 * (experimental) Version "9.6.9".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_9 = PostgresEngineVersion.of('9.6.9', '9.6');
/**
 * (experimental) Version "9.6.10".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_10 = PostgresEngineVersion.of('9.6.10', '9.6');
/**
 * (experimental) Version "9.6.11".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_11 = PostgresEngineVersion.of('9.6.11', '9.6');
/**
 * (experimental) Version "9.6.12".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_12 = PostgresEngineVersion.of('9.6.12', '9.6');
/**
 * (experimental) Version "9.6.14".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_14 = PostgresEngineVersion.of('9.6.14', '9.6');
/**
 * (experimental) Version "9.6.15".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_15 = PostgresEngineVersion.of('9.6.15', '9.6');
/**
 * (experimental) Version "9.6.16".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_16 = PostgresEngineVersion.of('9.6.16', '9.6');
/**
 * (experimental) Version "9.6.17".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_17 = PostgresEngineVersion.of('9.6.17', '9.6');
/**
 * (experimental) Version "9.6.18".
 *
 * @experimental
 */
PostgresEngineVersion.VER_9_6_18 = PostgresEngineVersion.of('9.6.18', '9.6');
/**
 * (experimental) Version "10" (only a major version, without a specific minor version).
 *
 * @experimental
 */
PostgresEngineVersion.VER_10 = PostgresEngineVersion.of('10', '10');
/**
 * (experimental) Version "10.1".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_1 = PostgresEngineVersion.of('10.1', '10');
/**
 * (experimental) Version "10.3".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_3 = PostgresEngineVersion.of('10.3', '10');
/**
 * (experimental) Version "10.4".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_4 = PostgresEngineVersion.of('10.4', '10');
/**
 * (experimental) Version "10.5".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_5 = PostgresEngineVersion.of('10.5', '10');
/**
 * (experimental) Version "10.6".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_6 = PostgresEngineVersion.of('10.6', '10');
/**
 * (experimental) Version "10.7".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_7 = PostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 * (experimental) Version "10.9".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_9 = PostgresEngineVersion.of('10.9', '10', { s3Import: true });
/**
 * (experimental) Version "10.10".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_10 = PostgresEngineVersion.of('10.10', '10', { s3Import: true });
/**
 * (experimental) Version "10.11".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_11 = PostgresEngineVersion.of('10.11', '10', { s3Import: true });
/**
 * (experimental) Version "10.12".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_12 = PostgresEngineVersion.of('10.12', '10', { s3Import: true });
/**
 * (experimental) Version "10.13".
 *
 * @experimental
 */
PostgresEngineVersion.VER_10_13 = PostgresEngineVersion.of('10.13', '10', { s3Import: true });
/**
 * (experimental) Version "11" (only a major version, without a specific minor version).
 *
 * @experimental
 */
PostgresEngineVersion.VER_11 = PostgresEngineVersion.of('11', '11', { s3Import: true });
/**
 * (experimental) Version "11.1".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_1 = PostgresEngineVersion.of('11.1', '11', { s3Import: true });
/**
 * (experimental) Version "11.2".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_2 = PostgresEngineVersion.of('11.2', '11', { s3Import: true });
/**
 * (experimental) Version "11.4".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_4 = PostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 * (experimental) Version "11.5".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_5 = PostgresEngineVersion.of('11.5', '11', { s3Import: true });
/**
 * (experimental) Version "11.6".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_6 = PostgresEngineVersion.of('11.6', '11', { s3Import: true });
/**
 * (experimental) Version "11.7".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_7 = PostgresEngineVersion.of('11.7', '11', { s3Import: true });
/**
 * (experimental) Version "11.8".
 *
 * @experimental
 */
PostgresEngineVersion.VER_11_8 = PostgresEngineVersion.of('11.8', '11', { s3Import: true });
/**
 * (experimental) Version "12" (only a major version, without a specific minor version).
 *
 * @experimental
 */
PostgresEngineVersion.VER_12 = PostgresEngineVersion.of('12', '12', { s3Import: true });
/**
 * (experimental) Version "12.2".
 *
 * @experimental
 */
PostgresEngineVersion.VER_12_2 = PostgresEngineVersion.of('12.2', '12', { s3Import: true });
/**
 * (experimental) Version "12.3".
 *
 * @experimental
 */
PostgresEngineVersion.VER_12_3 = PostgresEngineVersion.of('12.3', '12', { s3Import: true });
/**
 * The instance engine for PostgreSQL.
 */
class PostgresInstanceEngine extends InstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'postgres',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            version: version
                ? {
                    fullVersion: version.postgresFullVersion,
                    majorVersion: version.postgresMajorVersion,
                }
                : undefined,
            features: version ? version === null || version === void 0 ? void 0 : version._features : { s3Import: 's3Import' },
            engineFamily: 'POSTGRESQL',
        });
        this.defaultUsername = 'postgres';
    }
}
/**
 * (deprecated) The versions for the legacy Oracle instance engines (those returned by {@link DatabaseInstanceEngine.oracleSe} and {@link DatabaseInstanceEngine.oracleSe1}). Note: RDS will stop allowing creating new databases with this version in August 2020.
 *
 * @deprecated instances can no longer be created with these engine versions. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
class OracleLegacyEngineVersion {
    constructor(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        this.oracleLegacyFullVersion = oracleLegacyFullVersion;
        this.oracleLegacyMajorVersion = oracleLegacyMajorVersion;
    }
    static of(oracleLegacyFullVersion, oracleLegacyMajorVersion) {
        return new OracleLegacyEngineVersion(oracleLegacyFullVersion, oracleLegacyMajorVersion);
    }
}
exports.OracleLegacyEngineVersion = OracleLegacyEngineVersion;
/**
 * (deprecated) Version "11.2" (only a major version, without a specific minor version).
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2 = OracleLegacyEngineVersion.of('11.2', '11.2');
/**
 * (deprecated) Version "11.2.0.2.v2".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_2_V2 = OracleLegacyEngineVersion.of('11.2.0.2.v2', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v1".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V1 = OracleLegacyEngineVersion.of('11.2.0.4.v1', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v3".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V3 = OracleLegacyEngineVersion.of('11.2.0.4.v3', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v4".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V4 = OracleLegacyEngineVersion.of('11.2.0.4.v4', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v5".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V5 = OracleLegacyEngineVersion.of('11.2.0.4.v5', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v6".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V6 = OracleLegacyEngineVersion.of('11.2.0.4.v6', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v7".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V7 = OracleLegacyEngineVersion.of('11.2.0.4.v7', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v8".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V8 = OracleLegacyEngineVersion.of('11.2.0.4.v8', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v9".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V9 = OracleLegacyEngineVersion.of('11.2.0.4.v9', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v10".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V10 = OracleLegacyEngineVersion.of('11.2.0.4.v10', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v11".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V11 = OracleLegacyEngineVersion.of('11.2.0.4.v11', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v12".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V12 = OracleLegacyEngineVersion.of('11.2.0.4.v12', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v13".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V13 = OracleLegacyEngineVersion.of('11.2.0.4.v13', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v14".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V14 = OracleLegacyEngineVersion.of('11.2.0.4.v14', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v15".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V15 = OracleLegacyEngineVersion.of('11.2.0.4.v15', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v16".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V16 = OracleLegacyEngineVersion.of('11.2.0.4.v16', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v17".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V17 = OracleLegacyEngineVersion.of('11.2.0.4.v17', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v18".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V18 = OracleLegacyEngineVersion.of('11.2.0.4.v18', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v19".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V19 = OracleLegacyEngineVersion.of('11.2.0.4.v19', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v20".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V20 = OracleLegacyEngineVersion.of('11.2.0.4.v20', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v21".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V21 = OracleLegacyEngineVersion.of('11.2.0.4.v21', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v22".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V22 = OracleLegacyEngineVersion.of('11.2.0.4.v22', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v23".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V23 = OracleLegacyEngineVersion.of('11.2.0.4.v23', '11.2');
/**
 * (deprecated) Version "11.2.0.4.v24".
 *
 * @deprecated
 */
OracleLegacyEngineVersion.VER_11_2_0_4_V24 = OracleLegacyEngineVersion.of('11.2.0.4.v24', '11.2');
/**
 * (experimental) The versions for the Oracle instance engines (those returned by {@link DatabaseInstanceEngine.oracleSe2} and {@link DatabaseInstanceEngine.oracleEe}).
 *
 * @experimental
 */
class OracleEngineVersion {
    constructor(oracleFullVersion, oracleMajorVersion) {
        this.oracleFullVersion = oracleFullVersion;
        this.oracleMajorVersion = oracleMajorVersion;
    }
    /**
     * (experimental) Creates a new OracleEngineVersion with an arbitrary version.
     *
     * @param oracleFullVersion the full version string, for example "19.0.0.0.ru-2019-10.rur-2019-10.r1".
     * @param oracleMajorVersion the major version of the engine, for example "19".
     * @experimental
     */
    static of(oracleFullVersion, oracleMajorVersion) {
        return new OracleEngineVersion(oracleFullVersion, oracleMajorVersion);
    }
}
exports.OracleEngineVersion = OracleEngineVersion;
/**
 * (experimental) Version "12.1" (only a major version, without a specific minor version).
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1 = OracleEngineVersion.of('12.1', '12.1');
/**
 * (experimental) Version "12.1.0.2.v1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V1 = OracleEngineVersion.of('12.1.0.2.v1', '12.1');
/**
 * (experimental) Version "12.1.0.2.v2".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V2 = OracleEngineVersion.of('12.1.0.2.v2', '12.1');
/**
 * (experimental) Version "12.1.0.2.v3".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V3 = OracleEngineVersion.of('12.1.0.2.v3', '12.1');
/**
 * (experimental) Version "12.1.0.2.v4".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V4 = OracleEngineVersion.of('12.1.0.2.v4', '12.1');
/**
 * (experimental) Version "12.1.0.2.v5".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V5 = OracleEngineVersion.of('12.1.0.2.v5', '12.1');
/**
 * (experimental) Version "12.1.0.2.v6".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V6 = OracleEngineVersion.of('12.1.0.2.v6', '12.1');
/**
 * (experimental) Version "12.1.0.2.v7".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V7 = OracleEngineVersion.of('12.1.0.2.v7', '12.1');
/**
 * (experimental) Version "12.1.0.2.v8".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V8 = OracleEngineVersion.of('12.1.0.2.v8', '12.1');
/**
 * (experimental) Version "12.1.0.2.v9".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V9 = OracleEngineVersion.of('12.1.0.2.v9', '12.1');
/**
 * (experimental) Version "12.1.0.2.v10".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V10 = OracleEngineVersion.of('12.1.0.2.v10', '12.1');
/**
 * (experimental) Version "12.1.0.2.v11".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V11 = OracleEngineVersion.of('12.1.0.2.v11', '12.1');
/**
 * (experimental) Version "12.1.0.2.v12".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V12 = OracleEngineVersion.of('12.1.0.2.v12', '12.1');
/**
 * (experimental) Version "12.1.0.2.v13".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V13 = OracleEngineVersion.of('12.1.0.2.v13', '12.1');
/**
 * (experimental) Version "12.1.0.2.v14".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V14 = OracleEngineVersion.of('12.1.0.2.v14', '12.1');
/**
 * (experimental) Version "12.1.0.2.v15".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V15 = OracleEngineVersion.of('12.1.0.2.v15', '12.1');
/**
 * (experimental) Version "12.1.0.2.v16".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V16 = OracleEngineVersion.of('12.1.0.2.v16', '12.1');
/**
 * (experimental) Version "12.1.0.2.v17".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V17 = OracleEngineVersion.of('12.1.0.2.v17', '12.1');
/**
 * (experimental) Version "12.1.0.2.v18".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V18 = OracleEngineVersion.of('12.1.0.2.v18', '12.1');
/**
 * (experimental) Version "12.1.0.2.v19".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V19 = OracleEngineVersion.of('12.1.0.2.v19', '12.1');
/**
 * (experimental) Version "12.1.0.2.v20".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_1_0_2_V20 = OracleEngineVersion.of('12.1.0.2.v20', '12.1');
/**
 * (experimental) Version "12.2" (only a major version, without a specific minor version).
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2 = OracleEngineVersion.of('12.2', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2018-10.rur-2018-10.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2018_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2018-10.rur-2018-10.r1', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2019-01.rur-2019-01.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2019_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-01.rur-2019-01.r1', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2019-04.rur-2019-04.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2019_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-04.rur-2019-04.r1', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2019-07.rur-2019-07.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2019_07_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-07.rur-2019-07.r1', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2019-10.rur-2019-10.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2019_10_R1 = OracleEngineVersion.of('12.2.0.1.ru-2019-10.rur-2019-10.r1', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2020-01.rur-2020-01.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2020_01_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-01.rur-2020-01.r1', '12.2');
/**
 * (experimental) Version "12.2.0.1.ru-2020-04.rur-2020-04.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_12_2_0_1_2020_04_R1 = OracleEngineVersion.of('12.2.0.1.ru-2020-04.rur-2020-04.r1', '12.2');
/**
 * (experimental) Version "18" (only a major version, without a specific minor version).
 *
 * @experimental
 */
OracleEngineVersion.VER_18 = OracleEngineVersion.of('18', '18');
/**
 * (experimental) Version "18.0.0.0.ru-2019-07.rur-2019-07.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_18_0_0_0_2019_07_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-07.rur-2019-07.r1', '18');
/**
 * (experimental) Version "18.0.0.0.ru-2019-10.rur-2019-10.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_18_0_0_0_2019_10_R1 = OracleEngineVersion.of('18.0.0.0.ru-2019-10.rur-2019-10.r1', '18');
/**
 * (experimental) Version "18.0.0.0.ru-2020-01.rur-2020-01.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_18_0_0_0_2020_01_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-01.rur-2020-01.r1', '18');
/**
 * (experimental) Version "18.0.0.0.ru-2020-04.rur-2020-04.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_18_0_0_0_2020_04_R1 = OracleEngineVersion.of('18.0.0.0.ru-2020-04.rur-2020-04.r1', '18');
/**
 * (experimental) Version "19" (only a major version, without a specific minor version).
 *
 * @experimental
 */
OracleEngineVersion.VER_19 = OracleEngineVersion.of('19', '19');
/**
 * (experimental) Version "19.0.0.0.ru-2019-07.rur-2019-07.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_19_0_0_0_2019_07_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-07.rur-2019-07.r1', '19');
/**
 * (experimental) Version "19.0.0.0.ru-2019-10.rur-2019-10.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_19_0_0_0_2019_10_R1 = OracleEngineVersion.of('19.0.0.0.ru-2019-10.rur-2019-10.r1', '19');
/**
 * (experimental) Version "19.0.0.0.ru-2020-01.rur-2020-01.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_19_0_0_0_2020_01_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-01.rur-2020-01.r1', '19');
/**
 * (experimental) Version "19.0.0.0.ru-2020-04.rur-2020-04.r1".
 *
 * @experimental
 */
OracleEngineVersion.VER_19_0_0_0_2020_04_R1 = OracleEngineVersion.of('19.0.0.0.ru-2020-04.rur-2020-04.r1', '19');
class OracleInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER,
            parameterGroupFamily: props.version ? `${props.engineType}-${props.version.majorVersion}` : undefined,
            features: {
                s3Import: 'S3_INTEGRATION',
                s3Export: 'S3_INTEGRATION',
            },
        });
    }
    bindToInstance(scope, options) {
        const config = super.bindToInstance(scope, options);
        let optionGroup = options.optionGroup;
        if (options.s3ImportRole || options.s3ExportRole) {
            if (!optionGroup) {
                optionGroup = new option_group_1.OptionGroup(scope, 'InstanceOptionGroup', {
                    engine: this,
                    configurations: [],
                });
            }
            // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/oracle-s3-integration.html
            optionGroup.addConfiguration({
                name: 'S3_INTEGRATION',
                version: '1.0',
            });
        }
        return {
            ...config,
            optionGroup,
        };
    }
}
/** @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341 */
class OracleSeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
/** @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341 */
class OracleSe1InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se1',
            version: version
                ? {
                    fullVersion: version.oracleLegacyFullVersion,
                    majorVersion: version.oracleLegacyMajorVersion,
                }
                : {
                    majorVersion: '11.2',
                },
        });
    }
}
class OracleSe2InstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-se2',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
class OracleEeInstanceEngine extends OracleInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'oracle-ee',
            version: version
                ? {
                    fullVersion: version.oracleFullVersion,
                    majorVersion: version.oracleMajorVersion,
                }
                : undefined,
        });
    }
}
/**
 * (experimental) The versions for the SQL Server instance engines (those returned by {@link DatabaseInstanceEngine.sqlServerSe}, {@link DatabaseInstanceEngine.sqlServerEx}, {@link DatabaseInstanceEngine.sqlServerWeb} and {@link DatabaseInstanceEngine.sqlServerEe}).
 *
 * @experimental
 */
class SqlServerEngineVersion {
    constructor(sqlServerFullVersion, sqlServerMajorVersion) {
        this.sqlServerFullVersion = sqlServerFullVersion;
        this.sqlServerMajorVersion = sqlServerMajorVersion;
    }
    /**
     * (experimental) Create a new SqlServerEngineVersion with an arbitrary version.
     *
     * @param sqlServerFullVersion the full version string, for example "15.00.3049.1.v1".
     * @param sqlServerMajorVersion the major version of the engine, for example "15.00".
     * @experimental
     */
    static of(sqlServerFullVersion, sqlServerMajorVersion) {
        return new SqlServerEngineVersion(sqlServerFullVersion, sqlServerMajorVersion);
    }
}
exports.SqlServerEngineVersion = SqlServerEngineVersion;
/**
 * (experimental) Version "11.00" (only a major version, without a specific minor version).
 *
 * @experimental
 */
SqlServerEngineVersion.VER_11 = SqlServerEngineVersion.of('11.00', '11.00');
/**
 * (experimental) Version "11.00.5058.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_11_00_5058_0_V1 = SqlServerEngineVersion.of('11.00.5058.0.v1', '11.00');
/**
 * (experimental) Version "11.00.6020.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_11_00_6020_0_V1 = SqlServerEngineVersion.of('11.00.6020.0.v1', '11.00');
/**
 * (experimental) Version "11.00.6594.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_11_00_6594_0_V1 = SqlServerEngineVersion.of('11.00.6594.0.v1', '11.00');
/**
 * (experimental) Version "11.00.7462.6.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_11_00_7462_6_V1 = SqlServerEngineVersion.of('11.00.7462.6.v1', '11.00');
/**
 * (experimental) Version "11.00.7493.4.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_11_00_7493_4_V1 = SqlServerEngineVersion.of('11.00.7493.4.v1', '11.00');
/**
 * (experimental) Version "12.00" (only a major version, without a specific minor version).
 *
 * @experimental
 */
SqlServerEngineVersion.VER_12 = SqlServerEngineVersion.of('12.00', '12.00');
/**
 * (experimental) Version "12.00.5000.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_12_00_5000_0_V1 = SqlServerEngineVersion.of('12.00.5000.0.v1', '12.00');
/**
 * (experimental) Version "12.00.5546.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_12_00_5546_0_V1 = SqlServerEngineVersion.of('12.00.5546.0.v1', '12.00');
/**
 * (experimental) Version "12.00.5571.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_12_00_5571_0_V1 = SqlServerEngineVersion.of('12.00.5571.0.v1', '12.00');
/**
 * (experimental) Version "12.00.6293.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_12_00_6293_0_V1 = SqlServerEngineVersion.of('12.00.6293.0.v1', '12.00');
/**
 * (experimental) Version "12.00.6329.1.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_12_00_6329_1_V1 = SqlServerEngineVersion.of('12.00.6329.1.v1', '12.00');
/**
 * (experimental) Version "13.00" (only a major version, without a specific minor version).
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13 = SqlServerEngineVersion.of('13.00', '13.00');
/**
 * (experimental) Version "13.00.2164.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_2164_0_V1 = SqlServerEngineVersion.of('13.00.2164.0.v1', '13.00');
/**
 * (experimental) Version "13.00.4422.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_4422_0_V1 = SqlServerEngineVersion.of('13.00.4422.0.v1', '13.00');
/**
 * (experimental) Version "13.00.4451.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_4451_0_V1 = SqlServerEngineVersion.of('13.00.4451.0.v1', '13.00');
/**
 * (experimental) Version "13.00.4466.4.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_4466_4_V1 = SqlServerEngineVersion.of('13.00.4466.4.v1', '13.00');
/**
 * (experimental) Version "13.00.4522.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_4522_0_V1 = SqlServerEngineVersion.of('13.00.4522.0.v1', '13.00');
/**
 * (experimental) Version "13.00.5216.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_5216_0_V1 = SqlServerEngineVersion.of('13.00.5216.0.v1', '13.00');
/**
 * (experimental) Version "13.00.5292.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_5292_0_V1 = SqlServerEngineVersion.of('13.00.5292.0.v1', '13.00');
/**
 * (experimental) Version "13.00.5366.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_5366_0_V1 = SqlServerEngineVersion.of('13.00.5366.0.v1', '13.00');
/**
 * (experimental) Version "13.00.5426.0.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_5426_0_V1 = SqlServerEngineVersion.of('13.00.5426.0.v1', '13.00');
/**
 * (experimental) Version "13.00.5598.27.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_13_00_5598_27_V1 = SqlServerEngineVersion.of('13.00.5598.27.v1', '13.00');
/**
 * (experimental) Version "14.00" (only a major version, without a specific minor version).
 *
 * @experimental
 */
SqlServerEngineVersion.VER_14 = SqlServerEngineVersion.of('14.00', '14.00');
/**
 * (experimental) Version "14.00.1000.169.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_14_00_1000_169_V1 = SqlServerEngineVersion.of('14.00.1000.169.v1', '14.00');
/**
 * (experimental) Version "14.00.3015.40.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_14_00_3015_40_V1 = SqlServerEngineVersion.of('14.00.3015.40.v1', '14.00');
/**
 * (experimental) Version "14.00.3035.2.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_14_00_3035_2_V1 = SqlServerEngineVersion.of('14.00.3035.2.v1', '14.00');
/**
 * (experimental) Version "14.00.3049.1.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_14_00_3049_1_V1 = SqlServerEngineVersion.of('14.00.3049.1.v1', '14.00');
/**
 * (experimental) Version "14.00.3192.2.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_14_00_3192_2_V1 = SqlServerEngineVersion.of('14.00.3192.2.v1', '14.00');
/**
 * (experimental) Version "15.00" (only a major version, without a specific minor version).
 *
 * @experimental
 */
SqlServerEngineVersion.VER_15 = SqlServerEngineVersion.of('15.00', '15.00');
/**
 * (experimental) Version "15.00.4043.16.v1".
 *
 * @experimental
 */
SqlServerEngineVersion.VER_15_00_4043_16_V1 = SqlServerEngineVersion.of('15.00.4043.16.v1', '15.00');
class SqlServerInstanceEngineBase extends InstanceEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER,
            version: props.version
                ? {
                    fullVersion: props.version.sqlServerFullVersion,
                    majorVersion: props.version.sqlServerMajorVersion,
                }
                : undefined,
            parameterGroupFamily: props.version
                // for some reason, even though SQL Server major versions usually end in '.00',
                // the ParameterGroup family has to end in '.0'
                ? `${props.engineType}-${props.version.sqlServerMajorVersion.endsWith('.00')
                    ? props.version.sqlServerMajorVersion.slice(0, -1)
                    : props.version.sqlServerMajorVersion}`
                : undefined,
            features: {
                s3Import: 'S3_INTEGRATION',
                s3Export: 'S3_INTEGRATION',
            },
        });
    }
    bindToInstance(scope, options) {
        var _a;
        const config = super.bindToInstance(scope, options);
        let optionGroup = options.optionGroup;
        const s3Role = (_a = options.s3ImportRole) !== null && _a !== void 0 ? _a : options.s3ExportRole;
        if (s3Role) {
            if (options.s3ImportRole && options.s3ExportRole && options.s3ImportRole !== options.s3ExportRole) {
                throw new Error('S3 import and export roles must be the same for SQL Server engines');
            }
            if (!optionGroup) {
                optionGroup = new option_group_1.OptionGroup(scope, 'InstanceOptionGroup', {
                    engine: this,
                    configurations: [],
                });
            }
            // https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/Appendix.SQLServer.Options.BackupRestore.html
            optionGroup.addConfiguration({
                name: 'SQLSERVER_BACKUP_RESTORE',
                settings: { IAM_ROLE_ARN: s3Role.roleArn },
            });
        }
        return {
            ...config,
            optionGroup,
        };
    }
    get supportsTimezone() { return true; }
}
class SqlServerSeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-se',
            version,
        });
    }
}
class SqlServerExInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ex',
            version,
        });
    }
}
class SqlServerWebInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-web',
            version,
        });
    }
}
class SqlServerEeInstanceEngine extends SqlServerInstanceEngineBase {
    constructor(version) {
        super({
            engineType: 'sqlserver-ee',
            version,
        });
    }
}
/**
 * (experimental) A database instance engine.
 *
 * Provides mapping to DatabaseEngine used for
 * secret rotation.
 *
 * @experimental
 */
class DatabaseInstanceEngine {
    /**
     * (experimental) Creates a new MariaDB instance engine.
     *
     * @experimental
     */
    static mariaDb(props) {
        return new MariaDbInstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new MySQL instance engine.
     *
     * @experimental
     */
    static mysql(props) {
        return new MySqlInstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new PostgreSQL instance engine.
     *
     * @experimental
     */
    static postgres(props) {
        return new PostgresInstanceEngine(props.version);
    }
    /**
     * (deprecated) Creates a new Oracle Standard Edition instance engine.
     *
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe(props) {
        return new OracleSeInstanceEngine(props.version);
    }
    /**
     * (deprecated) Creates a new Oracle Standard Edition 1 instance engine.
     *
     * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
     */
    static oracleSe1(props) {
        return new OracleSe1InstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new Oracle Standard Edition 1 instance engine.
     *
     * @experimental
     */
    static oracleSe2(props) {
        return new OracleSe2InstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new Oracle Enterprise Edition instance engine.
     *
     * @experimental
     */
    static oracleEe(props) {
        return new OracleEeInstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new SQL Server Standard Edition instance engine.
     *
     * @experimental
     */
    static sqlServerSe(props) {
        return new SqlServerSeInstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new SQL Server Express Edition instance engine.
     *
     * @experimental
     */
    static sqlServerEx(props) {
        return new SqlServerExInstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new SQL Server Web Edition instance engine.
     *
     * @experimental
     */
    static sqlServerWeb(props) {
        return new SqlServerWebInstanceEngine(props.version);
    }
    /**
     * (experimental) Creates a new SQL Server Enterprise Edition instance engine.
     *
     * @experimental
     */
    static sqlServerEe(props) {
        return new SqlServerEeInstanceEngine(props.version);
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/**
 * (deprecated) The unversioned 'mariadb' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link mariaDb()} method
 */
DatabaseInstanceEngine.MARIADB = new MariaDbInstanceEngine();
/**
 * (deprecated) The unversioned 'mysql' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link mysql()} method
 */
DatabaseInstanceEngine.MYSQL = new MySqlInstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link oracleEe()} method
 */
DatabaseInstanceEngine.ORACLE_EE = new OracleEeInstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-se2' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link oracleSe2()} method
 */
DatabaseInstanceEngine.ORACLE_SE2 = new OracleSe2InstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-se1' instance engine.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
DatabaseInstanceEngine.ORACLE_SE1 = new OracleSe1InstanceEngine();
/**
 * (deprecated) The unversioned 'oracle-se' instance engine.
 *
 * @deprecated instances can no longer be created with this engine. See https://forums.aws.amazon.com/ann.jspa?annID=7341
 */
DatabaseInstanceEngine.ORACLE_SE = new OracleSeInstanceEngine();
/**
 * (deprecated) The unversioned 'postgres' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link postgres()} method
 */
DatabaseInstanceEngine.POSTGRES = new PostgresInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-ee' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerEe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EE = new SqlServerEeInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-se' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerSe()} method
 */
DatabaseInstanceEngine.SQL_SERVER_SE = new SqlServerSeInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-ex' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerEx()} method
 */
DatabaseInstanceEngine.SQL_SERVER_EX = new SqlServerExInstanceEngine();
/**
 * (deprecated) The unversioned 'sqlserver-web' instance engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 * We recommend using versioned engines created using the {@link sqlServerWeb()} method
 */
DatabaseInstanceEngine.SQL_SERVER_WEB = new SqlServerWebInstanceEngine();
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5zdGFuY2UtZW5naW5lLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW5zdGFuY2UtZW5naW5lLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUNBLDJEQUEyRCxDQUFDLDhEQUE4RDtBQUkxSCxpREFBMkQ7QUFnRzNELE1BQWUsa0JBQWtCO0lBUTdCLFlBQVksS0FBOEI7O1FBQ3RDLElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsQ0FBQztRQUNuQyxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsSUFBSSxDQUFDLDZCQUE2QixHQUFHLEtBQUssQ0FBQyw2QkFBNkIsQ0FBQztRQUN6RSxJQUFJLENBQUMsNEJBQTRCLEdBQUcsS0FBSyxDQUFDLDRCQUE0QixDQUFDO1FBQ3ZFLElBQUksQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUNuQyxJQUFJLENBQUMsb0JBQW9CLFNBQUcsS0FBSyxDQUFDLG9CQUFvQixtQ0FDbEQsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxHQUFHLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDOUYsSUFBSSxDQUFDLFlBQVksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDO0lBQzNDLENBQUM7SUFDTSxjQUFjLENBQUMsTUFBc0IsRUFBRSxPQUFrQztRQUM1RSxJQUFJLE9BQU8sQ0FBQyxRQUFRLElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDNUMsTUFBTSxJQUFJLEtBQUssQ0FBQywrQ0FBK0MsSUFBSSxDQUFDLFVBQVUsRUFBRSxDQUFDLENBQUM7U0FDckY7UUFDRCxPQUFPO1lBQ0gsUUFBUSxFQUFFLElBQUksQ0FBQyxRQUFRO1lBQ3ZCLFdBQVcsRUFBRSxPQUFPLENBQUMsV0FBVztTQUNuQyxDQUFDO0lBQ04sQ0FBQztJQUNELDRFQUE0RTtJQUM1RSxJQUFjLGdCQUFnQixLQUFLLE9BQU8sS0FBSyxDQUFDLENBQUMsQ0FBQztDQUNyRDs7Ozs7O0FBS0QsTUFBYSxvQkFBb0I7SUF3RTdCLFlBQW9CLGtCQUEwQixFQUFFLG1CQUEyQjtRQUN2RSxJQUFJLENBQUMsa0JBQWtCLEdBQUcsa0JBQWtCLENBQUM7UUFDN0MsSUFBSSxDQUFDLG1CQUFtQixHQUFHLG1CQUFtQixDQUFDO0lBQ25ELENBQUM7Ozs7Ozs7O0lBVk0sTUFBTSxDQUFDLEVBQUUsQ0FBQyxrQkFBMEIsRUFBRSxtQkFBMkI7UUFDcEUsT0FBTyxJQUFJLG9CQUFvQixDQUFDLGtCQUFrQixFQUFFLG1CQUFtQixDQUFDLENBQUM7SUFDN0UsQ0FBQzs7QUFuRUwsb0RBNEVDOzs7Ozs7QUExRTBCLDZCQUFRLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRW5ELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELDZCQUFRLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRW5ELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELDZCQUFRLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRW5ELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELDZCQUFRLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRW5ELCtCQUFVLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXZELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXpELDZCQUFRLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRW5ELCtCQUFVLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXZELGdDQUFXLEdBQUcsb0JBQW9CLENBQUMsRUFBRSxDQUFDLFNBQVMsRUFBRSxNQUFNLENBQUMsQ0FBQztBQTZCcEYsTUFBTSxxQkFBc0IsU0FBUSxrQkFBa0I7SUFDbEQsWUFBWSxPQUE4QjtRQUN0QyxLQUFLLENBQUM7WUFDRixVQUFVLEVBQUUsU0FBUztZQUNyQiw2QkFBNkIsRUFBRSxjQUFjLENBQUMseUJBQXlCLENBQUMsNEJBQTRCO1lBQ3BHLDRCQUE0QixFQUFFLGNBQWMsQ0FBQyx5QkFBeUIsQ0FBQywyQkFBMkI7WUFDbEcsT0FBTyxFQUFFLE9BQU87Z0JBQ1osQ0FBQyxDQUFDO29CQUNFLFdBQVcsRUFBRSxPQUFPLENBQUMsa0JBQWtCO29CQUN2QyxZQUFZLEVBQUUsT0FBTyxDQUFDLG1CQUFtQjtpQkFDNUM7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7U0FDbEIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNNLGNBQWMsQ0FBQyxLQUFxQixFQUFFLE9BQWtDO1FBQzNFLElBQUksT0FBTyxDQUFDLE1BQU0sRUFBRTtZQUNoQixNQUFNLElBQUksS0FBSyxDQUFDLDRDQUE0QyxJQUFJLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUNsRjtRQUNELE9BQU8sS0FBSyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDaEQsQ0FBQztDQUNKOzs7Ozs7QUFLRCxNQUFhLGtCQUFrQjtJQXdGM0IsWUFBb0IsZ0JBQXdCLEVBQUUsaUJBQXlCO1FBQ25FLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxnQkFBZ0IsQ0FBQztRQUN6QyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsaUJBQWlCLENBQUM7SUFDL0MsQ0FBQzs7Ozs7Ozs7SUFWTSxNQUFNLENBQUMsRUFBRSxDQUFDLGdCQUF3QixFQUFFLGlCQUF5QjtRQUNoRSxPQUFPLElBQUksa0JBQWtCLENBQUMsZ0JBQWdCLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztJQUN2RSxDQUFDOztBQW5GTCxnREE0RkM7Ozs7OztBQTFGMEIsMEJBQU8sR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFOUMsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsMEJBQU8sR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFOUMsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsMEJBQU8sR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFOUMsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsMEJBQU8sR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFOUMsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDOzs7Ozs7QUFFcEQsNkJBQVUsR0FBRyxrQkFBa0IsQ0FBQyxFQUFFLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBNkIvRSxNQUFNLG1CQUFvQixTQUFRLGtCQUFrQjtJQUNoRCxZQUFZLE9BQTRCO1FBQ3BDLEtBQUssQ0FBQztZQUNGLFVBQVUsRUFBRSxPQUFPO1lBQ25CLDZCQUE2QixFQUFFLGNBQWMsQ0FBQyx5QkFBeUIsQ0FBQywwQkFBMEI7WUFDbEcsNEJBQTRCLEVBQUUsY0FBYyxDQUFDLHlCQUF5QixDQUFDLHlCQUF5QjtZQUNoRyxPQUFPLEVBQUUsT0FBTztnQkFDWixDQUFDLENBQUM7b0JBQ0UsV0FBVyxFQUFFLE9BQU8sQ0FBQyxnQkFBZ0I7b0JBQ3JDLFlBQVksRUFBRSxPQUFPLENBQUMsaUJBQWlCO2lCQUMxQztnQkFDRCxDQUFDLENBQUMsU0FBUztZQUNmLFlBQVksRUFBRSxPQUFPO1NBQ3hCLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjs7Ozs7O0FBZ0JELE1BQWEscUJBQXFCO0lBcUk5QixZQUFvQixtQkFBMkIsRUFBRSxvQkFBNEIsRUFBRSxnQkFBeUM7UUFDcEgsSUFBSSxDQUFDLG1CQUFtQixHQUFHLG1CQUFtQixDQUFDO1FBQy9DLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxvQkFBb0IsQ0FBQztRQUNqRCxJQUFJLENBQUMsU0FBUyxHQUFHO1lBQ2IsUUFBUSxFQUFFLENBQUEsZ0JBQWdCLGFBQWhCLGdCQUFnQix1QkFBaEIsZ0JBQWdCLENBQUUsUUFBUSxFQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFNBQVM7U0FDaEUsQ0FBQztJQUNOLENBQUM7Ozs7Ozs7O0lBbEJNLE1BQU0sQ0FBQyxFQUFFLENBQUMsbUJBQTJCLEVBQUUsb0JBQTRCLEVBQUUsZ0JBQXlDO1FBQ2pILE9BQU8sSUFBSSxxQkFBcUIsQ0FBQyxtQkFBbUIsRUFBRSxvQkFBb0IsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0lBQ2xHLENBQUM7O0FBM0hMLHNEQTRJQzs7Ozs7O0FBMUkwQiw2QkFBTyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVqRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCw2QkFBTyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVqRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUVyRCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCxnQ0FBVSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLENBQUM7Ozs7OztBQUV2RCw0QkFBTSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLENBQUM7Ozs7OztBQUU5Qyw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7Ozs7OztBQUVsRCw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7Ozs7OztBQUVsRCw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7Ozs7OztBQUVsRCw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7Ozs7OztBQUVsRCw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxDQUFDLENBQUM7Ozs7OztBQUVsRCw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV4RSwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV4RSwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV4RSwrQkFBUyxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxPQUFPLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV4RSw0QkFBTSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUVsRSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw0QkFBTSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUVsRSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7Ozs7OztBQUV0RSw4QkFBUSxHQUFHLHFCQUFxQixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLENBQUM7QUFxQ2pHOztHQUVHO0FBQ0gsTUFBTSxzQkFBdUIsU0FBUSxrQkFBa0I7SUFFbkQsWUFBWSxPQUErQjtRQUN2QyxLQUFLLENBQUM7WUFDRixVQUFVLEVBQUUsVUFBVTtZQUN0Qiw2QkFBNkIsRUFBRSxjQUFjLENBQUMseUJBQXlCLENBQUMsNkJBQTZCO1lBQ3JHLDRCQUE0QixFQUFFLGNBQWMsQ0FBQyx5QkFBeUIsQ0FBQyw0QkFBNEI7WUFDbkcsT0FBTyxFQUFFLE9BQU87Z0JBQ1osQ0FBQyxDQUFDO29CQUNFLFdBQVcsRUFBRSxPQUFPLENBQUMsbUJBQW1CO29CQUN4QyxZQUFZLEVBQUUsT0FBTyxDQUFDLG9CQUFvQjtpQkFDN0M7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7WUFDZixRQUFRLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxPQUFPLGFBQVAsT0FBTyx1QkFBUCxPQUFPLENBQUUsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLFFBQVEsRUFBRSxVQUFVLEVBQUU7WUFDakUsWUFBWSxFQUFFLFlBQVk7U0FDN0IsQ0FBQyxDQUFDO1FBZFMsb0JBQWUsR0FBRyxVQUFVLENBQUM7SUFlN0MsQ0FBQztDQUNKOzs7Ozs7QUFTRCxNQUFhLHlCQUF5QjtJQTBEbEMsWUFBb0IsdUJBQStCLEVBQUUsd0JBQWdDO1FBQ2pGLElBQUksQ0FBQyx1QkFBdUIsR0FBRyx1QkFBdUIsQ0FBQztRQUN2RCxJQUFJLENBQUMsd0JBQXdCLEdBQUcsd0JBQXdCLENBQUM7SUFDN0QsQ0FBQztJQVZPLE1BQU0sQ0FBQyxFQUFFLENBQUMsdUJBQStCLEVBQUUsd0JBQWdDO1FBQy9FLE9BQU8sSUFBSSx5QkFBeUIsQ0FBQyx1QkFBdUIsRUFBRSx3QkFBd0IsQ0FBQyxDQUFDO0lBQzVGLENBQUM7O0FBckRMLDhEQThEQzs7Ozs7O0FBNUQwQixrQ0FBUSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxNQUFNLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV4RCx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSx5Q0FBZSxHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxhQUFhLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV0RSwwQ0FBZ0IsR0FBRyx5QkFBeUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFeEUsMENBQWdCLEdBQUcseUJBQXlCLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXhFLDBDQUFnQixHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV4RSwwQ0FBZ0IsR0FBRyx5QkFBeUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFeEUsMENBQWdCLEdBQUcseUJBQXlCLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXhFLDBDQUFnQixHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV4RSwwQ0FBZ0IsR0FBRyx5QkFBeUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFeEUsMENBQWdCLEdBQUcseUJBQXlCLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXhFLDBDQUFnQixHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV4RSwwQ0FBZ0IsR0FBRyx5QkFBeUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFeEUsMENBQWdCLEdBQUcseUJBQXlCLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXhFLDBDQUFnQixHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUV4RSwwQ0FBZ0IsR0FBRyx5QkFBeUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFeEUsMENBQWdCLEdBQUcseUJBQXlCLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRXhFLDBDQUFnQixHQUFHLHlCQUF5QixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQWtCbkcsTUFBYSxtQkFBbUI7SUE4RjVCLFlBQW9CLGlCQUF5QixFQUFFLGtCQUEwQjtRQUNyRSxJQUFJLENBQUMsaUJBQWlCLEdBQUcsaUJBQWlCLENBQUM7UUFDM0MsSUFBSSxDQUFDLGtCQUFrQixHQUFHLGtCQUFrQixDQUFDO0lBQ2pELENBQUM7Ozs7Ozs7O0lBVk0sTUFBTSxDQUFDLEVBQUUsQ0FBQyxpQkFBeUIsRUFBRSxrQkFBMEI7UUFDbEUsT0FBTyxJQUFJLG1CQUFtQixDQUFDLGlCQUFpQixFQUFFLGtCQUFrQixDQUFDLENBQUM7SUFDMUUsQ0FBQzs7QUF6Rkwsa0RBa0dDOzs7Ozs7QUFoRzBCLDRCQUFRLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLE1BQU0sRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWxELG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG1DQUFlLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGFBQWEsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWhFLG9DQUFnQixHQUFHLG1CQUFtQixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUVsRSxvQ0FBZ0IsR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFbEUsb0NBQWdCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWxFLG9DQUFnQixHQUFHLG1CQUFtQixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUVsRSxvQ0FBZ0IsR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFbEUsb0NBQWdCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWxFLG9DQUFnQixHQUFHLG1CQUFtQixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUVsRSxvQ0FBZ0IsR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFbEUsb0NBQWdCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLGNBQWMsRUFBRSxNQUFNLENBQUMsQ0FBQzs7Ozs7O0FBRWxFLG9DQUFnQixHQUFHLG1CQUFtQixDQUFDLEVBQUUsQ0FBQyxjQUFjLEVBQUUsTUFBTSxDQUFDLENBQUM7Ozs7OztBQUVsRSxvQ0FBZ0IsR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsY0FBYyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFbEUsNEJBQVEsR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsTUFBTSxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFbEQsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLE1BQU0sQ0FBQyxDQUFDOzs7Ozs7QUFFL0YsMEJBQU0sR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFNUMsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMEJBQU0sR0FBRyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFNUMsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDOzs7Ozs7QUFFN0YsMkNBQXVCLEdBQUcsbUJBQW1CLENBQUMsRUFBRSxDQUFDLG9DQUFvQyxFQUFFLElBQUksQ0FBQyxDQUFDO0FBeUJ4SCxNQUFlLHdCQUF5QixTQUFRLGtCQUFrQjtJQUM5RCxZQUFZLEtBQW9DO1FBQzVDLEtBQUssQ0FBQztZQUNGLEdBQUcsS0FBSztZQUNSLDZCQUE2QixFQUFFLGNBQWMsQ0FBQyx5QkFBeUIsQ0FBQywyQkFBMkI7WUFDbkcsNEJBQTRCLEVBQUUsY0FBYyxDQUFDLHlCQUF5QixDQUFDLDBCQUEwQjtZQUNqRyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxZQUFZLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNyRyxRQUFRLEVBQUU7Z0JBQ04sUUFBUSxFQUFFLGdCQUFnQjtnQkFDMUIsUUFBUSxFQUFFLGdCQUFnQjthQUM3QjtTQUNKLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDTSxjQUFjLENBQUMsS0FBcUIsRUFBRSxPQUFrQztRQUMzRSxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNwRCxJQUFJLFdBQVcsR0FBRyxPQUFPLENBQUMsV0FBVyxDQUFDO1FBQ3RDLElBQUksT0FBTyxDQUFDLFlBQVksSUFBSSxPQUFPLENBQUMsWUFBWSxFQUFFO1lBQzlDLElBQUksQ0FBQyxXQUFXLEVBQUU7Z0JBQ2QsV0FBVyxHQUFHLElBQUksMEJBQVcsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7b0JBQ3hELE1BQU0sRUFBRSxJQUFJO29CQUNaLGNBQWMsRUFBRSxFQUFFO2lCQUNyQixDQUFDLENBQUM7YUFDTjtZQUNELG9GQUFvRjtZQUNwRixXQUFXLENBQUMsZ0JBQWdCLENBQUM7Z0JBQ3pCLElBQUksRUFBRSxnQkFBZ0I7Z0JBQ3RCLE9BQU8sRUFBRSxLQUFLO2FBQ2pCLENBQUMsQ0FBQztTQUNOO1FBQ0QsT0FBTztZQUNILEdBQUcsTUFBTTtZQUNULFdBQVc7U0FDZCxDQUFDO0lBQ04sQ0FBQztDQUNKO0FBZUQsNkhBQTZIO0FBQzdILE1BQU0sc0JBQXVCLFNBQVEsd0JBQXdCO0lBQ3pELFlBQVksT0FBbUM7UUFDM0MsS0FBSyxDQUFDO1lBQ0YsVUFBVSxFQUFFLFdBQVc7WUFDdkIsT0FBTyxFQUFFLE9BQU87Z0JBQ1osQ0FBQyxDQUFDO29CQUNFLFdBQVcsRUFBRSxPQUFPLENBQUMsdUJBQXVCO29CQUM1QyxZQUFZLEVBQUUsT0FBTyxDQUFDLHdCQUF3QjtpQkFDakQ7Z0JBQ0QsQ0FBQyxDQUFDO29CQUNFLFlBQVksRUFBRSxNQUFNO2lCQUN2QjtTQUNSLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQVdELDZIQUE2SDtBQUM3SCxNQUFNLHVCQUF3QixTQUFRLHdCQUF3QjtJQUMxRCxZQUFZLE9BQW1DO1FBQzNDLEtBQUssQ0FBQztZQUNGLFVBQVUsRUFBRSxZQUFZO1lBQ3hCLE9BQU8sRUFBRSxPQUFPO2dCQUNaLENBQUMsQ0FBQztvQkFDRSxXQUFXLEVBQUUsT0FBTyxDQUFDLHVCQUF1QjtvQkFDNUMsWUFBWSxFQUFFLE9BQU8sQ0FBQyx3QkFBd0I7aUJBQ2pEO2dCQUNELENBQUMsQ0FBQztvQkFDRSxZQUFZLEVBQUUsTUFBTTtpQkFDdkI7U0FDUixDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUFPRCxNQUFNLHVCQUF3QixTQUFRLHdCQUF3QjtJQUMxRCxZQUFZLE9BQTZCO1FBQ3JDLEtBQUssQ0FBQztZQUNGLFVBQVUsRUFBRSxZQUFZO1lBQ3hCLE9BQU8sRUFBRSxPQUFPO2dCQUNaLENBQUMsQ0FBQztvQkFDRSxXQUFXLEVBQUUsT0FBTyxDQUFDLGlCQUFpQjtvQkFDdEMsWUFBWSxFQUFFLE9BQU8sQ0FBQyxrQkFBa0I7aUJBQzNDO2dCQUNELENBQUMsQ0FBQyxTQUFTO1NBQ2xCLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQU9ELE1BQU0sc0JBQXVCLFNBQVEsd0JBQXdCO0lBQ3pELFlBQVksT0FBNkI7UUFDckMsS0FBSyxDQUFDO1lBQ0YsVUFBVSxFQUFFLFdBQVc7WUFDdkIsT0FBTyxFQUFFLE9BQU87Z0JBQ1osQ0FBQyxDQUFDO29CQUNFLFdBQVcsRUFBRSxPQUFPLENBQUMsaUJBQWlCO29CQUN0QyxZQUFZLEVBQUUsT0FBTyxDQUFDLGtCQUFrQjtpQkFDM0M7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7U0FDbEIsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztDQUNKOzs7Ozs7QUFPRCxNQUFhLHNCQUFzQjtJQThFL0IsWUFBb0Isb0JBQTRCLEVBQUUscUJBQTZCO1FBQzNFLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxvQkFBb0IsQ0FBQztRQUNqRCxJQUFJLENBQUMscUJBQXFCLEdBQUcscUJBQXFCLENBQUM7SUFDdkQsQ0FBQzs7Ozs7Ozs7SUFWTSxNQUFNLENBQUMsRUFBRSxDQUFDLG9CQUE0QixFQUFFLHFCQUE2QjtRQUN4RSxPQUFPLElBQUksc0JBQXNCLENBQUMsb0JBQW9CLEVBQUUscUJBQXFCLENBQUMsQ0FBQztJQUNuRixDQUFDOztBQXpFTCx3REFrRkM7Ozs7OztBQWhGMEIsNkJBQU0sR0FBRyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFckQsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsNkJBQU0sR0FBRyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFckQsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsNkJBQU0sR0FBRyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFckQsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMkNBQW9CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFOUUsNkJBQU0sR0FBRyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFckQsNENBQXFCLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLG1CQUFtQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFaEYsMkNBQW9CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFOUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsMENBQW1CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGlCQUFpQixFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFNUUsNkJBQU0sR0FBRyxzQkFBc0IsQ0FBQyxFQUFFLENBQUMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxDQUFDOzs7Ozs7QUFFckQsMkNBQW9CLEdBQUcsc0JBQXNCLENBQUMsRUFBRSxDQUFDLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDO0FBNkJ6RyxNQUFlLDJCQUE0QixTQUFRLGtCQUFrQjtJQUNqRSxZQUFZLEtBQXVDO1FBQy9DLEtBQUssQ0FBQztZQUNGLEdBQUcsS0FBSztZQUNSLDZCQUE2QixFQUFFLGNBQWMsQ0FBQyx5QkFBeUIsQ0FBQyw4QkFBOEI7WUFDdEcsNEJBQTRCLEVBQUUsY0FBYyxDQUFDLHlCQUF5QixDQUFDLDZCQUE2QjtZQUNwRyxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87Z0JBQ2xCLENBQUMsQ0FBQztvQkFDRSxXQUFXLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxvQkFBb0I7b0JBQy9DLFlBQVksRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLHFCQUFxQjtpQkFDcEQ7Z0JBQ0QsQ0FBQyxDQUFDLFNBQVM7WUFDZixvQkFBb0IsRUFBRSxLQUFLLENBQUMsT0FBTztnQkFDL0IsK0VBQStFO2dCQUMvRSwrQ0FBK0M7Z0JBQy9DLENBQUMsQ0FBQyxHQUFHLEtBQUssQ0FBQyxVQUFVLElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDO29CQUN4RSxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO29CQUNsRCxDQUFDLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxxQkFBcUIsRUFBRTtnQkFDM0MsQ0FBQyxDQUFDLFNBQVM7WUFDZixRQUFRLEVBQUU7Z0JBQ04sUUFBUSxFQUFFLGdCQUFnQjtnQkFDMUIsUUFBUSxFQUFFLGdCQUFnQjthQUM3QjtTQUNKLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDTSxjQUFjLENBQUMsS0FBcUIsRUFBRSxPQUFrQzs7UUFDM0UsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDcEQsSUFBSSxXQUFXLEdBQUcsT0FBTyxDQUFDLFdBQVcsQ0FBQztRQUN0QyxNQUFNLE1BQU0sU0FBRyxPQUFPLENBQUMsWUFBWSxtQ0FBSSxPQUFPLENBQUMsWUFBWSxDQUFDO1FBQzVELElBQUksTUFBTSxFQUFFO1lBQ1IsSUFBSSxPQUFPLENBQUMsWUFBWSxJQUFJLE9BQU8sQ0FBQyxZQUFZLElBQUksT0FBTyxDQUFDLFlBQVksS0FBSyxPQUFPLENBQUMsWUFBWSxFQUFFO2dCQUMvRixNQUFNLElBQUksS0FBSyxDQUFDLG9FQUFvRSxDQUFDLENBQUM7YUFDekY7WUFDRCxJQUFJLENBQUMsV0FBVyxFQUFFO2dCQUNkLFdBQVcsR0FBRyxJQUFJLDBCQUFXLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO29CQUN4RCxNQUFNLEVBQUUsSUFBSTtvQkFDWixjQUFjLEVBQUUsRUFBRTtpQkFDckIsQ0FBQyxDQUFDO2FBQ047WUFDRCx1R0FBdUc7WUFDdkcsV0FBVyxDQUFDLGdCQUFnQixDQUFDO2dCQUN6QixJQUFJLEVBQUUsMEJBQTBCO2dCQUNoQyxRQUFRLEVBQUUsRUFBRSxZQUFZLEVBQUUsTUFBTSxDQUFDLE9BQU8sRUFBRTthQUM3QyxDQUFDLENBQUM7U0FDTjtRQUNELE9BQU87WUFDSCxHQUFHLE1BQU07WUFDVCxXQUFXO1NBQ2QsQ0FBQztJQUNOLENBQUM7SUFDRCxJQUFjLGdCQUFnQixLQUFLLE9BQU8sSUFBSSxDQUFDLENBQUMsQ0FBQztDQUNwRDtBQU9ELE1BQU0seUJBQTBCLFNBQVEsMkJBQTJCO0lBQy9ELFlBQVksT0FBZ0M7UUFDeEMsS0FBSyxDQUFDO1lBQ0YsVUFBVSxFQUFFLGNBQWM7WUFDMUIsT0FBTztTQUNWLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQU9ELE1BQU0seUJBQTBCLFNBQVEsMkJBQTJCO0lBQy9ELFlBQVksT0FBZ0M7UUFDeEMsS0FBSyxDQUFDO1lBQ0YsVUFBVSxFQUFFLGNBQWM7WUFDMUIsT0FBTztTQUNWLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQU9ELE1BQU0sMEJBQTJCLFNBQVEsMkJBQTJCO0lBQ2hFLFlBQVksT0FBZ0M7UUFDeEMsS0FBSyxDQUFDO1lBQ0YsVUFBVSxFQUFFLGVBQWU7WUFDM0IsT0FBTztTQUNWLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjtBQU9ELE1BQU0seUJBQTBCLFNBQVEsMkJBQTJCO0lBQy9ELFlBQVksT0FBZ0M7UUFDeEMsS0FBSyxDQUFDO1lBQ0YsVUFBVSxFQUFFLGNBQWM7WUFDMUIsT0FBTztTQUNWLENBQUMsQ0FBQztJQUNQLENBQUM7Q0FDSjs7Ozs7Ozs7O0FBS0QsTUFBYSxzQkFBc0I7Ozs7OztJQTZFeEIsTUFBTSxDQUFDLE9BQU8sQ0FBQyxLQUFpQztRQUNuRCxPQUFPLElBQUkscUJBQXFCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3BELENBQUM7Ozs7OztJQUVNLE1BQU0sQ0FBQyxLQUFLLENBQUMsS0FBK0I7UUFDL0MsT0FBTyxJQUFJLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNsRCxDQUFDOzs7Ozs7SUFFTSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQWtDO1FBQ3JELE9BQU8sSUFBSSxzQkFBc0IsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDckQsQ0FBQzs7Ozs7O0lBS00sTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFrQztRQUNyRCxPQUFPLElBQUksc0JBQXNCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3JELENBQUM7Ozs7OztJQUtNLE1BQU0sQ0FBQyxTQUFTLENBQUMsS0FBbUM7UUFDdkQsT0FBTyxJQUFJLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN0RCxDQUFDOzs7Ozs7SUFFTSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQW1DO1FBQ3ZELE9BQU8sSUFBSSx1QkFBdUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdEQsQ0FBQzs7Ozs7O0lBRU0sTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFrQztRQUNyRCxPQUFPLElBQUksc0JBQXNCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3JELENBQUM7Ozs7OztJQUVNLE1BQU0sQ0FBQyxXQUFXLENBQUMsS0FBcUM7UUFDM0QsT0FBTyxJQUFJLHlCQUF5QixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN4RCxDQUFDOzs7Ozs7SUFFTSxNQUFNLENBQUMsV0FBVyxDQUFDLEtBQXFDO1FBQzNELE9BQU8sSUFBSSx5QkFBeUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDeEQsQ0FBQzs7Ozs7O0lBRU0sTUFBTSxDQUFDLFlBQVksQ0FBQyxLQUFzQztRQUM3RCxPQUFPLElBQUksMEJBQTBCLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3pELENBQUM7Ozs7OztJQUVNLE1BQU0sQ0FBQyxXQUFXLENBQUMsS0FBcUM7UUFDM0QsT0FBTyxJQUFJLHlCQUF5QixDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN4RCxDQUFDOztBQTdITCx3REE4SEM7Ozs7Ozs7QUF2SDBCLDhCQUFPLEdBQW9CLElBQUkscUJBQXFCLEVBQUUsQ0FBQzs7Ozs7OztBQU92RCw0QkFBSyxHQUFvQixJQUFJLG1CQUFtQixFQUFFLENBQUM7Ozs7Ozs7QUFPbkQsZ0NBQVMsR0FBb0IsSUFBSSxzQkFBc0IsRUFBRSxDQUFDOzs7Ozs7O0FBTzFELGlDQUFVLEdBQW9CLElBQUksdUJBQXVCLEVBQUUsQ0FBQzs7Ozs7O0FBTTVELGlDQUFVLEdBQW9CLElBQUksdUJBQXVCLEVBQUUsQ0FBQzs7Ozs7O0FBTTVELGdDQUFTLEdBQW9CLElBQUksc0JBQXNCLEVBQUUsQ0FBQzs7Ozs7OztBQU8xRCwrQkFBUSxHQUFvQixJQUFJLHNCQUFzQixFQUFFLENBQUM7Ozs7Ozs7QUFPekQsb0NBQWEsR0FBb0IsSUFBSSx5QkFBeUIsRUFBRSxDQUFDOzs7Ozs7O0FBT2pFLG9DQUFhLEdBQW9CLElBQUkseUJBQXlCLEVBQUUsQ0FBQzs7Ozs7OztBQU9qRSxvQ0FBYSxHQUFvQixJQUFJLHlCQUF5QixFQUFFLENBQUM7Ozs7Ozs7QUFPakUscUNBQWMsR0FBb0IsSUFBSSwwQkFBMEIsRUFBRSxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0ICogYXMgc2VjcmV0c21hbmFnZXIgZnJvbSBcIi4uLy4uL2F3cy1zZWNyZXRzbWFuYWdlclwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXNlY3JldHNtYW5hZ2VyJ1xuaW1wb3J0ICogYXMgY29yZSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IElFbmdpbmUgfSBmcm9tICcuL2VuZ2luZSc7XG5pbXBvcnQgeyBFbmdpbmVWZXJzaW9uIH0gZnJvbSAnLi9lbmdpbmUtdmVyc2lvbic7XG5pbXBvcnQgeyBJT3B0aW9uR3JvdXAsIE9wdGlvbkdyb3VwIH0gZnJvbSAnLi9vcHRpb24tZ3JvdXAnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VFbmdpbmVCaW5kT3B0aW9ucyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBkb21haW4/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHRpbWV6b25lPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgczNJbXBvcnRSb2xlPzogaWFtLklSb2xlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgczNFeHBvcnRSb2xlPzogaWFtLklSb2xlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IG9wdGlvbkdyb3VwPzogSU9wdGlvbkdyb3VwO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSW5zdGFuY2VFbmdpbmVDb25maWcge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZmVhdHVyZXM/OiBJbnN0YW5jZUVuZ2luZUZlYXR1cmVzO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IG9wdGlvbkdyb3VwPzogSU9wdGlvbkdyb3VwO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJbnN0YW5jZUVuZ2luZUZlYXR1cmVzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgczNJbXBvcnQ/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHMzRXhwb3J0Pzogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJSW5zdGFuY2VFbmdpbmUgZXh0ZW5kcyBJRW5naW5lIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBzaW5nbGVVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IG11bHRpVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb247XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIGJpbmRUb0luc3RhbmNlKHNjb3BlOiBjb3JlLkNvbnN0cnVjdCwgb3B0aW9uczogSW5zdGFuY2VFbmdpbmVCaW5kT3B0aW9ucyk6IEluc3RhbmNlRW5naW5lQ29uZmlnO1xufVxuaW50ZXJmYWNlIEluc3RhbmNlRW5naW5lQmFzZVByb3BzIHtcbiAgICByZWFkb25seSBlbmdpbmVUeXBlOiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb247XG4gICAgcmVhZG9ubHkgbXVsdGlVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbjtcbiAgICByZWFkb25seSB2ZXJzaW9uPzogRW5naW5lVmVyc2lvbjtcbiAgICByZWFkb25seSBwYXJhbWV0ZXJHcm91cEZhbWlseT86IHN0cmluZztcbiAgICByZWFkb25seSBlbmdpbmVGYW1pbHk/OiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgZmVhdHVyZXM/OiBJbnN0YW5jZUVuZ2luZUZlYXR1cmVzO1xufVxuYWJzdHJhY3QgY2xhc3MgSW5zdGFuY2VFbmdpbmVCYXNlIGltcGxlbWVudHMgSUluc3RhbmNlRW5naW5lIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgZW5naW5lVHlwZTogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBlbmdpbmVWZXJzaW9uPzogRW5naW5lVmVyc2lvbjtcbiAgICBwdWJsaWMgcmVhZG9ubHkgcGFyYW1ldGVyR3JvdXBGYW1pbHk/OiBzdHJpbmc7XG4gICAgcHVibGljIHJlYWRvbmx5IHNpbmdsZVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uO1xuICAgIHB1YmxpYyByZWFkb25seSBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uO1xuICAgIHB1YmxpYyByZWFkb25seSBlbmdpbmVGYW1pbHk/OiBzdHJpbmc7XG4gICAgcHJpdmF0ZSByZWFkb25seSBmZWF0dXJlcz86IEluc3RhbmNlRW5naW5lRmVhdHVyZXM7XG4gICAgY29uc3RydWN0b3IocHJvcHM6IEluc3RhbmNlRW5naW5lQmFzZVByb3BzKSB7XG4gICAgICAgIHRoaXMuZW5naW5lVHlwZSA9IHByb3BzLmVuZ2luZVR5cGU7XG4gICAgICAgIHRoaXMuZmVhdHVyZXMgPSBwcm9wcy5mZWF0dXJlcztcbiAgICAgICAgdGhpcy5zaW5nbGVVc2VyUm90YXRpb25BcHBsaWNhdGlvbiA9IHByb3BzLnNpbmdsZVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uO1xuICAgICAgICB0aGlzLm11bHRpVXNlclJvdGF0aW9uQXBwbGljYXRpb24gPSBwcm9wcy5tdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uO1xuICAgICAgICB0aGlzLmVuZ2luZVZlcnNpb24gPSBwcm9wcy52ZXJzaW9uO1xuICAgICAgICB0aGlzLnBhcmFtZXRlckdyb3VwRmFtaWx5ID0gcHJvcHMucGFyYW1ldGVyR3JvdXBGYW1pbHkgPz9cbiAgICAgICAgICAgICh0aGlzLmVuZ2luZVZlcnNpb24gPyBgJHt0aGlzLmVuZ2luZVR5cGV9JHt0aGlzLmVuZ2luZVZlcnNpb24ubWFqb3JWZXJzaW9ufWAgOiB1bmRlZmluZWQpO1xuICAgICAgICB0aGlzLmVuZ2luZUZhbWlseSA9IHByb3BzLmVuZ2luZUZhbWlseTtcbiAgICB9XG4gICAgcHVibGljIGJpbmRUb0luc3RhbmNlKF9zY29wZTogY29yZS5Db25zdHJ1Y3QsIG9wdGlvbnM6IEluc3RhbmNlRW5naW5lQmluZE9wdGlvbnMpOiBJbnN0YW5jZUVuZ2luZUNvbmZpZyB7XG4gICAgICAgIGlmIChvcHRpb25zLnRpbWV6b25lICYmICF0aGlzLnN1cHBvcnRzVGltZXpvbmUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgdGltZXpvbmUgcHJvcGVydHkgY2FuIG5vdCBiZSBjb25maWd1cmVkIGZvciAke3RoaXMuZW5naW5lVHlwZX1gKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgZmVhdHVyZXM6IHRoaXMuZmVhdHVyZXMsXG4gICAgICAgICAgICBvcHRpb25Hcm91cDogb3B0aW9ucy5vcHRpb25Hcm91cCxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgLyoqIERlZmluZXMgd2hldGhlciB0aGlzIEluc3RhbmNlIEVuZ2luZSBjYW4gc3VwcG9ydCB0aW1lem9uZSBwcm9wZXJ0aWVzLiAqL1xuICAgIHByb3RlY3RlZCBnZXQgc3VwcG9ydHNUaW1lem9uZSgpIHsgcmV0dXJuIGZhbHNlOyB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE1hcmlhRGJFbmdpbmVWZXJzaW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wJywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8xNyA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjE3JywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8yNCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjI0JywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8yOCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjI4JywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8zMSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjMxJywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8zMiA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjMyJywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8zNCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjM0JywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMF8zNSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4wLjM1JywgJzEwLjAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xJywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMV8xNCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xLjE0JywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMV8xOSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xLjE5JywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMV8yMyA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xLjIzJywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMV8yNiA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xLjI2JywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMV8zMSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xLjMxJywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMV8zNCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4xLjM0JywgJzEwLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMiA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4yJywgJzEwLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMl8xMSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4yLjExJywgJzEwLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMl8xMiA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4yLjEyJywgJzEwLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMl8xNSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4yLjE1JywgJzEwLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMl8yMSA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4yLjIxJywgJzEwLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMyA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4zJywgJzEwLjMnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMF8zXzggPSBNYXJpYURiRW5naW5lVmVyc2lvbi5vZignMTAuMy44JywgJzEwLjMnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfM18xMyA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4zLjEzJywgJzEwLjMnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfM18yMCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4zLjIwJywgJzEwLjMnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfM18yMyA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC4zLjIzJywgJzEwLjMnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNCA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC40JywgJzEwLjQnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMF80XzggPSBNYXJpYURiRW5naW5lVmVyc2lvbi5vZignMTAuNC44JywgJzEwLjQnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNF8xMyA9IE1hcmlhRGJFbmdpbmVWZXJzaW9uLm9mKCcxMC40LjEzJywgJzEwLjQnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIG9mKG1hcmlhRGJGdWxsVmVyc2lvbjogc3RyaW5nLCBtYXJpYURiTWFqb3JWZXJzaW9uOiBzdHJpbmcpOiBNYXJpYURiRW5naW5lVmVyc2lvbiB7XG4gICAgICAgIHJldHVybiBuZXcgTWFyaWFEYkVuZ2luZVZlcnNpb24obWFyaWFEYkZ1bGxWZXJzaW9uLCBtYXJpYURiTWFqb3JWZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBtYXJpYURiRnVsbFZlcnNpb246IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgbWFyaWFEYk1ham9yVmVyc2lvbjogc3RyaW5nO1xuICAgIHByaXZhdGUgY29uc3RydWN0b3IobWFyaWFEYkZ1bGxWZXJzaW9uOiBzdHJpbmcsIG1hcmlhRGJNYWpvclZlcnNpb246IHN0cmluZykge1xuICAgICAgICB0aGlzLm1hcmlhRGJGdWxsVmVyc2lvbiA9IG1hcmlhRGJGdWxsVmVyc2lvbjtcbiAgICAgICAgdGhpcy5tYXJpYURiTWFqb3JWZXJzaW9uID0gbWFyaWFEYk1ham9yVmVyc2lvbjtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgTWFyaWFEYkluc3RhbmNlRW5naW5lUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2ZXJzaW9uOiBNYXJpYURiRW5naW5lVmVyc2lvbjtcbn1cbmNsYXNzIE1hcmlhRGJJbnN0YW5jZUVuZ2luZSBleHRlbmRzIEluc3RhbmNlRW5naW5lQmFzZSB7XG4gICAgY29uc3RydWN0b3IodmVyc2lvbj86IE1hcmlhRGJFbmdpbmVWZXJzaW9uKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIGVuZ2luZVR5cGU6ICdtYXJpYWRiJyxcbiAgICAgICAgICAgIHNpbmdsZVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLk1BUklBREJfUk9UQVRJT05fU0lOR0xFX1VTRVIsXG4gICAgICAgICAgICBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLk1BUklBREJfUk9UQVRJT05fTVVMVElfVVNFUixcbiAgICAgICAgICAgIHZlcnNpb246IHZlcnNpb25cbiAgICAgICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICAgICAgZnVsbFZlcnNpb246IHZlcnNpb24ubWFyaWFEYkZ1bGxWZXJzaW9uLFxuICAgICAgICAgICAgICAgICAgICBtYWpvclZlcnNpb246IHZlcnNpb24ubWFyaWFEYk1ham9yVmVyc2lvbixcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZFRvSW5zdGFuY2Uoc2NvcGU6IGNvcmUuQ29uc3RydWN0LCBvcHRpb25zOiBJbnN0YW5jZUVuZ2luZUJpbmRPcHRpb25zKTogSW5zdGFuY2VFbmdpbmVDb25maWcge1xuICAgICAgICBpZiAob3B0aW9ucy5kb21haW4pIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgZG9tYWluIHByb3BlcnR5IGNhbm5vdCBiZSBjb25maWd1cmVkIGZvciAke3RoaXMuZW5naW5lVHlwZX1gKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gc3VwZXIuYmluZFRvSW5zdGFuY2Uoc2NvcGUsIG9wdGlvbnMpO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBNeXNxbEVuZ2luZVZlcnNpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS41JywgJzUuNScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNV80NiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS41LjQ2JywgJzUuNScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNV81MyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS41LjUzJywgJzUuNScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNV81NyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS41LjU3JywgJzUuNScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNV81OSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS41LjU5JywgJzUuNScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNV82MSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS41LjYxJywgJzUuNScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl8zNCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjM0JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl8zNSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjM1JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl8zNyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjM3JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl8zOSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjM5JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl80MCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjQwJywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl80MSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjQxJywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl80MyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjQzJywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl80NCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjQ0JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl80NiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjQ2JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNl80OCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS42LjQ4JywgJzUuNicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfNyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18xNiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjE2JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18xNyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjE3JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18xOSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjE5JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yMSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjIxJywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yMiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjIyJywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yMyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjIzJywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yNCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjI0JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yNSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjI1JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yNiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjI2JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18yOCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjI4JywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzVfN18zMCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignNS43LjMwJywgJzUuNycpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMCA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wJywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMF8xMSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wLjExJywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMF8xMyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wLjEzJywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMF8xNSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wLjE1JywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMF8xNiA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wLjE2JywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMF8xNyA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wLjE3JywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzhfMF8xOSA9IE15c3FsRW5naW5lVmVyc2lvbi5vZignOC4wLjE5JywgJzguMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIG9mKG15c3FsRnVsbFZlcnNpb246IHN0cmluZywgbXlzcWxNYWpvclZlcnNpb246IHN0cmluZyk6IE15c3FsRW5naW5lVmVyc2lvbiB7XG4gICAgICAgIHJldHVybiBuZXcgTXlzcWxFbmdpbmVWZXJzaW9uKG15c3FsRnVsbFZlcnNpb24sIG15c3FsTWFqb3JWZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBteXNxbEZ1bGxWZXJzaW9uOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IG15c3FsTWFqb3JWZXJzaW9uOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihteXNxbEZ1bGxWZXJzaW9uOiBzdHJpbmcsIG15c3FsTWFqb3JWZXJzaW9uOiBzdHJpbmcpIHtcbiAgICAgICAgdGhpcy5teXNxbEZ1bGxWZXJzaW9uID0gbXlzcWxGdWxsVmVyc2lvbjtcbiAgICAgICAgdGhpcy5teXNxbE1ham9yVmVyc2lvbiA9IG15c3FsTWFqb3JWZXJzaW9uO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIE15U3FsSW5zdGFuY2VFbmdpbmVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZlcnNpb246IE15c3FsRW5naW5lVmVyc2lvbjtcbn1cbmNsYXNzIE15U3FsSW5zdGFuY2VFbmdpbmUgZXh0ZW5kcyBJbnN0YW5jZUVuZ2luZUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBNeXNxbEVuZ2luZVZlcnNpb24pIHtcbiAgICAgICAgc3VwZXIoe1xuICAgICAgICAgICAgZW5naW5lVHlwZTogJ215c3FsJyxcbiAgICAgICAgICAgIHNpbmdsZVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLk1ZU1FMX1JPVEFUSU9OX1NJTkdMRV9VU0VSLFxuICAgICAgICAgICAgbXVsdGlVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbi5NWVNRTF9ST1RBVElPTl9NVUxUSV9VU0VSLFxuICAgICAgICAgICAgdmVyc2lvbjogdmVyc2lvblxuICAgICAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgICAgICBmdWxsVmVyc2lvbjogdmVyc2lvbi5teXNxbEZ1bGxWZXJzaW9uLFxuICAgICAgICAgICAgICAgICAgICBtYWpvclZlcnNpb246IHZlcnNpb24ubXlzcWxNYWpvclZlcnNpb24sXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgZW5naW5lRmFtaWx5OiAnTVlTUUwnLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFBvc3RncmVzRW5naW5lRmVhdHVyZXMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgczNJbXBvcnQ/OiBib29sZWFuO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgUG9zdGdyZXNFbmdpbmVWZXJzaW9uIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzUgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNScsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNV8yID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjUuMicsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNV80ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjUuNCcsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNV82ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjUuNicsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNV83ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjUuNycsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNV85ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjUuOScsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTAgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xMCcsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTIgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xMicsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTMgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xMycsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTQgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xNCcsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTUgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xNScsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTYgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xNicsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTggPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xOCcsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMTkgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4xOScsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMjAgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4yMCcsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMjEgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4yMScsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzVfMjIgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNS4yMicsICc5LjUnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzYgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNicsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl8xID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuMScsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl8yID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuMicsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl8zID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuMycsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl81ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuNScsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl82ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuNicsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl84ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuOCcsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzlfNl85ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCc5LjYuOScsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTAgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xMCcsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTEgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xMScsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTIgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xMicsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTQgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xNCcsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTUgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xNScsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTYgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xNicsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTcgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xNycsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl85XzZfMTggPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzkuNi4xOCcsICc5LjYnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEwID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMCcsICcxMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMF8xID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMC4xJywgJzEwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEwXzMgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjMnLCAnMTAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNCA9IFBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuNCcsICcxMCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMF81ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMC41JywgJzEwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEwXzYgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjYnLCAnMTAnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfNyA9IFBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuNycsICcxMCcsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEwXzkgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjknLCAnMTAnLCB7IHMzSW1wb3J0OiB0cnVlIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTAgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjEwJywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEwXzExID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMC4xMScsICcxMCcsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMF8xMiA9IFBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTAuMTInLCAnMTAnLCB7IHMzSW1wb3J0OiB0cnVlIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTBfMTMgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEwLjEzJywgJzEwJywgeyBzM0ltcG9ydDogdHJ1ZSB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMScsICcxMScsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzEgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjEnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMS4yJywgJzExJywgeyBzM0ltcG9ydDogdHJ1ZSB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfNCA9IFBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTEuNCcsICcxMScsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzUgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjUnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV82ID0gUG9zdGdyZXNFbmdpbmVWZXJzaW9uLm9mKCcxMS42JywgJzExJywgeyBzM0ltcG9ydDogdHJ1ZSB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfNyA9IFBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTEuNycsICcxMScsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzggPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzExLjgnLCAnMTEnLCB7IHMzSW1wb3J0OiB0cnVlIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTIgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEyJywgJzEyJywgeyBzM0ltcG9ydDogdHJ1ZSB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMiA9IFBvc3RncmVzRW5naW5lVmVyc2lvbi5vZignMTIuMicsICcxMicsIHsgczNJbXBvcnQ6IHRydWUgfSk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzMgPSBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ub2YoJzEyLjMnLCAnMTInLCB7IHMzSW1wb3J0OiB0cnVlIH0pO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBvZihwb3N0Z3Jlc0Z1bGxWZXJzaW9uOiBzdHJpbmcsIHBvc3RncmVzTWFqb3JWZXJzaW9uOiBzdHJpbmcsIHBvc3RncmVzRmVhdHVyZXM/OiBQb3N0Z3Jlc0VuZ2luZUZlYXR1cmVzKTogUG9zdGdyZXNFbmdpbmVWZXJzaW9uIHtcbiAgICAgICAgcmV0dXJuIG5ldyBQb3N0Z3Jlc0VuZ2luZVZlcnNpb24ocG9zdGdyZXNGdWxsVmVyc2lvbiwgcG9zdGdyZXNNYWpvclZlcnNpb24sIHBvc3RncmVzRmVhdHVyZXMpO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBwb3N0Z3Jlc0Z1bGxWZXJzaW9uOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBwb3N0Z3Jlc01ham9yVmVyc2lvbjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBzdXBwb3J0ZWQgZmVhdHVyZXMgZm9yIHRoZSBEQiBlbmdpbmVcbiAgICAgKiBAaW50ZXJuYWxcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgX2ZlYXR1cmVzOiBJbnN0YW5jZUVuZ2luZUZlYXR1cmVzO1xuICAgIHByaXZhdGUgY29uc3RydWN0b3IocG9zdGdyZXNGdWxsVmVyc2lvbjogc3RyaW5nLCBwb3N0Z3Jlc01ham9yVmVyc2lvbjogc3RyaW5nLCBwb3N0Z3Jlc0ZlYXR1cmVzPzogUG9zdGdyZXNFbmdpbmVGZWF0dXJlcykge1xuICAgICAgICB0aGlzLnBvc3RncmVzRnVsbFZlcnNpb24gPSBwb3N0Z3Jlc0Z1bGxWZXJzaW9uO1xuICAgICAgICB0aGlzLnBvc3RncmVzTWFqb3JWZXJzaW9uID0gcG9zdGdyZXNNYWpvclZlcnNpb247XG4gICAgICAgIHRoaXMuX2ZlYXR1cmVzID0ge1xuICAgICAgICAgICAgczNJbXBvcnQ6IHBvc3RncmVzRmVhdHVyZXM/LnMzSW1wb3J0ID8gJ3MzSW1wb3J0JyA6IHVuZGVmaW5lZCxcbiAgICAgICAgfTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFBvc3RncmVzSW5zdGFuY2VFbmdpbmVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZlcnNpb246IFBvc3RncmVzRW5naW5lVmVyc2lvbjtcbn1cbi8qKlxuICogVGhlIGluc3RhbmNlIGVuZ2luZSBmb3IgUG9zdGdyZVNRTC5cbiAqL1xuY2xhc3MgUG9zdGdyZXNJbnN0YW5jZUVuZ2luZSBleHRlbmRzIEluc3RhbmNlRW5naW5lQmFzZSB7XG4gICAgcHVibGljIHJlYWRvbmx5IGRlZmF1bHRVc2VybmFtZSA9ICdwb3N0Z3Jlcyc7XG4gICAgY29uc3RydWN0b3IodmVyc2lvbj86IFBvc3RncmVzRW5naW5lVmVyc2lvbikge1xuICAgICAgICBzdXBlcih7XG4gICAgICAgICAgICBlbmdpbmVUeXBlOiAncG9zdGdyZXMnLFxuICAgICAgICAgICAgc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24uUE9TVEdSRVNfUk9UQVRJT05fU0lOR0xFX1VTRVIsXG4gICAgICAgICAgICBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLlBPU1RHUkVTX1JPVEFUSU9OX01VTFRJX1VTRVIsXG4gICAgICAgICAgICB2ZXJzaW9uOiB2ZXJzaW9uXG4gICAgICAgICAgICAgICAgPyB7XG4gICAgICAgICAgICAgICAgICAgIGZ1bGxWZXJzaW9uOiB2ZXJzaW9uLnBvc3RncmVzRnVsbFZlcnNpb24sXG4gICAgICAgICAgICAgICAgICAgIG1ham9yVmVyc2lvbjogdmVyc2lvbi5wb3N0Z3Jlc01ham9yVmVyc2lvbixcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICBmZWF0dXJlczogdmVyc2lvbiA/IHZlcnNpb24/Ll9mZWF0dXJlcyA6IHsgczNJbXBvcnQ6ICdzM0ltcG9ydCcgfSxcbiAgICAgICAgICAgIGVuZ2luZUZhbWlseTogJ1BPU1RHUkVTUUwnLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMicsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfMl9WMiA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC4yLnYyJywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1YxID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjEnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjMgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52MycsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WNCA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnY0JywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1Y1ID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjUnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjYgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52NicsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WNyA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnY3JywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1Y4ID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjgnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjkgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52OScsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjEwID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjEwJywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WMTEgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52MTEnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1YxMiA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnYxMicsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjEzID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjEzJywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WMTQgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52MTQnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1YxNSA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnYxNScsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjE2ID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjE2JywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WMTcgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52MTcnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1YxOCA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnYxOCcsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjE5ID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjE5JywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WMjAgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52MjAnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1YyMSA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnYyMScsICcxMS4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMl8wXzRfVjIyID0gT3JhY2xlTGVnYWN5RW5naW5lVmVyc2lvbi5vZignMTEuMi4wLjQudjIyJywgJzExLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMV8yXzBfNF9WMjMgPSBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uLm9mKCcxMS4yLjAuNC52MjMnLCAnMTEuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzExXzJfMF80X1YyNCA9IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb24ub2YoJzExLjIuMC40LnYyNCcsICcxMS4yJyk7XG4gICAgcHJpdmF0ZSBzdGF0aWMgb2Yob3JhY2xlTGVnYWN5RnVsbFZlcnNpb246IHN0cmluZywgb3JhY2xlTGVnYWN5TWFqb3JWZXJzaW9uOiBzdHJpbmcpOiBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uIHtcbiAgICAgICAgcmV0dXJuIG5ldyBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uKG9yYWNsZUxlZ2FjeUZ1bGxWZXJzaW9uLCBvcmFjbGVMZWdhY3lNYWpvclZlcnNpb24pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgb3JhY2xlTGVnYWN5RnVsbFZlcnNpb246IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgb3JhY2xlTGVnYWN5TWFqb3JWZXJzaW9uOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihvcmFjbGVMZWdhY3lGdWxsVmVyc2lvbjogc3RyaW5nLCBvcmFjbGVMZWdhY3lNYWpvclZlcnNpb246IHN0cmluZykge1xuICAgICAgICB0aGlzLm9yYWNsZUxlZ2FjeUZ1bGxWZXJzaW9uID0gb3JhY2xlTGVnYWN5RnVsbFZlcnNpb247XG4gICAgICAgIHRoaXMub3JhY2xlTGVnYWN5TWFqb3JWZXJzaW9uID0gb3JhY2xlTGVnYWN5TWFqb3JWZXJzaW9uO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgT3JhY2xlRW5naW5lVmVyc2lvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzEgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xJywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1YxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMS4wLjIudjEnLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMV8wXzJfVjIgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52MicsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WMyA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnYzJywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1Y0ID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMS4wLjIudjQnLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMV8wXzJfVjUgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52NScsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WNiA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnY2JywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1Y3ID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMS4wLjIudjcnLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMV8wXzJfVjggPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52OCcsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WOSA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnY5JywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WMTAgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52MTAnLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1YxMSA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnYxMScsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMV8wXzJfVjEyID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMS4wLjIudjEyJywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WMTMgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52MTMnLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1YxNCA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnYxNCcsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMV8wXzJfVjE1ID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMS4wLjIudjE1JywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WMTYgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52MTYnLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1YxNyA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnYxNycsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMV8wXzJfVjE4ID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMS4wLjIudjE4JywgJzEyLjEnKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8xXzBfMl9WMTkgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4xLjAuMi52MTknLCAnMTIuMScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzFfMF8yX1YyMCA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjEuMC4yLnYyMCcsICcxMi4xJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzIgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4yJywgJzEyLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMl8wXzFfMjAxOF8xMF9SMSA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjIuMC4xLnJ1LTIwMTgtMTAucnVyLTIwMTgtMTAucjEnLCAnMTIuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8yXzBfMV8yMDE5XzAxX1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMi4wLjEucnUtMjAxOS0wMS5ydXItMjAxOS0wMS5yMScsICcxMi4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzJfMF8xXzIwMTlfMDRfUjEgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4yLjAuMS5ydS0yMDE5LTA0LnJ1ci0yMDE5LTA0LnIxJywgJzEyLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMl8wXzFfMjAxOV8wN19SMSA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjIuMC4xLnJ1LTIwMTktMDcucnVyLTIwMTktMDcucjEnLCAnMTIuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMl8yXzBfMV8yMDE5XzEwX1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTIuMi4wLjEucnUtMjAxOS0xMC5ydXItMjAxOS0xMC5yMScsICcxMi4yJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEyXzJfMF8xXzIwMjBfMDFfUjEgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxMi4yLjAuMS5ydS0yMDIwLTAxLnJ1ci0yMDIwLTAxLnIxJywgJzEyLjInKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMl8wXzFfMjAyMF8wNF9SMSA9IE9yYWNsZUVuZ2luZVZlcnNpb24ub2YoJzEyLjIuMC4xLnJ1LTIwMjAtMDQucnVyLTIwMjAtMDQucjEnLCAnMTIuMicpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTggPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxOCcsICcxOCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOF8wXzBfMF8yMDE5XzA3X1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTguMC4wLjAucnUtMjAxOS0wNy5ydXItMjAxOS0wNy5yMScsICcxOCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOF8wXzBfMF8yMDE5XzEwX1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTguMC4wLjAucnUtMjAxOS0xMC5ydXItMjAxOS0xMC5yMScsICcxOCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOF8wXzBfMF8yMDIwXzAxX1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTguMC4wLjAucnUtMjAyMC0wMS5ydXItMjAyMC0wMS5yMScsICcxOCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOF8wXzBfMF8yMDIwXzA0X1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTguMC4wLjAucnUtMjAyMC0wNC5ydXItMjAyMC0wNC5yMScsICcxOCcpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTkgPSBPcmFjbGVFbmdpbmVWZXJzaW9uLm9mKCcxOScsICcxOScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOV8wXzBfMF8yMDE5XzA3X1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTkuMC4wLjAucnUtMjAxOS0wNy5ydXItMjAxOS0wNy5yMScsICcxOScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOV8wXzBfMF8yMDE5XzEwX1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTkuMC4wLjAucnUtMjAxOS0xMC5ydXItMjAxOS0xMC5yMScsICcxOScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOV8wXzBfMF8yMDIwXzAxX1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTkuMC4wLjAucnUtMjAyMC0wMS5ydXItMjAyMC0wMS5yMScsICcxOScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xOV8wXzBfMF8yMDIwXzA0X1IxID0gT3JhY2xlRW5naW5lVmVyc2lvbi5vZignMTkuMC4wLjAucnUtMjAyMC0wNC5ydXItMjAyMC0wNC5yMScsICcxOScpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBvZihvcmFjbGVGdWxsVmVyc2lvbjogc3RyaW5nLCBvcmFjbGVNYWpvclZlcnNpb246IHN0cmluZyk6IE9yYWNsZUVuZ2luZVZlcnNpb24ge1xuICAgICAgICByZXR1cm4gbmV3IE9yYWNsZUVuZ2luZVZlcnNpb24ob3JhY2xlRnVsbFZlcnNpb24sIG9yYWNsZU1ham9yVmVyc2lvbik7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgb3JhY2xlRnVsbFZlcnNpb246IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IG9yYWNsZU1ham9yVmVyc2lvbjogc3RyaW5nO1xuICAgIHByaXZhdGUgY29uc3RydWN0b3Iob3JhY2xlRnVsbFZlcnNpb246IHN0cmluZywgb3JhY2xlTWFqb3JWZXJzaW9uOiBzdHJpbmcpIHtcbiAgICAgICAgdGhpcy5vcmFjbGVGdWxsVmVyc2lvbiA9IG9yYWNsZUZ1bGxWZXJzaW9uO1xuICAgICAgICB0aGlzLm9yYWNsZU1ham9yVmVyc2lvbiA9IG9yYWNsZU1ham9yVmVyc2lvbjtcbiAgICB9XG59XG5pbnRlcmZhY2UgT3JhY2xlSW5zdGFuY2VFbmdpbmVCYXNlUHJvcHMge1xuICAgIHJlYWRvbmx5IGVuZ2luZVR5cGU6IHN0cmluZztcbiAgICByZWFkb25seSB2ZXJzaW9uPzogRW5naW5lVmVyc2lvbjtcbn1cbmFic3RyYWN0IGNsYXNzIE9yYWNsZUluc3RhbmNlRW5naW5lQmFzZSBleHRlbmRzIEluc3RhbmNlRW5naW5lQmFzZSB7XG4gICAgY29uc3RydWN0b3IocHJvcHM6IE9yYWNsZUluc3RhbmNlRW5naW5lQmFzZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgc2luZ2xlVXNlclJvdGF0aW9uQXBwbGljYXRpb246IHNlY3JldHNtYW5hZ2VyLlNlY3JldFJvdGF0aW9uQXBwbGljYXRpb24uT1JBQ0xFX1JPVEFUSU9OX1NJTkdMRV9VU0VSLFxuICAgICAgICAgICAgbXVsdGlVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbi5PUkFDTEVfUk9UQVRJT05fTVVMVElfVVNFUixcbiAgICAgICAgICAgIHBhcmFtZXRlckdyb3VwRmFtaWx5OiBwcm9wcy52ZXJzaW9uID8gYCR7cHJvcHMuZW5naW5lVHlwZX0tJHtwcm9wcy52ZXJzaW9uLm1ham9yVmVyc2lvbn1gIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgZmVhdHVyZXM6IHtcbiAgICAgICAgICAgICAgICBzM0ltcG9ydDogJ1MzX0lOVEVHUkFUSU9OJyxcbiAgICAgICAgICAgICAgICBzM0V4cG9ydDogJ1MzX0lOVEVHUkFUSU9OJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZFRvSW5zdGFuY2Uoc2NvcGU6IGNvcmUuQ29uc3RydWN0LCBvcHRpb25zOiBJbnN0YW5jZUVuZ2luZUJpbmRPcHRpb25zKTogSW5zdGFuY2VFbmdpbmVDb25maWcge1xuICAgICAgICBjb25zdCBjb25maWcgPSBzdXBlci5iaW5kVG9JbnN0YW5jZShzY29wZSwgb3B0aW9ucyk7XG4gICAgICAgIGxldCBvcHRpb25Hcm91cCA9IG9wdGlvbnMub3B0aW9uR3JvdXA7XG4gICAgICAgIGlmIChvcHRpb25zLnMzSW1wb3J0Um9sZSB8fCBvcHRpb25zLnMzRXhwb3J0Um9sZSkge1xuICAgICAgICAgICAgaWYgKCFvcHRpb25Hcm91cCkge1xuICAgICAgICAgICAgICAgIG9wdGlvbkdyb3VwID0gbmV3IE9wdGlvbkdyb3VwKHNjb3BlLCAnSW5zdGFuY2VPcHRpb25Hcm91cCcsIHtcbiAgICAgICAgICAgICAgICAgICAgZW5naW5lOiB0aGlzLFxuICAgICAgICAgICAgICAgICAgICBjb25maWd1cmF0aW9uczogW10sXG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvb3JhY2xlLXMzLWludGVncmF0aW9uLmh0bWxcbiAgICAgICAgICAgIG9wdGlvbkdyb3VwLmFkZENvbmZpZ3VyYXRpb24oe1xuICAgICAgICAgICAgICAgIG5hbWU6ICdTM19JTlRFR1JBVElPTicsXG4gICAgICAgICAgICAgICAgdmVyc2lvbjogJzEuMCcsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgLi4uY29uZmlnLFxuICAgICAgICAgICAgb3B0aW9uR3JvdXAsXG4gICAgICAgIH07XG4gICAgfVxufVxuaW50ZXJmYWNlIE9yYWNsZUluc3RhbmNlRW5naW5lUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2ZXJzaW9uOiBPcmFjbGVFbmdpbmVWZXJzaW9uO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBPcmFjbGVTZUluc3RhbmNlRW5naW5lUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2ZXJzaW9uOiBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uO1xufVxuLyoqIEBkZXByZWNhdGVkIGluc3RhbmNlcyBjYW4gbm8gbG9uZ2VyIGJlIGNyZWF0ZWQgd2l0aCB0aGlzIGVuZ2luZS4gU2VlIGh0dHBzOi8vZm9ydW1zLmF3cy5hbWF6b24uY29tL2Fubi5qc3BhP2FubklEPTczNDEgKi9cbmNsYXNzIE9yYWNsZVNlSW5zdGFuY2VFbmdpbmUgZXh0ZW5kcyBPcmFjbGVJbnN0YW5jZUVuZ2luZUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIGVuZ2luZVR5cGU6ICdvcmFjbGUtc2UnLFxuICAgICAgICAgICAgdmVyc2lvbjogdmVyc2lvblxuICAgICAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgICAgICBmdWxsVmVyc2lvbjogdmVyc2lvbi5vcmFjbGVMZWdhY3lGdWxsVmVyc2lvbixcbiAgICAgICAgICAgICAgICAgICAgbWFqb3JWZXJzaW9uOiB2ZXJzaW9uLm9yYWNsZUxlZ2FjeU1ham9yVmVyc2lvbixcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgOiB7XG4gICAgICAgICAgICAgICAgICAgIG1ham9yVmVyc2lvbjogJzExLjInLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgT3JhY2xlU2UxSW5zdGFuY2VFbmdpbmVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZlcnNpb246IE9yYWNsZUxlZ2FjeUVuZ2luZVZlcnNpb247XG59XG4vKiogQGRlcHJlY2F0ZWQgaW5zdGFuY2VzIGNhbiBubyBsb25nZXIgYmUgY3JlYXRlZCB3aXRoIHRoaXMgZW5naW5lLiBTZWUgaHR0cHM6Ly9mb3J1bXMuYXdzLmFtYXpvbi5jb20vYW5uLmpzcGE/YW5uSUQ9NzM0MSAqL1xuY2xhc3MgT3JhY2xlU2UxSW5zdGFuY2VFbmdpbmUgZXh0ZW5kcyBPcmFjbGVJbnN0YW5jZUVuZ2luZUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBPcmFjbGVMZWdhY3lFbmdpbmVWZXJzaW9uKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIGVuZ2luZVR5cGU6ICdvcmFjbGUtc2UxJyxcbiAgICAgICAgICAgIHZlcnNpb246IHZlcnNpb25cbiAgICAgICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICAgICAgZnVsbFZlcnNpb246IHZlcnNpb24ub3JhY2xlTGVnYWN5RnVsbFZlcnNpb24sXG4gICAgICAgICAgICAgICAgICAgIG1ham9yVmVyc2lvbjogdmVyc2lvbi5vcmFjbGVMZWdhY3lNYWpvclZlcnNpb24sXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIDoge1xuICAgICAgICAgICAgICAgICAgICBtYWpvclZlcnNpb246ICcxMS4yJyxcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBPcmFjbGVTZTJJbnN0YW5jZUVuZ2luZVByb3BzIGV4dGVuZHMgT3JhY2xlSW5zdGFuY2VFbmdpbmVQcm9wcyB7XG59XG5jbGFzcyBPcmFjbGVTZTJJbnN0YW5jZUVuZ2luZSBleHRlbmRzIE9yYWNsZUluc3RhbmNlRW5naW5lQmFzZSB7XG4gICAgY29uc3RydWN0b3IodmVyc2lvbj86IE9yYWNsZUVuZ2luZVZlcnNpb24pIHtcbiAgICAgICAgc3VwZXIoe1xuICAgICAgICAgICAgZW5naW5lVHlwZTogJ29yYWNsZS1zZTInLFxuICAgICAgICAgICAgdmVyc2lvbjogdmVyc2lvblxuICAgICAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgICAgICBmdWxsVmVyc2lvbjogdmVyc2lvbi5vcmFjbGVGdWxsVmVyc2lvbixcbiAgICAgICAgICAgICAgICAgICAgbWFqb3JWZXJzaW9uOiB2ZXJzaW9uLm9yYWNsZU1ham9yVmVyc2lvbixcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgT3JhY2xlRWVJbnN0YW5jZUVuZ2luZVByb3BzIGV4dGVuZHMgT3JhY2xlSW5zdGFuY2VFbmdpbmVQcm9wcyB7XG59XG5jbGFzcyBPcmFjbGVFZUluc3RhbmNlRW5naW5lIGV4dGVuZHMgT3JhY2xlSW5zdGFuY2VFbmdpbmVCYXNlIHtcbiAgICBjb25zdHJ1Y3Rvcih2ZXJzaW9uPzogT3JhY2xlRW5naW5lVmVyc2lvbikge1xuICAgICAgICBzdXBlcih7XG4gICAgICAgICAgICBlbmdpbmVUeXBlOiAnb3JhY2xlLWVlJyxcbiAgICAgICAgICAgIHZlcnNpb246IHZlcnNpb25cbiAgICAgICAgICAgICAgICA/IHtcbiAgICAgICAgICAgICAgICAgICAgZnVsbFZlcnNpb246IHZlcnNpb24ub3JhY2xlRnVsbFZlcnNpb24sXG4gICAgICAgICAgICAgICAgICAgIG1ham9yVmVyc2lvbjogdmVyc2lvbi5vcmFjbGVNYWpvclZlcnNpb24sXG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgU3FsU2VydmVyRW5naW5lVmVyc2lvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMSA9IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ub2YoJzExLjAwJywgJzExLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMDBfNTA1OF8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTEuMDAuNTA1OC4wLnYxJywgJzExLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMDBfNjAyMF8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTEuMDAuNjAyMC4wLnYxJywgJzExLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMDBfNjU5NF8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTEuMDAuNjU5NC4wLnYxJywgJzExLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMDBfNzQ2Ml82X1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTEuMDAuNzQ2Mi42LnYxJywgJzExLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTFfMDBfNzQ5M180X1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTEuMDAuNzQ5My40LnYxJywgJzExLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMiA9IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ub2YoJzEyLjAwJywgJzEyLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMDBfNTAwMF8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTIuMDAuNTAwMC4wLnYxJywgJzEyLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMDBfNTU0Nl8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTIuMDAuNTU0Ni4wLnYxJywgJzEyLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMDBfNTU3MV8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTIuMDAuNTU3MS4wLnYxJywgJzEyLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMDBfNjI5M18wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTIuMDAuNjI5My4wLnYxJywgJzEyLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTJfMDBfNjMyOV8xX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTIuMDAuNjMyOS4xLnYxJywgJzEyLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xMyA9IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ub2YoJzEzLjAwJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfMjE2NF8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuMjE2NC4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNDQyMl8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNDQyMi4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNDQ1MV8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNDQ1MS4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNDQ2Nl80X1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNDQ2Ni40LnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNDUyMl8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNDUyMi4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNTIxNl8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNTIxNi4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNTI5Ml8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNTI5Mi4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNTM2Nl8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNTM2Ni4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTNfMDBfNTQyNl8wX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTMuMDAuNTQyNi4wLnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzEzXzAwXzU1OThfMjdfVjEgPSBTcWxTZXJ2ZXJFbmdpbmVWZXJzaW9uLm9mKCcxMy4wMC41NTk4LjI3LnYxJywgJzEzLjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xNCA9IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ub2YoJzE0LjAwJywgJzE0LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xNF8wMF8xMDAwXzE2OV9WMSA9IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ub2YoJzE0LjAwLjEwMDAuMTY5LnYxJywgJzE0LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzE0XzAwXzMwMTVfNDBfVjEgPSBTcWxTZXJ2ZXJFbmdpbmVWZXJzaW9uLm9mKCcxNC4wMC4zMDE1LjQwLnYxJywgJzE0LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTRfMDBfMzAzNV8yX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTQuMDAuMzAzNS4yLnYxJywgJzE0LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTRfMDBfMzA0OV8xX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTQuMDAuMzA0OS4xLnYxJywgJzE0LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBWRVJfMTRfMDBfMzE5Ml8yX1YxID0gU3FsU2VydmVyRW5naW5lVmVyc2lvbi5vZignMTQuMDAuMzE5Mi4yLnYxJywgJzE0LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFZFUl8xNSA9IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ub2YoJzE1LjAwJywgJzE1LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgVkVSXzE1XzAwXzQwNDNfMTZfVjEgPSBTcWxTZXJ2ZXJFbmdpbmVWZXJzaW9uLm9mKCcxNS4wMC40MDQzLjE2LnYxJywgJzE1LjAwJyk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBvZihzcWxTZXJ2ZXJGdWxsVmVyc2lvbjogc3RyaW5nLCBzcWxTZXJ2ZXJNYWpvclZlcnNpb246IHN0cmluZyk6IFNxbFNlcnZlckVuZ2luZVZlcnNpb24ge1xuICAgICAgICByZXR1cm4gbmV3IFNxbFNlcnZlckVuZ2luZVZlcnNpb24oc3FsU2VydmVyRnVsbFZlcnNpb24sIHNxbFNlcnZlck1ham9yVmVyc2lvbik7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBzcWxTZXJ2ZXJGdWxsVmVyc2lvbjogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgc3FsU2VydmVyTWFqb3JWZXJzaW9uOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihzcWxTZXJ2ZXJGdWxsVmVyc2lvbjogc3RyaW5nLCBzcWxTZXJ2ZXJNYWpvclZlcnNpb246IHN0cmluZykge1xuICAgICAgICB0aGlzLnNxbFNlcnZlckZ1bGxWZXJzaW9uID0gc3FsU2VydmVyRnVsbFZlcnNpb247XG4gICAgICAgIHRoaXMuc3FsU2VydmVyTWFqb3JWZXJzaW9uID0gc3FsU2VydmVyTWFqb3JWZXJzaW9uO1xuICAgIH1cbn1cbmludGVyZmFjZSBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgdmVyc2lvbjogU3FsU2VydmVyRW5naW5lVmVyc2lvbjtcbn1cbmludGVyZmFjZSBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZUJhc2VQcm9wcyB7XG4gICAgcmVhZG9ubHkgZW5naW5lVHlwZTogc3RyaW5nO1xuICAgIHJlYWRvbmx5IHZlcnNpb24/OiBTcWxTZXJ2ZXJFbmdpbmVWZXJzaW9uO1xufVxuYWJzdHJhY3QgY2xhc3MgU3FsU2VydmVySW5zdGFuY2VFbmdpbmVCYXNlIGV4dGVuZHMgSW5zdGFuY2VFbmdpbmVCYXNlIHtcbiAgICBjb25zdHJ1Y3Rvcihwcm9wczogU3FsU2VydmVySW5zdGFuY2VFbmdpbmVCYXNlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoe1xuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICBzaW5nbGVVc2VyUm90YXRpb25BcHBsaWNhdGlvbjogc2VjcmV0c21hbmFnZXIuU2VjcmV0Um90YXRpb25BcHBsaWNhdGlvbi5TUUxTRVJWRVJfUk9UQVRJT05fU0lOR0xFX1VTRVIsXG4gICAgICAgICAgICBtdWx0aVVzZXJSb3RhdGlvbkFwcGxpY2F0aW9uOiBzZWNyZXRzbWFuYWdlci5TZWNyZXRSb3RhdGlvbkFwcGxpY2F0aW9uLlNRTFNFUlZFUl9ST1RBVElPTl9NVUxUSV9VU0VSLFxuICAgICAgICAgICAgdmVyc2lvbjogcHJvcHMudmVyc2lvblxuICAgICAgICAgICAgICAgID8ge1xuICAgICAgICAgICAgICAgICAgICBmdWxsVmVyc2lvbjogcHJvcHMudmVyc2lvbi5zcWxTZXJ2ZXJGdWxsVmVyc2lvbixcbiAgICAgICAgICAgICAgICAgICAgbWFqb3JWZXJzaW9uOiBwcm9wcy52ZXJzaW9uLnNxbFNlcnZlck1ham9yVmVyc2lvbixcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICBwYXJhbWV0ZXJHcm91cEZhbWlseTogcHJvcHMudmVyc2lvblxuICAgICAgICAgICAgICAgIC8vIGZvciBzb21lIHJlYXNvbiwgZXZlbiB0aG91Z2ggU1FMIFNlcnZlciBtYWpvciB2ZXJzaW9ucyB1c3VhbGx5IGVuZCBpbiAnLjAwJyxcbiAgICAgICAgICAgICAgICAvLyB0aGUgUGFyYW1ldGVyR3JvdXAgZmFtaWx5IGhhcyB0byBlbmQgaW4gJy4wJ1xuICAgICAgICAgICAgICAgID8gYCR7cHJvcHMuZW5naW5lVHlwZX0tJHtwcm9wcy52ZXJzaW9uLnNxbFNlcnZlck1ham9yVmVyc2lvbi5lbmRzV2l0aCgnLjAwJylcbiAgICAgICAgICAgICAgICAgICAgPyBwcm9wcy52ZXJzaW9uLnNxbFNlcnZlck1ham9yVmVyc2lvbi5zbGljZSgwLCAtMSlcbiAgICAgICAgICAgICAgICAgICAgOiBwcm9wcy52ZXJzaW9uLnNxbFNlcnZlck1ham9yVmVyc2lvbn1gXG4gICAgICAgICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICBmZWF0dXJlczoge1xuICAgICAgICAgICAgICAgIHMzSW1wb3J0OiAnUzNfSU5URUdSQVRJT04nLFxuICAgICAgICAgICAgICAgIHMzRXhwb3J0OiAnUzNfSU5URUdSQVRJT04nLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHB1YmxpYyBiaW5kVG9JbnN0YW5jZShzY29wZTogY29yZS5Db25zdHJ1Y3QsIG9wdGlvbnM6IEluc3RhbmNlRW5naW5lQmluZE9wdGlvbnMpOiBJbnN0YW5jZUVuZ2luZUNvbmZpZyB7XG4gICAgICAgIGNvbnN0IGNvbmZpZyA9IHN1cGVyLmJpbmRUb0luc3RhbmNlKHNjb3BlLCBvcHRpb25zKTtcbiAgICAgICAgbGV0IG9wdGlvbkdyb3VwID0gb3B0aW9ucy5vcHRpb25Hcm91cDtcbiAgICAgICAgY29uc3QgczNSb2xlID0gb3B0aW9ucy5zM0ltcG9ydFJvbGUgPz8gb3B0aW9ucy5zM0V4cG9ydFJvbGU7XG4gICAgICAgIGlmIChzM1JvbGUpIHtcbiAgICAgICAgICAgIGlmIChvcHRpb25zLnMzSW1wb3J0Um9sZSAmJiBvcHRpb25zLnMzRXhwb3J0Um9sZSAmJiBvcHRpb25zLnMzSW1wb3J0Um9sZSAhPT0gb3B0aW9ucy5zM0V4cG9ydFJvbGUpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1MzIGltcG9ydCBhbmQgZXhwb3J0IHJvbGVzIG11c3QgYmUgdGhlIHNhbWUgZm9yIFNRTCBTZXJ2ZXIgZW5naW5lcycpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgaWYgKCFvcHRpb25Hcm91cCkge1xuICAgICAgICAgICAgICAgIG9wdGlvbkdyb3VwID0gbmV3IE9wdGlvbkdyb3VwKHNjb3BlLCAnSW5zdGFuY2VPcHRpb25Hcm91cCcsIHtcbiAgICAgICAgICAgICAgICAgICAgZW5naW5lOiB0aGlzLFxuICAgICAgICAgICAgICAgICAgICBjb25maWd1cmF0aW9uczogW10sXG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUkRTL2xhdGVzdC9Vc2VyR3VpZGUvQXBwZW5kaXguU1FMU2VydmVyLk9wdGlvbnMuQmFja3VwUmVzdG9yZS5odG1sXG4gICAgICAgICAgICBvcHRpb25Hcm91cC5hZGRDb25maWd1cmF0aW9uKHtcbiAgICAgICAgICAgICAgICBuYW1lOiAnU1FMU0VSVkVSX0JBQ0tVUF9SRVNUT1JFJyxcbiAgICAgICAgICAgICAgICBzZXR0aW5nczogeyBJQU1fUk9MRV9BUk46IHMzUm9sZS5yb2xlQXJuIH0sXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgLi4uY29uZmlnLFxuICAgICAgICAgICAgb3B0aW9uR3JvdXAsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByb3RlY3RlZCBnZXQgc3VwcG9ydHNUaW1lem9uZSgpIHsgcmV0dXJuIHRydWU7IH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTcWxTZXJ2ZXJTZUluc3RhbmNlRW5naW5lUHJvcHMgZXh0ZW5kcyBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZVByb3BzIHtcbn1cbmNsYXNzIFNxbFNlcnZlclNlSW5zdGFuY2VFbmdpbmUgZXh0ZW5kcyBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBTcWxTZXJ2ZXJFbmdpbmVWZXJzaW9uKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIGVuZ2luZVR5cGU6ICdzcWxzZXJ2ZXItc2UnLFxuICAgICAgICAgICAgdmVyc2lvbixcbiAgICAgICAgfSk7XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgU3FsU2VydmVyRXhJbnN0YW5jZUVuZ2luZVByb3BzIGV4dGVuZHMgU3FsU2VydmVySW5zdGFuY2VFbmdpbmVQcm9wcyB7XG59XG5jbGFzcyBTcWxTZXJ2ZXJFeEluc3RhbmNlRW5naW5lIGV4dGVuZHMgU3FsU2VydmVySW5zdGFuY2VFbmdpbmVCYXNlIHtcbiAgICBjb25zdHJ1Y3Rvcih2ZXJzaW9uPzogU3FsU2VydmVyRW5naW5lVmVyc2lvbikge1xuICAgICAgICBzdXBlcih7XG4gICAgICAgICAgICBlbmdpbmVUeXBlOiAnc3Fsc2VydmVyLWV4JyxcbiAgICAgICAgICAgIHZlcnNpb24sXG4gICAgICAgIH0pO1xuICAgIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFNxbFNlcnZlcldlYkluc3RhbmNlRW5naW5lUHJvcHMgZXh0ZW5kcyBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZVByb3BzIHtcbn1cbmNsYXNzIFNxbFNlcnZlcldlYkluc3RhbmNlRW5naW5lIGV4dGVuZHMgU3FsU2VydmVySW5zdGFuY2VFbmdpbmVCYXNlIHtcbiAgICBjb25zdHJ1Y3Rvcih2ZXJzaW9uPzogU3FsU2VydmVyRW5naW5lVmVyc2lvbikge1xuICAgICAgICBzdXBlcih7XG4gICAgICAgICAgICBlbmdpbmVUeXBlOiAnc3Fsc2VydmVyLXdlYicsXG4gICAgICAgICAgICB2ZXJzaW9uLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBTcWxTZXJ2ZXJFZUluc3RhbmNlRW5naW5lUHJvcHMgZXh0ZW5kcyBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZVByb3BzIHtcbn1cbmNsYXNzIFNxbFNlcnZlckVlSW5zdGFuY2VFbmdpbmUgZXh0ZW5kcyBTcWxTZXJ2ZXJJbnN0YW5jZUVuZ2luZUJhc2Uge1xuICAgIGNvbnN0cnVjdG9yKHZlcnNpb24/OiBTcWxTZXJ2ZXJFbmdpbmVWZXJzaW9uKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIGVuZ2luZVR5cGU6ICdzcWxzZXJ2ZXItZWUnLFxuICAgICAgICAgICAgdmVyc2lvbixcbiAgICAgICAgfSk7XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgRGF0YWJhc2VJbnN0YW5jZUVuZ2luZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE1BUklBREI6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBNYXJpYURiSW5zdGFuY2VFbmdpbmUoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBNWVNRTDogSUluc3RhbmNlRW5naW5lID0gbmV3IE15U3FsSW5zdGFuY2VFbmdpbmUoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgT1JBQ0xFX0VFOiBJSW5zdGFuY2VFbmdpbmUgPSBuZXcgT3JhY2xlRWVJbnN0YW5jZUVuZ2luZSgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IE9SQUNMRV9TRTI6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBPcmFjbGVTZTJJbnN0YW5jZUVuZ2luZSgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBPUkFDTEVfU0UxOiBJSW5zdGFuY2VFbmdpbmUgPSBuZXcgT3JhY2xlU2UxSW5zdGFuY2VFbmdpbmUoKTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBPUkFDTEVfU0U6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBPcmFjbGVTZUluc3RhbmNlRW5naW5lKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgUE9TVEdSRVM6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBQb3N0Z3Jlc0luc3RhbmNlRW5naW5lKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNRTF9TRVJWRVJfRUU6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBTcWxTZXJ2ZXJFZUluc3RhbmNlRW5naW5lKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNRTF9TRVJWRVJfU0U6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBTcWxTZXJ2ZXJTZUluc3RhbmNlRW5naW5lKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIHJlYWRvbmx5IFNRTF9TRVJWRVJfRVg6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBTcWxTZXJ2ZXJFeEluc3RhbmNlRW5naW5lKCk7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgcmVhZG9ubHkgU1FMX1NFUlZFUl9XRUI6IElJbnN0YW5jZUVuZ2luZSA9IG5ldyBTcWxTZXJ2ZXJXZWJJbnN0YW5jZUVuZ2luZSgpO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgbWFyaWFEYihwcm9wczogTWFyaWFEYkluc3RhbmNlRW5naW5lUHJvcHMpOiBJSW5zdGFuY2VFbmdpbmUge1xuICAgICAgICByZXR1cm4gbmV3IE1hcmlhRGJJbnN0YW5jZUVuZ2luZShwcm9wcy52ZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgbXlzcWwocHJvcHM6IE15U3FsSW5zdGFuY2VFbmdpbmVQcm9wcyk6IElJbnN0YW5jZUVuZ2luZSB7XG4gICAgICAgIHJldHVybiBuZXcgTXlTcWxJbnN0YW5jZUVuZ2luZShwcm9wcy52ZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBwb3N0Z3Jlcyhwcm9wczogUG9zdGdyZXNJbnN0YW5jZUVuZ2luZVByb3BzKTogSUluc3RhbmNlRW5naW5lIHtcbiAgICAgICAgcmV0dXJuIG5ldyBQb3N0Z3Jlc0luc3RhbmNlRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBvcmFjbGVTZShwcm9wczogT3JhY2xlU2VJbnN0YW5jZUVuZ2luZVByb3BzKTogSUluc3RhbmNlRW5naW5lIHtcbiAgICAgICAgcmV0dXJuIG5ldyBPcmFjbGVTZUluc3RhbmNlRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIG9yYWNsZVNlMShwcm9wczogT3JhY2xlU2UxSW5zdGFuY2VFbmdpbmVQcm9wcyk6IElJbnN0YW5jZUVuZ2luZSB7XG4gICAgICAgIHJldHVybiBuZXcgT3JhY2xlU2UxSW5zdGFuY2VFbmdpbmUocHJvcHMudmVyc2lvbik7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgb3JhY2xlU2UyKHByb3BzOiBPcmFjbGVTZTJJbnN0YW5jZUVuZ2luZVByb3BzKTogSUluc3RhbmNlRW5naW5lIHtcbiAgICAgICAgcmV0dXJuIG5ldyBPcmFjbGVTZTJJbnN0YW5jZUVuZ2luZShwcm9wcy52ZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBvcmFjbGVFZShwcm9wczogT3JhY2xlRWVJbnN0YW5jZUVuZ2luZVByb3BzKTogSUluc3RhbmNlRW5naW5lIHtcbiAgICAgICAgcmV0dXJuIG5ldyBPcmFjbGVFZUluc3RhbmNlRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgc3FsU2VydmVyU2UocHJvcHM6IFNxbFNlcnZlclNlSW5zdGFuY2VFbmdpbmVQcm9wcyk6IElJbnN0YW5jZUVuZ2luZSB7XG4gICAgICAgIHJldHVybiBuZXcgU3FsU2VydmVyU2VJbnN0YW5jZUVuZ2luZShwcm9wcy52ZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyBzdGF0aWMgc3FsU2VydmVyRXgocHJvcHM6IFNxbFNlcnZlckV4SW5zdGFuY2VFbmdpbmVQcm9wcyk6IElJbnN0YW5jZUVuZ2luZSB7XG4gICAgICAgIHJldHVybiBuZXcgU3FsU2VydmVyRXhJbnN0YW5jZUVuZ2luZShwcm9wcy52ZXJzaW9uKTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBzcWxTZXJ2ZXJXZWIocHJvcHM6IFNxbFNlcnZlcldlYkluc3RhbmNlRW5naW5lUHJvcHMpOiBJSW5zdGFuY2VFbmdpbmUge1xuICAgICAgICByZXR1cm4gbmV3IFNxbFNlcnZlcldlYkluc3RhbmNlRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBzcWxTZXJ2ZXJFZShwcm9wczogU3FsU2VydmVyRWVJbnN0YW5jZUVuZ2luZVByb3BzKTogSUluc3RhbmNlRW5naW5lIHtcbiAgICAgICAgcmV0dXJuIG5ldyBTcWxTZXJ2ZXJFZUluc3RhbmNlRW5naW5lKHByb3BzLnZlcnNpb24pO1xuICAgIH1cbn1cbiJdfQ==