"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.features = props.features;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
            features: this.features,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.engineFamily = 'MYSQL';
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * (experimental) The versions for the Aurora cluster engine (those returned by {@link DatabaseClusterEngine.aurora}).
 *
 * @experimental
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * (experimental) Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string, for example "5.6.mysql_aurora.1.78.3.6".
     * @param auroraMajorVersion the major version of the engine, defaults to "5.6".
     * @experimental
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/**
 * (experimental) Version "5.6.10a".
 *
 * @experimental
 */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/**
 * (experimental) Version "5.6.mysql_aurora.1.17.9".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/**
 * (experimental) Version "5.6.mysql_aurora.1.19.0".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/**
 * (experimental) Version "5.6.mysql_aurora.1.19.1".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/**
 * (experimental) Version "5.6.mysql_aurora.1.19.2".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/**
 * (experimental) Version "5.6.mysql_aurora.1.19.5".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/**
 * (experimental) Version "5.6.mysql_aurora.1.19.6".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/**
 * (experimental) Version "5.6.mysql_aurora.1.20.0".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/**
 * (experimental) Version "5.6.mysql_aurora.1.20.1".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/**
 * (experimental) Version "5.6.mysql_aurora.1.21.0".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/**
 * (experimental) Version "5.6.mysql_aurora.1.22.0".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/**
 * (experimental) Version "5.6.mysql_aurora.1.22.1".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/**
 * (experimental) Version "5.6.mysql_aurora.1.22.1.3".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/**
 * (experimental) Version "5.6.mysql_aurora.1.22.2".
 *
 * @experimental
 */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * (experimental) The versions for the Aurora MySQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 *
 * @experimental
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * (experimental) Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string, for example "5.7.mysql_aurora.2.78.3.6".
     * @param auroraMysqlMajorVersion the major version of the engine, defaults to "5.7".
     * @experimental
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/**
 * (experimental) Version "5.7.12".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/**
 * (experimental) Version "5.7.mysql_aurora.2.03.2".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/**
 * (experimental) Version "5.7.mysql_aurora.2.03.3".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/**
 * (experimental) Version "5.7.mysql_aurora.2.03.4".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.0".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.1".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.2".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.3".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.4".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.5".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.6".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.7".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/**
 * (experimental) Version "5.7.mysql_aurora.2.04.8".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/**
 * (experimental) Version "5.7.mysql_aurora.2.05.0".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/**
 * (experimental) Version "5.7.mysql_aurora.2.06.0".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/**
 * (experimental) Version "5.7.mysql_aurora.2.07.0".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/**
 * (experimental) Version "5.7.mysql_aurora.2.07.1".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/**
 * (experimental) Version "5.7.mysql_aurora.2.07.2".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/**
 * (experimental) Version "5.7.mysql_aurora.2.08.0".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/**
 * (experimental) Version "5.7.mysql_aurora.2.08.1".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
/**
 * (experimental) Version "5.7.mysql_aurora.2.08.2".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_08_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.2');
/**
 * (experimental) Version "5.7.mysql_aurora.2.09.0".
 *
 * @experimental
 */
AuroraMysqlEngineVersion.VER_2_09_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.09.0');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * (experimental) The versions for the Aurora PostgreSQL cluster engine (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 *
 * @experimental
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
        this._features = {
            s3Import: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Import) ? 's3Import' : undefined,
            s3Export: (auroraPostgresFeatures === null || auroraPostgresFeatures === void 0 ? void 0 : auroraPostgresFeatures.s3Export) ? 's3Export' : undefined,
        };
    }
    /**
     * (experimental) Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string, for example "9.6.25.1".
     * @param auroraPostgresMajorVersion the major version of the engine, for example "9.6".
     * @experimental
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion, auroraPostgresFeatures);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/**
 * (experimental) Version "9.6.8".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/**
 * (experimental) Version "9.6.9".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/**
 * (experimental) Version "9.6.11".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/**
 * (experimental) Version "9.6.12".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/**
 * (experimental) Version "9.6.16".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/**
 * (experimental) Version "9.6.17".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/**
 * (experimental) Version "10.4".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/**
 * (experimental) Version "10.5".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/**
 * (experimental) Version "10.6".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/**
 * (experimental) Version "10.7".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10', { s3Import: true });
/**
 * (experimental) Version "10.11".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10', { s3Import: true, s3Export: true });
/**
 * (experimental) Version "10.12".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10', { s3Import: true, s3Export: true });
/**
 * (experimental) Version "11.4".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11', { s3Import: true });
/**
 * (experimental) Version "11.6".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11', { s3Import: true, s3Export: true });
/**
 * (experimental) Version "11.7".
 *
 * @experimental
 */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11', { s3Import: true, s3Export: true });
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
            features: version
                ? {
                    s3Import: version._features.s3Import ? AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME : undefined,
                    s3Export: version._features.s3Export ? AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME : undefined,
                }
                : {
                    s3Import: AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME,
                    s3Export: AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME,
                },
        });
        this.engineFamily = 'POSTGRESQL';
        this.defaultUsername = 'postgres';
        this.supportedLogTypes = ['postgresql'];
    }
    bindToCluster(scope, options) {
        var _a, _b, _c;
        const config = super.bindToCluster(scope, options);
        // skip validation for unversioned as it might be supported/unsupported. we cannot reliably tell at compile-time
        if ((_a = this.engineVersion) === null || _a === void 0 ? void 0 : _a.fullVersion) {
            if (options.s3ImportRole && !((_b = config.features) === null || _b === void 0 ? void 0 : _b.s3Import)) {
                throw new Error(`s3Import is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Import feature.`);
            }
            if (options.s3ExportRole && !((_c = config.features) === null || _c === void 0 ? void 0 : _c.s3Export)) {
                throw new Error(`s3Export is not supported for Postgres version: ${this.engineVersion.fullVersion}. Use a version that supports the s3Export feature.`);
            }
        }
        return config;
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * feature name for the S3 data import feature
 */
AuroraPostgresClusterEngine.S3_IMPORT_FEATURE_NAME = 's3Import';
/**
 * feature name for the S3 data export feature
 */
AuroraPostgresClusterEngine.S3_EXPORT_FEATURE_NAME = 's3Export';
/**
 * (experimental) A database cluster engine.
 *
 * Provides mapping to the serverless application
 * used for secret rotation.
 *
 * @experimental
 */
class DatabaseClusterEngine {
    /**
     * (experimental) Creates a new plain Aurora database cluster engine.
     *
     * @experimental
     */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /**
     * (experimental) Creates a new Aurora MySQL database cluster engine.
     *
     * @experimental
     */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /**
     * (experimental) Creates a new Aurora PostgreSQL database cluster engine.
     *
     * @experimental
     */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * (experimental) The unversioned 'aurora' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link aurora()} method
 *
 * @experimental
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * (experimental) The unversioned 'aurora-msql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraMysql()} method
 *
 * @experimental
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * (experimental) The unversioned 'aurora-postgresql' cluster engine.
 *
 * **Note**: we do not recommend using unversioned engines for non-serverless Clusters,
 *    as that can pose an availability risk.
 *    We recommend using versioned engines created using the {@link auroraPostgres()} method
 *
 * @experimental
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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