"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossAccountDestination = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const logs_generated_1 = require("./logs.generated");
/**
 * (experimental) A new CloudWatch Logs Destination for use in cross-account scenarios.
 *
 * CrossAccountDestinations are used to subscribe a Kinesis stream in a
 * different account to a CloudWatch Subscription.
 *
 * Consumers will hardly ever need to use this class. Instead, directly
 * subscribe a Kinesis stream using the integration class in the
 * `@aws-cdk/aws-logs-destinations` package; if necessary, a
 * `CrossAccountDestination` will be created automatically.
 *
 * @experimental
 * @resource AWS::Logs::Destination
 */
class CrossAccountDestination extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.destinationName ||
                // In the underlying model, the name is not optional, but we make it so anyway.
                cdk.Lazy.stringValue({ produce: () => this.generateUniqueName() }),
        });
        /**
         * (experimental) Policy object of this CrossAccountDestination object.
         *
         * @experimental
         */
        this.policyDocument = new iam.PolicyDocument();
        this.resource = new logs_generated_1.CfnDestination(this, 'Resource', {
            destinationName: this.physicalName,
            // Must be stringified policy
            destinationPolicy: this.lazyStringifiedPolicyDocument(),
            roleArn: props.role.roleArn,
            targetArn: props.targetArn,
        });
        this.destinationArn = this.getResourceArnAttribute(this.resource.attrArn, {
            service: 'logs',
            resource: 'destination',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.destinationName = this.getResourceNameAttribute(this.resource.ref);
    }
    /**
     * @experimental
     */
    addToPolicy(statement) {
        this.policyDocument.addStatements(statement);
    }
    /**
     * (experimental) Return the properties required to send subscription events to this destination.
     *
     * If necessary, the destination can use the properties of the SubscriptionFilter
     * object itself to configure its permissions to allow the subscription to write
     * to it.
     *
     * The destination may reconfigure its own permissions in response to this
     * function call.
     *
     * @experimental
     */
    bind(_scope, _sourceLogGroup) {
        return { arn: this.destinationArn };
    }
    /**
     * Generate a unique Destination name in case the user didn't supply one
     */
    generateUniqueName() {
        // Combination of stack name and LogicalID, which are guaranteed to be unique.
        return cdk.Stack.of(this).stackName + '-' + this.resource.logicalId;
    }
    /**
     * Return a stringified JSON version of the PolicyDocument
     */
    lazyStringifiedPolicyDocument() {
        return cdk.Lazy.stringValue({
            produce: () => this.policyDocument.isEmpty ? '' : cdk.Stack.of(this).toJsonString(this.policyDocument),
        });
    }
}
exports.CrossAccountDestination = CrossAccountDestination;
//# sourceMappingURL=data:application/json;base64,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