"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
/**
 * @experimental
 */
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
    }
    /**
     * (experimental) Adds a permission to the Lambda resource policy.
     *
     * @param id The id ƒor the permission construct.
     * @param permission The permission to grant to this Lambda function.
     * @see Permission for details.
     * @experimental
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        const scope = permission.scope || this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    /**
     * (experimental) Adds a statement to the IAM role assumed by the instance.
     *
     * @experimental
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * (experimental) Access the Connections object.
     *
     * Will fail if not a VPC-enabled Lambda Function
     *
     * @experimental
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    /**
     * (experimental) The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @experimental
     */
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * (experimental) Whether or not this Lambda function was bound to a VPC.
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     *
     * @experimental
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    /**
     * (experimental) Adds an event source that maps to this AWS Lambda function.
     *
     * @experimental
     */
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * (experimental) Grant the given identity permissions to invoke this Lambda.
     *
     * @experimental
     */
    grantInvoke(grantee) {
        const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = iam.Grant.addToPrincipalOrResource({
                grantee,
                actions: ['lambda:InvokeFunction'],
                resourceArns: [this.functionArn],
                // Fake resource-like object on which to call addToResourcePolicy(), which actually
                // calls addPermission()
                resource: {
                    addToResourcePolicy: (_statement) => {
                        // Couldn't add permissions to the principal, so add them locally.
                        this.addPermission(identifier, {
                            principal: grantee.grantPrincipal,
                            action: 'lambda:InvokeFunction',
                        });
                        return { statementAdded: true, policyDependable: this._functionNode().findChild(identifier) };
                    },
                    node: this.node,
                    stack: this.stack,
                    env: this.env,
                },
            });
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * (experimental) Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *      import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *      myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function.
     * @experimental
     */
    addEventSource(source) {
        source.bind(this);
    }
    /**
     * (experimental) Configures options for asynchronous invocation.
     *
     * @experimental
     */
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    /**
     * Given the function arn, check if the account id matches this account
     *
     * Function ARNs look like this:
     *
     *   arn:aws:lambda:region:account-id:function:function-name
     *
     * ..which means that in order to extract the `account-id` component from the ARN, we can
     * split the ARN using ":" and select the component in index 4.
     *
     * @returns true if account id of function matches this account
     *
     * @internal
     */
    _isStackAccount() {
        if (core_1.Token.isUnresolved(this.stack.account) || core_1.Token.isUnresolved(this.functionArn)) {
            return false;
        }
        return this.stack.parseArn(this.functionArn).account === this.stack.account;
    }
    /**
     * Translate IPrincipal to something we can pass to AWS::Lambda::Permissions
     *
     * Do some nasty things because `Permission` supports a subset of what the
     * full IAM principal language supports, and we may not be able to parse strings
     * outright because they may be tokens.
     *
     * Try to recognize some specific Principal classes first, then try a generic
     * fallback.
     */
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // Try some specific common classes first.
        // use duck-typing, not instance of
        // @deprecated: after v2, we can change these to 'instanceof'
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        // Try a best-effort approach to support simple principals that are not any of the predefined
        // classes, but are simple enough that they will fit into the Permission model. Main target
        // here: imported Roles, Users, Groups.
        //
        // The principal cannot have conditions and must have a single { AWS: [arn] } entry.
        const json = principal.policyFragment.principalJson;
        if (Object.keys(principal.policyFragment.conditions).length === 0 && json.AWS) {
            if (typeof json.AWS === 'string') {
                return json.AWS;
            }
            if (Array.isArray(json.AWS) && json.AWS.length === 1 && typeof json.AWS[0] === 'string') {
                return json.AWS[0];
            }
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
/**
 * @experimental
 */
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        /**
         * (experimental) The construct node where permissions are attached.
         *
         * @experimental
         */
        this.permissionsNode = this.node;
    }
    /**
     * (experimental) The `$LATEST` version of this function.
     *
     * Note that this is reference to a non-specific AWS Lambda version, which
     * means the function this version refers to can return different results in
     * different invocations.
     *
     * To obtain a reference to an explicit version which references the current
     * function configuration, use `lambdaFunction.currentVersion` instead.
     *
     * @experimental
     */
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    /**
     * (experimental) Configures options for asynchronous invocation.
     *
     * @experimental
     */
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = false;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
}
//# sourceMappingURL=data:application/json;base64,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