"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
const PARCEL_VERSION = '2.0.0-beta.1';
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static runsLocally(resolvePath) {
        if (LocalBundler._runsLocally[resolvePath] !== undefined) {
            return LocalBundler._runsLocally[resolvePath];
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel', { paths: [resolvePath] }), ['--version']);
            const version = parcel.stdout.toString().trim();
            LocalBundler._runsLocally[resolvePath] = new RegExp(`^${PARCEL_VERSION}`).test(version); // Cache result to avoid unnecessary spawns
            if (!LocalBundler._runsLocally[resolvePath]) {
                process.stderr.write(`Incorrect parcel version detected: ${version} <> ${PARCEL_VERSION}. Switching to Docker bundling.\n`);
            }
            return LocalBundler._runsLocally[resolvePath];
        }
        catch (err) {
            return false;
        }
    }
    static clearRunsLocallyCache() {
        LocalBundler._runsLocally = {};
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally(this.props.projectRoot)) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
            bundlingEnvironment: BundlingEnvironment.LOCAL,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
            cwd: path.dirname(path.join(this.props.projectRoot, this.props.relativeEntryPath)),
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
LocalBundler._runsLocally = {};
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b, _c;
        const image = props.buildImage
            ? (_a = props.bundlingDockerImage) !== null && _a !== void 0 ? _a : cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_b = props.buildArgs) !== null && _b !== void 0 ? _b : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_c = props.parcelVersion) !== null && _c !== void 0 ? _c : PARCEL_VERSION,
                },
            }) : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
            bundlingEnvironment: BundlingEnvironment.DOCKER,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)).replace(/\\/g, '/'),
        };
    }
}
exports.DockerBundler = DockerBundler;
var BundlingEnvironment;
(function (BundlingEnvironment) {
    BundlingEnvironment["DOCKER"] = "docker";
    BundlingEnvironment["LOCAL"] = "local";
})(BundlingEnvironment || (BundlingEnvironment = {}));
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.(jsx|tsx?)$/, '.js');
    const parcelResolvePath = options.bundlingEnvironment === BundlingEnvironment.DOCKER
        ? '/' // Force using parcel installed at / in the image
        : entryPath; // Look up starting from entry path
    const parcelCommand = chain([
        [
            `$(node -p "require.resolve(\'parcel\', { paths: ['${parcelResolvePath}'] })")`,
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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