"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Database = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const glue_generated_1 = require("./glue.generated");
/**
 * (experimental) A Glue database.
 *
 * @experimental
 */
class Database extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.databaseName,
        });
        let databaseInput = {
            name: props.databaseName,
        };
        if (props.locationUri !== undefined) {
            validateLocationUri(props.locationUri);
            this.locationUri = props.locationUri;
            databaseInput = {
                locationUri: this.locationUri,
                ...databaseInput,
            };
        }
        this.catalogId = core_1.Stack.of(this).account;
        const resource = new glue_generated_1.CfnDatabase(this, 'Resource', {
            catalogId: this.catalogId,
            databaseInput,
        });
        // see https://docs.aws.amazon.com/glue/latest/dg/glue-specifying-resource-arns.html#data-catalog-resource-arns
        this.databaseName = this.getResourceNameAttribute(resource.ref);
        this.databaseArn = this.stack.formatArn({
            service: 'glue',
            resource: 'database',
            resourceName: this.databaseName,
        });
        // catalogId is implicitly the accountId, which is why we don't pass the catalogId here
        this.catalogArn = core_1.Stack.of(this).formatArn({
            service: 'glue',
            resource: 'catalog',
        });
    }
    /**
     * @experimental
     */
    static fromDatabaseArn(scope, id, databaseArn) {
        const stack = core_1.Stack.of(scope);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.databaseArn = databaseArn;
                this.databaseName = stack.parseArn(databaseArn).resourceName;
                this.catalogArn = stack.formatArn({ service: 'glue', resource: 'catalog' });
                this.catalogId = stack.account;
            }
        }
        return new Import(scope, id);
    }
}
exports.Database = Database;
function validateLocationUri(locationUri) {
    if (locationUri.length < 1 || locationUri.length > 1024) {
        throw new Error(`locationUri length must be (inclusively) between 1 and 1024, but was ${locationUri.length}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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