"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Domain = exports.TLSSecurityPolicy = exports.ElasticsearchVersion = void 0;
const url_1 = require("url");
const aws_cloudwatch_1 = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticsearch_access_policy_1 = require("./elasticsearch-access-policy");
const elasticsearch_generated_1 = require("./elasticsearch.generated");
const log_group_resource_policy_1 = require("./log-group-resource-policy");
const perms = require("./perms");
/**
 * (experimental) Elasticsearch version.
 *
 * @experimental
 */
class ElasticsearchVersion {
    /**
     *
     * @param version Elasticsearch version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * (experimental) Custom Elasticsearch version.
     *
     * @param version custom version number.
     * @experimental
     */
    static of(version) { return new ElasticsearchVersion(version); }
}
exports.ElasticsearchVersion = ElasticsearchVersion;
/**
 * (experimental) AWS Elasticsearch 1.5.
 *
 * @experimental
 */
ElasticsearchVersion.V1_5 = ElasticsearchVersion.of('1.5');
/**
 * (experimental) AWS Elasticsearch 2.3.
 *
 * @experimental
 */
ElasticsearchVersion.V2_3 = ElasticsearchVersion.of('2.3');
/**
 * (experimental) AWS Elasticsearch 5.1.
 *
 * @experimental
 */
ElasticsearchVersion.V5_1 = ElasticsearchVersion.of('5.1');
/**
 * (experimental) AWS Elasticsearch 5.3.
 *
 * @experimental
 */
ElasticsearchVersion.V5_3 = ElasticsearchVersion.of('5.3');
/**
 * (experimental) AWS Elasticsearch 5.5.
 *
 * @experimental
 */
ElasticsearchVersion.V5_5 = ElasticsearchVersion.of('5.5');
/**
 * (experimental) AWS Elasticsearch 5.6.
 *
 * @experimental
 */
ElasticsearchVersion.V5_6 = ElasticsearchVersion.of('5.6');
/**
 * (experimental) AWS Elasticsearch 6.0.
 *
 * @experimental
 */
ElasticsearchVersion.V6_0 = ElasticsearchVersion.of('6.0');
/**
 * (experimental) AWS Elasticsearch 6.2.
 *
 * @experimental
 */
ElasticsearchVersion.V6_2 = ElasticsearchVersion.of('6.2');
/**
 * (experimental) AWS Elasticsearch 6.3.
 *
 * @experimental
 */
ElasticsearchVersion.V6_3 = ElasticsearchVersion.of('6.3');
/**
 * (experimental) AWS Elasticsearch 6.4.
 *
 * @experimental
 */
ElasticsearchVersion.V6_4 = ElasticsearchVersion.of('6.4');
/**
 * (experimental) AWS Elasticsearch 6.5.
 *
 * @experimental
 */
ElasticsearchVersion.V6_5 = ElasticsearchVersion.of('6.5');
/**
 * (experimental) AWS Elasticsearch 6.7.
 *
 * @experimental
 */
ElasticsearchVersion.V6_7 = ElasticsearchVersion.of('6.7');
/**
 * (experimental) AWS Elasticsearch 6.8.
 *
 * @experimental
 */
ElasticsearchVersion.V6_8 = ElasticsearchVersion.of('6.8');
/**
 * (experimental) AWS Elasticsearch 7.1.
 *
 * @experimental
 */
ElasticsearchVersion.V7_1 = ElasticsearchVersion.of('7.1');
/**
 * (experimental) AWS Elasticsearch 7.4.
 *
 * @experimental
 */
ElasticsearchVersion.V7_4 = ElasticsearchVersion.of('7.4');
/**
 * (experimental) AWS Elasticsearch 7.7.
 *
 * @experimental
 */
ElasticsearchVersion.V7_7 = ElasticsearchVersion.of('7.7');
/**
 * (experimental) The minimum TLS version required for traffic to the domain.
 *
 * @experimental
 */
var TLSSecurityPolicy;
(function (TLSSecurityPolicy) {
    TLSSecurityPolicy["TLS_1_0"] = "Policy-Min-TLS-1-0-2019-07";
    TLSSecurityPolicy["TLS_1_2"] = "Policy-Min-TLS-1-2-2019-07";
})(TLSSecurityPolicy = exports.TLSSecurityPolicy || (exports.TLSSecurityPolicy = {}));
/**
 * A new or imported domain.
 */
class DomainBase extends cdk.Resource {
    /**
     * (experimental) Grant read permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantRead(identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantWrite(identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for this domain and its contents to an IAM principal (Role/Group/User).
     *
     * @param identity The principal.
     * @experimental
     */
    grantReadWrite(identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, this.domainArn, `${this.domainArn}/*`);
    }
    /**
     * (experimental) Grant read permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexRead(index, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexWrite(index, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read/write permissions for an index in this domain to an IAM principal (Role/Group/User).
     *
     * @param index The index to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantIndexReadWrite(index, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${index}`, `${this.domainArn}/${index}/*`);
    }
    /**
     * (experimental) Grant read permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathRead(path, identity) {
        return this.grant(identity, perms.ES_READ_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathWrite(path, identity) {
        return this.grant(identity, perms.ES_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Grant read/write permissions for a specific path in this domain to an IAM principal (Role/Group/User).
     *
     * @param path The path to grant permissions for.
     * @param identity The principal.
     * @experimental
     */
    grantPathReadWrite(path, identity) {
        return this.grant(identity, perms.ES_READ_WRITE_ACTIONS, `${this.domainArn}/${path}`);
    }
    /**
     * (experimental) Return the given named metric for this Domain.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new aws_cloudwatch_1.Metric({
            namespace: 'AWS/ES',
            metricName,
            dimensions: {
                DomainName: this.domainName,
                ClientId: this.stack.account,
            },
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is red.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusRed(props) {
        return this.metric('ClusterStatus.red', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the time the cluster status is yellow.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricClusterStatusYellow(props) {
        return this.metric('ClusterStatus.yellow', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the storage space of nodes in the cluster.
     *
     * @default minimum over 5 minutes
     * @experimental
     */
    metricFreeStorageSpace(props) {
        return this.metric('FreeStorageSpace', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the cluster blocking index writes.
     *
     * @default maximum over 1 minute
     * @experimental
     */
    metricClusterIndexWriteBlocked(props) {
        return this.metric('ClusterIndexWriteBlocked', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            period: cdk.Duration.minutes(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for the number of nodes.
     *
     * @default minimum over 1 hour
     * @experimental
     */
    metricNodes(props) {
        return this.metric('Nodes', {
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            period: cdk.Duration.hours(1),
            ...props,
        });
    }
    /**
     * (experimental) Metric for automated snapshot failures.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricAutomatedSnapshotFailure(props) {
        return this.metric('AutomatedSnapshotFailure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricCPUUtilization(props) {
        return this.metric('CPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricJVMMemoryPressure(props) {
        return this.metric('JVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master CPU utilization.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterCPUUtilization(props) {
        return this.metric('MasterCPUUtilization', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for master JVM memory pressure.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricMasterJVMMemoryPressure(props) {
        return this.metric('MasterJVMMemoryPressure', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key errors.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyError(props) {
        return this.metric('KMSKeyError', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for KMS key being inaccessible.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricKMSKeyInaccessible(props) {
        return this.metric('KMSKeyInaccessible', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for number of searchable documents.
     *
     * @default maximum over 5 minutes
     * @experimental
     */
    metricSearchableDocuments(props) {
        return this.metric('SearchableDocuments', {
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...props,
        });
    }
    /**
     * (experimental) Metric for search latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricSearchLatency(props) {
        return this.metric('SearchLatencyP99', { statistic: 'p99', ...props });
    }
    /**
     * (experimental) Metric for indexing latency.
     *
     * @default p99 over 5 minutes
     * @experimental
     */
    metricIndexingLatency(props) {
        return this.metric('IndexingLatencyP99', { statistic: 'p99', ...props });
    }
    grant(grantee, domainActions, resourceArn, ...otherResourceArns) {
        const resourceArns = [resourceArn, ...otherResourceArns];
        const grant = iam.Grant.addToPrincipal({
            grantee,
            actions: domainActions,
            resourceArns,
            scope: this,
        });
        return grant;
    }
}
/**
 * (experimental) Provides an Elasticsearch domain.
 *
 * @experimental
 */
class Domain extends DomainBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t, _u, _v, _w, _x, _y, _z, _0, _1, _2, _3, _4, _5, _6, _7, _8, _9, _10, _11, _12, _13, _14, _15, _16, _17, _18, _19, _20, _21, _22, _23, _24, _25, _26, _27, _28, _29;
        super(scope, id, {
            physicalName: props.domainName,
        });
        const defaultInstanceType = 'r5.large.elasticsearch';
        const dedicatedMasterType = (_c = (_b = (_a = props.capacity) === null || _a === void 0 ? void 0 : _a.masterNodeInstanceType) === null || _b === void 0 ? void 0 : _b.toLowerCase()) !== null && _c !== void 0 ? _c : defaultInstanceType;
        const dedicatedMasterCount = (_e = (_d = props.capacity) === null || _d === void 0 ? void 0 : _d.masterNodes) !== null && _e !== void 0 ? _e : 0;
        const dedicatedMasterEnabled = dedicatedMasterCount > 0;
        const instanceType = (_h = (_g = (_f = props.capacity) === null || _f === void 0 ? void 0 : _f.dataNodeInstanceType) === null || _g === void 0 ? void 0 : _g.toLowerCase()) !== null && _h !== void 0 ? _h : defaultInstanceType;
        const instanceCount = (_k = (_j = props.capacity) === null || _j === void 0 ? void 0 : _j.dataNodes) !== null && _k !== void 0 ? _k : 1;
        const availabilityZoneCount = (_m = (_l = props.zoneAwareness) === null || _l === void 0 ? void 0 : _l.availabilityZoneCount) !== null && _m !== void 0 ? _m : 2;
        if (![2, 3].includes(availabilityZoneCount)) {
            throw new Error('Invalid zone awareness configuration; availabilityZoneCount must be 2 or 3');
        }
        const zoneAwarenessEnabled = (_p = (_o = props.zoneAwareness) === null || _o === void 0 ? void 0 : _o.enabled) !== null && _p !== void 0 ? _p : ((_q = props.zoneAwareness) === null || _q === void 0 ? void 0 : _q.availabilityZoneCount) != null;
        // If VPC options are supplied ensure that the number of subnets matches the number AZ
        if (props.vpcOptions != null && zoneAwarenessEnabled &&
            new Set((_r = props.vpcOptions) === null || _r === void 0 ? void 0 : _r.subnets.map((subnet) => subnet.availabilityZone)).size < availabilityZoneCount) {
            throw new Error('When providing vpc options you need to provide a subnet for each AZ you are using');
        }
        ;
        if ([dedicatedMasterType, instanceType].some(t => !t.endsWith('.elasticsearch'))) {
            throw new Error('Master and data node instance types must end with ".elasticsearch".');
        }
        const elasticsearchVersion = props.version.version;
        const elasticsearchVersionNum = parseVersion(props.version);
        if (elasticsearchVersionNum <= 7.7 &&
            ![
                1.5, 2.3, 5.1, 5.3, 5.5, 5.6, 6.0,
                6.2, 6.3, 6.4, 6.5, 6.7, 6.8, 7.1, 7.4,
                7.7,
            ].includes(elasticsearchVersionNum)) {
            throw new Error(`Unknown Elasticsearch version: ${elasticsearchVersion}`);
        }
        const unsignedBasicAuthEnabled = (_s = props.useUnsignedBasicAuth) !== null && _s !== void 0 ? _s : false;
        if (unsignedBasicAuthEnabled) {
            if (props.enforceHttps == false) {
                throw new Error('You cannot disable HTTPS and use unsigned basic auth');
            }
            if (props.nodeToNodeEncryption == false) {
                throw new Error('You cannot disable node to node encryption and use unsigned basic auth');
            }
            if (((_t = props.encryptionAtRest) === null || _t === void 0 ? void 0 : _t.enabled) == false) {
                throw new Error('You cannot disable encryption at rest and use unsigned basic auth');
            }
        }
        const unsignedAccessPolicy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['es:ESHttp*'],
            principals: [new iam.Anyone()],
            resources: [cdk.Lazy.stringValue({ produce: () => `${this.domainArn}/*` })],
        });
        const masterUserArn = (_u = props.fineGrainedAccessControl) === null || _u === void 0 ? void 0 : _u.masterUserArn;
        const masterUserNameProps = (_v = props.fineGrainedAccessControl) === null || _v === void 0 ? void 0 : _v.masterUserName;
        // If basic auth is enabled set the user name to admin if no other user info is supplied.
        const masterUserName = unsignedBasicAuthEnabled
            ? (masterUserArn == null ? (masterUserNameProps !== null && masterUserNameProps !== void 0 ? masterUserNameProps : 'admin') : undefined)
            : masterUserNameProps;
        if (masterUserArn != null && masterUserName != null) {
            throw new Error('Invalid fine grained access control settings. Only provide one of master user ARN or master user name. Not both.');
        }
        const advancedSecurityEnabled = (masterUserArn !== null && masterUserArn !== void 0 ? masterUserArn : masterUserName) != null;
        const internalUserDatabaseEnabled = masterUserName != null;
        const masterUserPasswordProp = (_w = props.fineGrainedAccessControl) === null || _w === void 0 ? void 0 : _w.masterUserPassword;
        const createMasterUserPassword = () => {
            return new secretsmanager.Secret(this, 'MasterUser', {
                generateSecretString: {
                    secretStringTemplate: JSON.stringify({
                        username: masterUserName,
                    }),
                    generateStringKey: 'password',
                    excludeCharacters: "{}'\\*[]()`",
                },
            })
                .secretValueFromJson('password');
        };
        this.masterUserPassword = internalUserDatabaseEnabled ?
            (masterUserPasswordProp !== null && masterUserPasswordProp !== void 0 ? masterUserPasswordProp : createMasterUserPassword())
            : undefined;
        const encryptionAtRestEnabled = (_y = (_x = props.encryptionAtRest) === null || _x === void 0 ? void 0 : _x.enabled) !== null && _y !== void 0 ? _y : (((_z = props.encryptionAtRest) === null || _z === void 0 ? void 0 : _z.kmsKey) != null || unsignedBasicAuthEnabled);
        const nodeToNodeEncryptionEnabled = (_0 = props.nodeToNodeEncryption) !== null && _0 !== void 0 ? _0 : unsignedBasicAuthEnabled;
        const volumeSize = (_2 = (_1 = props.ebs) === null || _1 === void 0 ? void 0 : _1.volumeSize) !== null && _2 !== void 0 ? _2 : 10;
        const volumeType = (_4 = (_3 = props.ebs) === null || _3 === void 0 ? void 0 : _3.volumeType) !== null && _4 !== void 0 ? _4 : ec2.EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
        const ebsEnabled = (_6 = (_5 = props.ebs) === null || _5 === void 0 ? void 0 : _5.enabled) !== null && _6 !== void 0 ? _6 : true;
        const enforceHttps = (_7 = props.enforceHttps) !== null && _7 !== void 0 ? _7 : unsignedBasicAuthEnabled;
        function isInstanceType(t) {
            return dedicatedMasterType.startsWith(t) || instanceType.startsWith(t);
        }
        ;
        function isSomeInstanceType(...instanceTypes) {
            return instanceTypes.some(isInstanceType);
        }
        ;
        function isEveryInstanceType(...instanceTypes) {
            return instanceTypes.some(t => dedicatedMasterType.startsWith(t))
                && instanceTypes.some(t => instanceType.startsWith(t));
        }
        ;
        // Validate feature support for the given Elasticsearch version, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-features-by-version.html
        if (elasticsearchVersionNum < 5.1) {
            if (((_8 = props.logging) === null || _8 === void 0 ? void 0 : _8.slowIndexLogEnabled) || ((_9 = props.logging) === null || _9 === void 0 ? void 0 : _9.appLogEnabled)
                || ((_10 = props.logging) === null || _10 === void 0 ? void 0 : _10.slowSearchLogEnabled)) {
                throw new Error('Error and slow logs publishing requires Elasticsearch version 5.1 or later.');
            }
            if ((_11 = props.encryptionAtRest) === null || _11 === void 0 ? void 0 : _11.enabled) {
                throw new Error('Encryption of data at rest requires Elasticsearch version 5.1 or later.');
            }
            if (props.cognitoKibanaAuth != null) {
                throw new Error('Cognito authentication for Kibana requires Elasticsearch version 5.1 or later.');
            }
            if (isSomeInstanceType('c5', 'i3', 'm5', 'r5')) {
                throw new Error('C5, I3, M5, and R5 instance types require Elasticsearch version 5.1 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.0) {
            if (props.nodeToNodeEncryption) {
                throw new Error('Node-to-node encryption requires Elasticsearch version 6.0 or later.');
            }
        }
        if (elasticsearchVersionNum < 6.7) {
            if (unsignedBasicAuthEnabled) {
                throw new Error('Using unsigned basic auth requires Elasticsearch version 6.7 or later.');
            }
            if (advancedSecurityEnabled) {
                throw new Error('Fine-grained access control requires Elasticsearch version 6.7 or later.');
            }
        }
        // Validate against instance type restrictions, per
        // https://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/aes-supported-instance-types.html
        if (isInstanceType('i3') && ebsEnabled) {
            throw new Error('I3 instance types do not support EBS storage volumes.');
        }
        if (isSomeInstanceType('m3', 'r3', 't2') && encryptionAtRestEnabled) {
            throw new Error('M3, R3, and T2 instance types do not support encryption of data at rest.');
        }
        if (isInstanceType('t2.micro') && elasticsearchVersionNum > 2.3) {
            throw new Error('The t2.micro.elasticsearch instance type supports only Elasticsearch 1.5 and 2.3.');
        }
        // Only R3 and I3 support instance storage, per
        // https://aws.amazon.com/elasticsearch-service/pricing/
        if (!ebsEnabled && !isEveryInstanceType('r3', 'i3')) {
            throw new Error('EBS volumes are required when using instance types other than r3 or i3.');
        }
        // Fine-grained access control requires node-to-node encryption, encryption at rest,
        // and enforced HTTPS.
        if (advancedSecurityEnabled) {
            if (!nodeToNodeEncryptionEnabled) {
                throw new Error('Node-to-node encryption is required when fine-grained access control is enabled.');
            }
            if (!encryptionAtRestEnabled) {
                throw new Error('Encryption-at-rest is required when fine-grained access control is enabled.');
            }
            if (!enforceHttps) {
                throw new Error('Enforce HTTPS is required when fine-grained access control is enabled.');
            }
        }
        let cfnVpcOptions;
        if (props.vpcOptions) {
            cfnVpcOptions = {
                securityGroupIds: props.vpcOptions.securityGroups.map((sg) => sg.securityGroupId),
                subnetIds: props.vpcOptions.subnets.map((subnet) => subnet.subnetId),
            };
        }
        // Setup logging
        const logGroups = [];
        if ((_12 = props.logging) === null || _12 === void 0 ? void 0 : _12.slowSearchLogEnabled) {
            this.slowSearchLogGroup = (_13 = props.logging.slowSearchLogGroup) !== null && _13 !== void 0 ? _13 : new logs.LogGroup(scope, 'SlowSearchLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowSearchLogGroup);
        }
        ;
        if ((_14 = props.logging) === null || _14 === void 0 ? void 0 : _14.slowIndexLogEnabled) {
            this.slowIndexLogGroup = (_15 = props.logging.slowIndexLogGroup) !== null && _15 !== void 0 ? _15 : new logs.LogGroup(scope, 'SlowIndexLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.slowIndexLogGroup);
        }
        ;
        if ((_16 = props.logging) === null || _16 === void 0 ? void 0 : _16.appLogEnabled) {
            this.appLogGroup = (_17 = props.logging.appLogGroup) !== null && _17 !== void 0 ? _17 : new logs.LogGroup(scope, 'AppLogs', {
                retention: logs.RetentionDays.ONE_MONTH,
            });
            logGroups.push(this.appLogGroup);
        }
        ;
        let logGroupResourcePolicy = null;
        if (logGroups.length > 0) {
            const logPolicyStatement = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: logGroups.map((lg) => lg.logGroupArn),
                principals: [new iam.ServicePrincipal('es.amazonaws.com')],
            });
            // Use a custom resource to set the log group resource policy since it is not supported by CDK and cfn.
            // https://github.com/aws/aws-cdk/issues/5343
            logGroupResourcePolicy = new log_group_resource_policy_1.LogGroupResourcePolicy(this, 'ESLogGroupPolicy', {
                policyName: 'ESLogPolicy',
                policyStatements: [logPolicyStatement],
            });
        }
        // Create the domain
        this.domain = new elasticsearch_generated_1.CfnDomain(this, 'Resource', {
            domainName: this.physicalName,
            elasticsearchVersion,
            elasticsearchClusterConfig: {
                dedicatedMasterEnabled,
                dedicatedMasterCount: dedicatedMasterEnabled
                    ? dedicatedMasterCount
                    : undefined,
                dedicatedMasterType: dedicatedMasterEnabled
                    ? dedicatedMasterType
                    : undefined,
                instanceCount,
                instanceType,
                zoneAwarenessEnabled,
                zoneAwarenessConfig: zoneAwarenessEnabled
                    ? { availabilityZoneCount }
                    : undefined,
            },
            ebsOptions: {
                ebsEnabled,
                volumeSize: ebsEnabled ? volumeSize : undefined,
                volumeType: ebsEnabled ? volumeType : undefined,
                iops: ebsEnabled ? (_18 = props.ebs) === null || _18 === void 0 ? void 0 : _18.iops : undefined,
            },
            encryptionAtRestOptions: {
                enabled: encryptionAtRestEnabled,
                kmsKeyId: encryptionAtRestEnabled
                    ? (_20 = (_19 = props.encryptionAtRest) === null || _19 === void 0 ? void 0 : _19.kmsKey) === null || _20 === void 0 ? void 0 : _20.keyId : undefined,
            },
            nodeToNodeEncryptionOptions: { enabled: nodeToNodeEncryptionEnabled },
            logPublishingOptions: {
                ES_APPLICATION_LOGS: {
                    enabled: this.appLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_21 = this.appLogGroup) === null || _21 === void 0 ? void 0 : _21.logGroupArn,
                },
                SEARCH_SLOW_LOGS: {
                    enabled: this.slowSearchLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_22 = this.slowSearchLogGroup) === null || _22 === void 0 ? void 0 : _22.logGroupArn,
                },
                INDEX_SLOW_LOGS: {
                    enabled: this.slowIndexLogGroup != null,
                    cloudWatchLogsLogGroupArn: (_23 = this.slowIndexLogGroup) === null || _23 === void 0 ? void 0 : _23.logGroupArn,
                },
            },
            cognitoOptions: {
                enabled: props.cognitoKibanaAuth != null,
                identityPoolId: (_24 = props.cognitoKibanaAuth) === null || _24 === void 0 ? void 0 : _24.identityPoolId,
                roleArn: (_25 = props.cognitoKibanaAuth) === null || _25 === void 0 ? void 0 : _25.role.roleArn,
                userPoolId: (_26 = props.cognitoKibanaAuth) === null || _26 === void 0 ? void 0 : _26.userPoolId,
            },
            vpcOptions: cfnVpcOptions,
            snapshotOptions: props.automatedSnapshotStartHour
                ? { automatedSnapshotStartHour: props.automatedSnapshotStartHour }
                : undefined,
            domainEndpointOptions: {
                enforceHttps,
                tlsSecurityPolicy: (_27 = props.tlsSecurityPolicy) !== null && _27 !== void 0 ? _27 : TLSSecurityPolicy.TLS_1_0,
            },
            advancedSecurityOptions: advancedSecurityEnabled
                ? {
                    enabled: true,
                    internalUserDatabaseEnabled,
                    masterUserOptions: {
                        masterUserArn: masterUserArn,
                        masterUserName: masterUserName,
                        masterUserPassword: (_28 = this.masterUserPassword) === null || _28 === void 0 ? void 0 : _28.toString(),
                    },
                }
                : undefined,
        });
        if (logGroupResourcePolicy) {
            this.domain.node.addDependency(logGroupResourcePolicy);
        }
        if (props.domainName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', props.domainName);
        }
        this.domainName = this.getResourceNameAttribute(this.domain.ref);
        this.domainEndpoint = this.domain.getAtt('DomainEndpoint').toString();
        this.domainArn = this.getResourceArnAttribute(this.domain.attrArn, {
            service: 'es',
            resource: 'domain',
            resourceName: this.physicalName,
        });
        const accessPolicyStatements = unsignedBasicAuthEnabled
            ? ((_29 = props.accessPolicies) !== null && _29 !== void 0 ? _29 : []).concat(unsignedAccessPolicy)
            : props.accessPolicies;
        if (accessPolicyStatements != null) {
            const accessPolicy = new elasticsearch_access_policy_1.ElasticsearchAccessPolicy(this, 'ESAccessPolicy', {
                domainName: this.domainName,
                domainArn: this.domainArn,
                accessPolicies: accessPolicyStatements,
            });
            accessPolicy.node.addDependency(this.domain);
        }
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain via domain endpoint.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param domainEndpoint The domain's endpoint.
     * @experimental
     */
    static fromDomainEndpoint(scope, id, domainEndpoint) {
        const stack = cdk.Stack.of(scope);
        const domainName = extractNameFromEndpoint(domainEndpoint);
        const domainArn = stack.formatArn({
            service: 'es',
            resource: 'domain',
            resourceName: domainName,
        });
        return Domain.fromDomainAttributes(scope, id, {
            domainArn,
            domainEndpoint,
        });
    }
    /**
     * (experimental) Creates a Domain construct that represents an external domain.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `DomainAttributes` object.
     * @experimental
     */
    static fromDomainAttributes(scope, id, attrs) {
        const { domainArn, domainEndpoint } = attrs;
        const domainName = extractNameFromEndpoint(domainEndpoint);
        return new class extends DomainBase {
            constructor() {
                super(scope, id);
                this.domainArn = domainArn;
                this.domainName = domainName;
                this.domainEndpoint = domainEndpoint;
            }
        };
    }
}
exports.Domain = Domain;
/**
 * Given an Elasticsearch domain endpoint, returns a CloudFormation expression that
 * extracts the domain name.
 *
 * Domain endpoints look like this:
 *
 *   https://example-domain-jcjotrt6f7otem4sqcwbch3c4u.us-east-1.es.amazonaws.com
 *   https://<domain-name>-<suffix>.<region>.es.amazonaws.com
 *
 * ..which means that in order to extract the domain name from the endpoint, we can
 * split the endpoint using "-<suffix>" and select the component in index 0.
 *
 * @param domainEndpoint The Elasticsearch domain endpoint
 */
function extractNameFromEndpoint(domainEndpoint) {
    const { hostname } = new url_1.URL(domainEndpoint);
    const domain = hostname.split('.')[0];
    const suffix = '-' + domain.split('-').slice(-1)[0];
    return domain.split(suffix)[0];
}
/**
 * Converts an Elasticsearch version into a into a decimal number with major and minor version i.e x.y.
 *
 * @param version The Elasticsearch version object
 */
function parseVersion(version) {
    const versionStr = version.version;
    const firstDot = versionStr.indexOf('.');
    if (firstDot < 1) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
    const secondDot = versionStr.indexOf('.', firstDot + 1);
    try {
        if (secondDot == -1) {
            return parseFloat(versionStr);
        }
        else {
            return parseFloat(versionStr.substring(0, secondDot));
        }
    }
    catch (error) {
        throw new Error(`Invalid Elasticsearch version: ${versionStr}. Version string needs to start with major and minor version (x.y).`);
    }
}
//# sourceMappingURL=data:application/json;base64,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