"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateRedirectResponse = exports.validateFixedResponse = exports.ApplicationListenerRule = exports.ContentType = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * (experimental) The content type for a fixed response.
 *
 * @experimental
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * (experimental) Define a new listener rule.
 *
 * @experimental
 */
class ApplicationListenerRule extends cdk.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * (deprecated) Add a non-standard condition to this rule.
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * (experimental) Add a non-standard condition to this rule.
     *
     * @experimental
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * (experimental) Configure the action to perform for this rule.
     *
     * @experimental
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            cdk.Annotations.of(this).addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * (deprecated) Add a TargetGroup to load balance to.
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * (deprecated) Add a fixed response.
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * (deprecated) Add a redirect response.
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * (experimental) Validate the rule.
     *
     * @experimental
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbGlzdGVuZXItcnVsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwcGxpY2F0aW9uLWxpc3RlbmVyLXJ1bGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQXFDLENBQUMsZ0RBQWdEO0FBRXRGLDBGQUFzRTtBQUd0RSwrRUFBK0Q7Ozs7OztBQXFHL0QsSUFBWSxXQU1YO0FBTkQsV0FBWSxXQUFXO0lBQ25CLHdDQUF5QixDQUFBO0lBQ3pCLG9DQUFxQixDQUFBO0lBQ3JCLHNDQUF1QixDQUFBO0lBQ3ZCLGdFQUFpRCxDQUFBO0lBQ2pELG9EQUFxQyxDQUFBO0FBQ3pDLENBQUMsRUFOVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQU10Qjs7Ozs7O0FBb0VELE1BQWEsdUJBQXdCLFNBQVEsR0FBRyxDQUFDLFNBQVM7Ozs7SUFXdEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFtQztRQUN6RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBTkoscUJBQWdCLEdBRTdCLEVBQUUsQ0FBQztRQUtILElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7UUFDekMsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ2hFLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN2RSxNQUFNLElBQUksS0FBSyxDQUFDLHNJQUFzSSxDQUFDLENBQUM7U0FDM0o7UUFDRCxNQUFNLGVBQWUsR0FBOEMsQ0FBQyxRQUFRLEVBQUUsY0FBYyxFQUFFLGVBQWUsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1FBQ25JLE1BQU0sZUFBZSxHQUFHLGVBQWUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssU0FBUyxDQUFDLENBQUM7UUFDdEYsSUFBSSxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLElBQUksZUFBZSx3Q0FBd0MsQ0FBQyxDQUFDO1NBQ2hGO1FBQ0QsSUFBSSxLQUFLLENBQUMsUUFBUSxJQUFJLENBQUMsRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FDMUU7UUFDRCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxrREFBZSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDbkQsV0FBVyxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsV0FBVztZQUN2QyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLENBQUM7WUFDekUsT0FBTyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDO1NBQ2hHLENBQUMsQ0FBQztRQUNILElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNsQixJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsRUFBRSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1NBQ3hEO1FBQ0QsSUFBSSxlQUFlLEVBQUU7WUFDakIsSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7Z0JBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUVBQXVFLENBQUMsQ0FBQzthQUM1RjtZQUNELE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO1lBQ2pGLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ2xEO1FBQ0QsSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ2QsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDdEM7UUFDRCxDQUFDLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDbkUsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3JCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDOUM7YUFDSSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUM3QixJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7U0FDcEQ7UUFDRCxJQUFJLENBQUMsZUFBZSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7SUFDeEMsQ0FBQzs7Ozs7Ozs7SUFRTSxZQUFZLENBQUMsS0FBYSxFQUFFLE1BQTRCO1FBQzNELElBQUksTUFBTSxLQUFLLFNBQVMsRUFBRTtZQUN0QixPQUFPLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNwQyxPQUFPO1NBQ1Y7UUFDRCxJQUFJLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLEdBQUcsTUFBTSxDQUFDO0lBQzFDLENBQUM7Ozs7OztJQUlNLFlBQVksQ0FBQyxTQUE0QjtRQUM1QyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUNwQyxDQUFDOzs7Ozs7SUFJTSxlQUFlLENBQUMsTUFBc0I7UUFDekMsdUNBQXVDO1FBQ3ZDLEVBQUU7UUFDRixrRkFBa0Y7UUFDbEYseUZBQXlGO1FBQ3pGLDhGQUE4RjtRQUM5RixrREFBa0Q7UUFDbEQsRUFBRTtRQUNGLDBDQUEwQztRQUMxQyw0RUFBNEU7UUFDNUUsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2IsR0FBRyxDQUFDLFdBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsVUFBVSxDQUFDLHdHQUF3RyxDQUFDLENBQUM7U0FDako7UUFDRCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxJQUFJLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxDQUFDO1FBQ3ZDLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO0lBQ3pCLENBQUM7Ozs7OztJQU1NLGNBQWMsQ0FBQyxXQUFvQztRQUN0RCxJQUFJLENBQUMsZUFBZSxDQUFDLDRDQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7Ozs7OztJQU1NLGdCQUFnQixDQUFDLGFBQTRCO1FBQ2hELHFCQUFxQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxlQUFlLENBQUMsNENBQWMsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQzVGLFdBQVcsRUFBRSxhQUFhLENBQUMsV0FBVztZQUN0QyxXQUFXLEVBQUUsYUFBYSxDQUFDLFdBQVc7U0FDekMsQ0FBQyxDQUFDLENBQUM7SUFDUixDQUFDOzs7Ozs7SUFNTSxtQkFBbUIsQ0FBQyxnQkFBa0M7UUFDekQsd0JBQXdCLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsZUFBZSxDQUFDLDRDQUFjLENBQUMsUUFBUSxDQUFDO1lBQ3pDLElBQUksRUFBRSxnQkFBZ0IsQ0FBQyxJQUFJO1lBQzNCLElBQUksRUFBRSxnQkFBZ0IsQ0FBQyxJQUFJO1lBQzNCLFNBQVMsRUFBRSxnQkFBZ0IsQ0FBQyxVQUFVLEtBQUssVUFBVTtZQUNyRCxJQUFJLEVBQUUsZ0JBQWdCLENBQUMsSUFBSTtZQUMzQixRQUFRLEVBQUUsZ0JBQWdCLENBQUMsUUFBUTtZQUNuQyxLQUFLLEVBQUUsZ0JBQWdCLENBQUMsS0FBSztTQUNoQyxDQUFDLENBQUMsQ0FBQztJQUNSLENBQUM7Ozs7OztJQUlTLFFBQVE7UUFDZCxJQUFJLElBQUksQ0FBQyxNQUFNLEtBQUssU0FBUyxFQUFFO1lBQzNCLE9BQU8sQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1NBQ3REO1FBQ0QsTUFBTSxxQkFBcUIsR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQ2pFLElBQUkscUJBQXFCLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDcEUsT0FBTyxDQUFDLDRDQUE0QyxDQUFDLENBQUM7U0FDekQ7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRDs7T0FFRztJQUNLLGdCQUFnQjtRQUNwQixNQUFNLGdCQUFnQixHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLEVBQUUsRUFBRTtZQUNuRixPQUFPLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxDQUFDO1FBQzdCLENBQUMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxVQUFVLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQyxTQUFTLENBQUMsa0JBQWtCLEVBQUUsQ0FBQyxDQUFDO1FBQ3BGLE9BQU87WUFDSCxHQUFHLGdCQUFnQjtZQUNuQixHQUFHLFVBQVU7U0FDaEIsQ0FBQztJQUNOLENBQUM7Q0FDSjtBQTVKRCwwREE0SkM7QUFDRDs7OztHQUlHO0FBQ0gsU0FBZ0IscUJBQXFCLENBQUMsYUFBNEI7SUFDOUQsSUFBSSxhQUFhLENBQUMsVUFBVSxJQUFJLENBQUMsZUFBZSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDN0UsTUFBTSxJQUFJLEtBQUssQ0FBQyx1Q0FBdUMsQ0FBQyxDQUFDO0tBQzVEO0lBQ0QsSUFBSSxhQUFhLENBQUMsV0FBVyxJQUFJLGFBQWEsQ0FBQyxXQUFXLENBQUMsTUFBTSxHQUFHLElBQUksRUFBRTtRQUN0RSxNQUFNLElBQUksS0FBSyxDQUFDLHNEQUFzRCxDQUFDLENBQUM7S0FDM0U7QUFDTCxDQUFDO0FBUEQsc0RBT0M7QUFDRDs7OztHQUlHO0FBQ0gsU0FBZ0Isd0JBQXdCLENBQUMsZ0JBQWtDO0lBQ3ZFLElBQUksZ0JBQWdCLENBQUMsUUFBUSxJQUFJLENBQUMsMkJBQTJCLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQzNGLE1BQU0sSUFBSSxLQUFLLENBQUMsaURBQWlELENBQUMsQ0FBQztLQUN0RTtJQUNELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxFQUFFO1FBQ3BGLE1BQU0sSUFBSSxLQUFLLENBQUMsNENBQTRDLENBQUMsQ0FBQztLQUNqRTtBQUNMLENBQUM7QUFQRCw0REFPQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuTGlzdGVuZXJSdWxlIH0gZnJvbSAnLi4vZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUxpc3RlbmVyQWN0aW9uIH0gZnJvbSAnLi4vc2hhcmVkL2xpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25MaXN0ZW5lciB9IGZyb20gJy4vYXBwbGljYXRpb24tbGlzdGVuZXInO1xuaW1wb3J0IHsgTGlzdGVuZXJBY3Rpb24gfSBmcm9tICcuL2FwcGxpY2F0aW9uLWxpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25UYXJnZXRHcm91cCB9IGZyb20gJy4vYXBwbGljYXRpb24tdGFyZ2V0LWdyb3VwJztcbmltcG9ydCB7IExpc3RlbmVyQ29uZGl0aW9uIH0gZnJvbSAnLi9jb25kaXRpb25zJztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgQmFzZUFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBwcmlvcml0eTogbnVtYmVyO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHRhcmdldEdyb3Vwcz86IElBcHBsaWNhdGlvblRhcmdldEdyb3VwW107XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBhY3Rpb24/OiBMaXN0ZW5lckFjdGlvbjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBmaXhlZFJlc3BvbnNlPzogRml4ZWRSZXNwb25zZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSByZWRpcmVjdFJlc3BvbnNlPzogUmVkaXJlY3RSZXNwb25zZTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY29uZGl0aW9ucz86IExpc3RlbmVyQ29uZGl0aW9uW107XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaG9zdEhlYWRlcj86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcGF0aFBhdHRlcm4/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBwYXRoUGF0dGVybnM/OiBzdHJpbmdbXTtcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlUHJvcHMgZXh0ZW5kcyBCYXNlQXBwbGljYXRpb25MaXN0ZW5lclJ1bGVQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXI7XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIENvbnRlbnRUeXBlIHtcbiAgICBURVhUX1BMQUlOID0gJ3RleHQvcGxhaW4nLFxuICAgIFRFWFRfQ1NTID0gJ3RleHQvY3NzJyxcbiAgICBURVhUX0hUTUwgPSAndGV4dC9odG1sJyxcbiAgICBBUFBMSUNBVElPTl9KQVZBU0NSSVBUID0gJ2FwcGxpY2F0aW9uL2phdmFzY3JpcHQnLFxuICAgIEFQUExJQ0FUSU9OX0pTT04gPSAnYXBwbGljYXRpb24vanNvbidcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBGaXhlZFJlc3BvbnNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHN0YXR1c0NvZGU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBjb250ZW50VHlwZT86IENvbnRlbnRUeXBlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBtZXNzYWdlQm9keT86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBSZWRpcmVjdFJlc3BvbnNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaG9zdD86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHBhdGg/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcG9ydD86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcHJvdG9jb2w/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBxdWVyeT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgc3RhdHVzQ29kZTogc3RyaW5nO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHJlYWRvbmx5IGxpc3RlbmVyUnVsZUFybjogc3RyaW5nO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgY29uZGl0aW9uczogTGlzdGVuZXJDb25kaXRpb25bXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGxlZ2FjeUNvbmRpdGlvbnM6IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nW107XG4gICAgfSA9IHt9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgbGlzdGVuZXI6IElBcHBsaWNhdGlvbkxpc3RlbmVyO1xuICAgIHByaXZhdGUgYWN0aW9uPzogSUxpc3RlbmVyQWN0aW9uO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMuY29uZGl0aW9ucyA9IHByb3BzLmNvbmRpdGlvbnMgfHwgW107XG4gICAgICAgIGNvbnN0IGhhc1BhdGhQYXR0ZXJucyA9IHByb3BzLnBhdGhQYXR0ZXJucyB8fCBwcm9wcy5wYXRoUGF0dGVybjtcbiAgICAgICAgaWYgKHRoaXMuY29uZGl0aW9ucy5sZW5ndGggPT09IDAgJiYgIXByb3BzLmhvc3RIZWFkZXIgJiYgIWhhc1BhdGhQYXR0ZXJucykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgb2YgXFwnY29uZGl0aW9uc1xcJywgXFwnaG9zdEhlYWRlclxcJywgXFwncGF0aFBhdHRlcm5cXCcgb3IgXFwncGF0aFBhdHRlcm5zXFwnIGlzIHJlcXVpcmVkIHdoZW4gZGVmaW5pbmcgYSBsb2FkIGJhbGFuY2luZyBydWxlLicpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHBvc3NpYmxlQWN0aW9uczogQXJyYXk8a2V5b2YgQXBwbGljYXRpb25MaXN0ZW5lclJ1bGVQcm9wcz4gPSBbJ2FjdGlvbicsICd0YXJnZXRHcm91cHMnLCAnZml4ZWRSZXNwb25zZScsICdyZWRpcmVjdFJlc3BvbnNlJ107XG4gICAgICAgIGNvbnN0IHByb3ZpZGVkQWN0aW9ucyA9IHBvc3NpYmxlQWN0aW9ucy5maWx0ZXIoYWN0aW9uID0+IHByb3BzW2FjdGlvbl0gIT09IHVuZGVmaW5lZCk7XG4gICAgICAgIGlmIChwcm92aWRlZEFjdGlvbnMubGVuZ3RoID4gMSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAnJHtwcm92aWRlZEFjdGlvbnN9JyBzcGVjaWZpZWQgdG9nZXRoZXIsIHNwZWNpZnkgb25seSBvbmVgKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMucHJpb3JpdHkgPD0gMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQcmlvcml0eSBtdXN0IGhhdmUgdmFsdWUgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIDEnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmxpc3RlbmVyID0gcHJvcHMubGlzdGVuZXI7XG4gICAgICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkxpc3RlbmVyUnVsZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBsaXN0ZW5lckFybjogcHJvcHMubGlzdGVuZXIubGlzdGVuZXJBcm4sXG4gICAgICAgICAgICBwcmlvcml0eTogcHJvcHMucHJpb3JpdHksXG4gICAgICAgICAgICBjb25kaXRpb25zOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQ29uZGl0aW9ucygpIH0pLFxuICAgICAgICAgICAgYWN0aW9uczogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmFjdGlvbiA/IHRoaXMuYWN0aW9uLnJlbmRlckFjdGlvbnMoKSA6IFtdIH0pLFxuICAgICAgICB9KTtcbiAgICAgICAgaWYgKHByb3BzLmhvc3RIZWFkZXIpIHtcbiAgICAgICAgICAgIHRoaXMuc2V0Q29uZGl0aW9uKCdob3N0LWhlYWRlcicsIFtwcm9wcy5ob3N0SGVhZGVyXSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKGhhc1BhdGhQYXR0ZXJucykge1xuICAgICAgICAgICAgaWYgKHByb3BzLnBhdGhQYXR0ZXJuICYmIHByb3BzLnBhdGhQYXR0ZXJucykge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQm90aCBgcGF0aFBhdHRlcm5zYCBhbmQgYHBhdGhQYXR0ZXJuYCBhcmUgc3BlY2lmaWVkLCBzcGVjaWZ5IG9ubHkgb25lJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCBwYXRoUGF0dGVybiA9IHByb3BzLnBhdGhQYXR0ZXJuID8gW3Byb3BzLnBhdGhQYXR0ZXJuXSA6IHByb3BzLnBhdGhQYXR0ZXJucztcbiAgICAgICAgICAgIHRoaXMuc2V0Q29uZGl0aW9uKCdwYXRoLXBhdHRlcm4nLCBwYXRoUGF0dGVybik7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmFjdGlvbikge1xuICAgICAgICAgICAgdGhpcy5jb25maWd1cmVBY3Rpb24ocHJvcHMuYWN0aW9uKTtcbiAgICAgICAgfVxuICAgICAgICAocHJvcHMudGFyZ2V0R3JvdXBzIHx8IFtdKS5mb3JFYWNoKHRoaXMuYWRkVGFyZ2V0R3JvdXAuYmluZCh0aGlzKSk7XG4gICAgICAgIGlmIChwcm9wcy5maXhlZFJlc3BvbnNlKSB7XG4gICAgICAgICAgICB0aGlzLmFkZEZpeGVkUmVzcG9uc2UocHJvcHMuZml4ZWRSZXNwb25zZSk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAocHJvcHMucmVkaXJlY3RSZXNwb25zZSkge1xuICAgICAgICAgICAgdGhpcy5hZGRSZWRpcmVjdFJlc3BvbnNlKHByb3BzLnJlZGlyZWN0UmVzcG9uc2UpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMubGlzdGVuZXJSdWxlQXJuID0gcmVzb3VyY2UucmVmO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHNldENvbmRpdGlvbihmaWVsZDogc3RyaW5nLCB2YWx1ZXM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh2YWx1ZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgZGVsZXRlIHRoaXMubGVnYWN5Q29uZGl0aW9uc1tmaWVsZF07XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5sZWdhY3lDb25kaXRpb25zW2ZpZWxkXSA9IHZhbHVlcztcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZENvbmRpdGlvbihjb25kaXRpb246IExpc3RlbmVyQ29uZGl0aW9uKSB7XG4gICAgICAgIHRoaXMuY29uZGl0aW9ucy5wdXNoKGNvbmRpdGlvbik7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgY29uZmlndXJlQWN0aW9uKGFjdGlvbjogTGlzdGVuZXJBY3Rpb24pIHtcbiAgICAgICAgLy8gSXQgbWlnaHQgbWFrZSBzZW5zZSB0byAndGhyb3cnIGhlcmUuXG4gICAgICAgIC8vXG4gICAgICAgIC8vIEhvd2V2ZXIsIHByb2dyYW1zIG1heSBhbHJlYWR5IGV4aXN0IG91dCB0aGVyZSB3aGljaCBjb25maWd1cmVkIGFuIGFjdGlvbiB0d2ljZSxcbiAgICAgICAgLy8gaW4gd2hpY2ggY2FzZSB0aGUgc2Vjb25kIGFjdGlvbiBhY2NpZGVudGFsbHkgb3ZlcndyaXRlIHRoZSBpbml0aWFsIGFjdGlvbiwgYW5kIGluIHNvbWVcbiAgICAgICAgLy8gd2F5IGVuZGVkIHVwIHdpdGggYSBwcm9ncmFtIHRoYXQgZGlkIHdoYXQgdGhlIGF1dGhvciBpbnRlbmRlZC4gSWYgd2Ugd2VyZSB0byBhZGQgdGhyb3cgbm93LFxuICAgICAgICAvLyB0aGUgcHJldmlvdXNseSB3b3JraW5nIHByb2dyYW0gd291bGQgYmUgYnJva2VuLlxuICAgICAgICAvL1xuICAgICAgICAvLyBJbnN0ZWFkLCBzaWduYWwgdGhpcyB0aHJvdWdoIGEgd2FybmluZy5cbiAgICAgICAgLy8gQGRlcHJlY2F0ZTogdXBvbiB0aGUgbmV4dCBtYWpvciB2ZXJzaW9uIGJ1bXAsIHJlcGxhY2UgdGhpcyB3aXRoIGEgYHRocm93YFxuICAgICAgICBpZiAodGhpcy5hY3Rpb24pIHtcbiAgICAgICAgICAgIGNkay5Bbm5vdGF0aW9ucy5vZih0aGlzKS5hZGRXYXJuaW5nKCdBbiBBY3Rpb24gYWxyZWFkeSBleGlzdGVkIG9uIHRoaXMgTGlzdGVuZXJSdWxlIGFuZCB3YXMgcmVwbGFjZWQuIENvbmZpZ3VyZSBleGFjdGx5IG9uZSBkZWZhdWx0IEFjdGlvbi4nKTtcbiAgICAgICAgfVxuICAgICAgICBhY3Rpb24uYmluZCh0aGlzLCB0aGlzLmxpc3RlbmVyLCB0aGlzKTtcbiAgICAgICAgdGhpcy5hY3Rpb24gPSBhY3Rpb247XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkVGFyZ2V0R3JvdXAodGFyZ2V0R3JvdXA6IElBcHBsaWNhdGlvblRhcmdldEdyb3VwKSB7XG4gICAgICAgIHRoaXMuY29uZmlndXJlQWN0aW9uKExpc3RlbmVyQWN0aW9uLmZvcndhcmQoW3RhcmdldEdyb3VwXSkpO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkRml4ZWRSZXNwb25zZShmaXhlZFJlc3BvbnNlOiBGaXhlZFJlc3BvbnNlKSB7XG4gICAgICAgIHZhbGlkYXRlRml4ZWRSZXNwb25zZShmaXhlZFJlc3BvbnNlKTtcbiAgICAgICAgdGhpcy5jb25maWd1cmVBY3Rpb24oTGlzdGVuZXJBY3Rpb24uZml4ZWRSZXNwb25zZShjZGsuVG9rZW4uYXNOdW1iZXIoZml4ZWRSZXNwb25zZS5zdGF0dXNDb2RlKSwge1xuICAgICAgICAgICAgY29udGVudFR5cGU6IGZpeGVkUmVzcG9uc2UuY29udGVudFR5cGUsXG4gICAgICAgICAgICBtZXNzYWdlQm9keTogZml4ZWRSZXNwb25zZS5tZXNzYWdlQm9keSxcbiAgICAgICAgfSkpO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgYWRkUmVkaXJlY3RSZXNwb25zZShyZWRpcmVjdFJlc3BvbnNlOiBSZWRpcmVjdFJlc3BvbnNlKSB7XG4gICAgICAgIHZhbGlkYXRlUmVkaXJlY3RSZXNwb25zZShyZWRpcmVjdFJlc3BvbnNlKTtcbiAgICAgICAgdGhpcy5jb25maWd1cmVBY3Rpb24oTGlzdGVuZXJBY3Rpb24ucmVkaXJlY3Qoe1xuICAgICAgICAgICAgaG9zdDogcmVkaXJlY3RSZXNwb25zZS5ob3N0LFxuICAgICAgICAgICAgcGF0aDogcmVkaXJlY3RSZXNwb25zZS5wYXRoLFxuICAgICAgICAgICAgcGVybWFuZW50OiByZWRpcmVjdFJlc3BvbnNlLnN0YXR1c0NvZGUgPT09ICdIVFRQXzMwMScsXG4gICAgICAgICAgICBwb3J0OiByZWRpcmVjdFJlc3BvbnNlLnBvcnQsXG4gICAgICAgICAgICBwcm90b2NvbDogcmVkaXJlY3RSZXNwb25zZS5wcm90b2NvbCxcbiAgICAgICAgICAgIHF1ZXJ5OiByZWRpcmVjdFJlc3BvbnNlLnF1ZXJ5LFxuICAgICAgICB9KSk7XG4gICAgfVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICAgICAgaWYgKHRoaXMuYWN0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHJldHVybiBbJ0xpc3RlbmVyIHJ1bGUgbmVlZHMgYXQgbGVhc3Qgb25lIGFjdGlvbiddO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGxlZ2FjeUNvbmRpdGlvbkZpZWxkcyA9IE9iamVjdC5rZXlzKHRoaXMubGVnYWN5Q29uZGl0aW9ucyk7XG4gICAgICAgIGlmIChsZWdhY3lDb25kaXRpb25GaWVsZHMubGVuZ3RoID09PSAwICYmIHRoaXMuY29uZGl0aW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiBbJ0xpc3RlbmVyIHJ1bGUgbmVlZHMgYXQgbGVhc3Qgb25lIGNvbmRpdGlvbiddO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVuZGVyIHRoZSBjb25kaXRpb25zIGZvciB0aGlzIHJ1bGVcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlbmRlckNvbmRpdGlvbnMoKTogYW55IHtcbiAgICAgICAgY29uc3QgbGVnYWN5Q29uZGl0aW9ucyA9IE9iamVjdC5lbnRyaWVzKHRoaXMubGVnYWN5Q29uZGl0aW9ucykubWFwKChbZmllbGQsIHZhbHVlc10pID0+IHtcbiAgICAgICAgICAgIHJldHVybiB7IGZpZWxkLCB2YWx1ZXMgfTtcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGNvbmRpdGlvbnMgPSB0aGlzLmNvbmRpdGlvbnMubWFwKGNvbmRpdGlvbiA9PiBjb25kaXRpb24ucmVuZGVyUmF3Q29uZGl0aW9uKCkpO1xuICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgLi4ubGVnYWN5Q29uZGl0aW9ucyxcbiAgICAgICAgICAgIC4uLmNvbmRpdGlvbnMsXG4gICAgICAgIF07XG4gICAgfVxufVxuLyoqXG4gKiBWYWxpZGF0ZSB0aGUgc3RhdHVzIGNvZGUgYW5kIG1lc3NhZ2UgYm9keSBvZiBhIGZpeGVkIHJlc3BvbnNlXG4gKlxuICogQGludGVybmFsXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB2YWxpZGF0ZUZpeGVkUmVzcG9uc2UoZml4ZWRSZXNwb25zZTogRml4ZWRSZXNwb25zZSkge1xuICAgIGlmIChmaXhlZFJlc3BvbnNlLnN0YXR1c0NvZGUgJiYgIS9eKDJ8NHw1KVxcZFxcZCQvLnRlc3QoZml4ZWRSZXNwb25zZS5zdGF0dXNDb2RlKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2BzdGF0dXNDb2RlYCBtdXN0IGJlIDJYWCwgNFhYIG9yIDVYWC4nKTtcbiAgICB9XG4gICAgaWYgKGZpeGVkUmVzcG9uc2UubWVzc2FnZUJvZHkgJiYgZml4ZWRSZXNwb25zZS5tZXNzYWdlQm9keS5sZW5ndGggPiAxMDI0KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYG1lc3NhZ2VCb2R5YCBjYW5ub3QgaGF2ZSBtb3JlIHRoYW4gMTAyNCBjaGFyYWN0ZXJzLicpO1xuICAgIH1cbn1cbi8qKlxuICogVmFsaWRhdGUgdGhlIHN0YXR1cyBjb2RlIGFuZCBtZXNzYWdlIGJvZHkgb2YgYSByZWRpcmVjdCByZXNwb25zZVxuICpcbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgZnVuY3Rpb24gdmFsaWRhdGVSZWRpcmVjdFJlc3BvbnNlKHJlZGlyZWN0UmVzcG9uc2U6IFJlZGlyZWN0UmVzcG9uc2UpIHtcbiAgICBpZiAocmVkaXJlY3RSZXNwb25zZS5wcm90b2NvbCAmJiAhL14oSFRUUFM/fCNcXHtwcm90b2NvbFxcfSkkL2kudGVzdChyZWRpcmVjdFJlc3BvbnNlLnByb3RvY29sKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2Bwcm90b2NvbGAgbXVzdCBiZSBIVFRQLCBIVFRQUywgb3IgI3twcm90b2NvbH0uJyk7XG4gICAgfVxuICAgIGlmICghcmVkaXJlY3RSZXNwb25zZS5zdGF0dXNDb2RlIHx8ICEvXkhUVFBfMzBbMTJdJC8udGVzdChyZWRpcmVjdFJlc3BvbnNlLnN0YXR1c0NvZGUpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYHN0YXR1c0NvZGVgIG11c3QgYmUgSFRUUF8zMDEgb3IgSFRUUF8zMDIuJyk7XG4gICAgfVxufVxuIl19