"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MachineImageType = exports.DefaultCapacityType = exports.CoreDnsComputeType = exports.CpuArch = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = exports.KubernetesVersion = exports.EndpointAccess = void 0;
const fs = require("fs");
const path = require("path");
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const YAML = require("yaml");
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const fargate_profile_1 = require("./fargate-profile");
const helm_chart_1 = require("./helm-chart");
const instance_types_1 = require("./instance-types");
const k8s_manifest_1 = require("./k8s-manifest");
const k8s_object_value_1 = require("./k8s-object-value");
const k8s_patch_1 = require("./k8s-patch");
const kubectl_provider_1 = require("./kubectl-provider");
const managed_nodegroup_1 = require("./managed-nodegroup");
const bottlerocket_1 = require("./private/bottlerocket");
const service_account_1 = require("./service-account");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * (experimental) Endpoint access characteristics.
 *
 * @experimental
 */
class EndpointAccess {
    constructor(
    /**
     * Configuration properties.
     *
     * @internal
     */
    _config) {
        this._config = _config;
        if (!_config.publicAccess && _config.publicCidrs && _config.publicCidrs.length > 0) {
            throw new Error('CIDR blocks can only be configured when public access is enabled');
        }
    }
    /**
     * (experimental) Restrict public access to specific CIDR blocks.
     *
     * If public access is disabled, this method will result in an error.
     *
     * @param cidr CIDR blocks.
     * @experimental
     */
    onlyFrom(...cidr) {
        if (!this._config.privateAccess) {
            // when private access is disabled, we can't restric public
            // access since it will render the kubectl provider unusable.
            throw new Error('Cannot restric public access to endpoint when private access is disabled. Use PUBLIC_AND_PRIVATE.onlyFrom() instead.');
        }
        return new EndpointAccess({
            ...this._config,
            // override CIDR
            publicCidrs: cidr,
        });
    }
}
exports.EndpointAccess = EndpointAccess;
/**
 * (experimental) The cluster endpoint is accessible from outside of your VPC.
 *
 * Worker node traffic will leave your VPC to connect to the endpoint.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @experimental
 */
EndpointAccess.PUBLIC = new EndpointAccess({ privateAccess: false, publicAccess: true });
/**
 * (experimental) The cluster endpoint is only accessible through your VPC.
 *
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * @experimental
 */
EndpointAccess.PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: false });
/**
 * (experimental) The cluster endpoint is accessible from outside of your VPC.
 *
 * Worker node traffic to the endpoint will stay within your VPC.
 *
 * By default, the endpoint is exposed to all adresses. You can optionally limit the CIDR blocks that can access the public endpoint using the `PUBLIC_AND_PRIVATE.onlyFrom` method.
 * If you limit access to specific CIDR blocks, you must ensure that the CIDR blocks that you
 * specify include the addresses that worker nodes and Fargate pods (if you use them)
 * access the public endpoint from.
 *
 * @experimental
 */
EndpointAccess.PUBLIC_AND_PRIVATE = new EndpointAccess({ privateAccess: true, publicAccess: true });
/**
 * (experimental) Kubernetes cluster version.
 *
 * @experimental
 */
class KubernetesVersion {
    /**
     *
     * @param version cluster version number
     */
    constructor(version) {
        this.version = version;
    }
    /**
     * (experimental) Custom cluster version.
     *
     * @param version custom version number.
     * @experimental
     */
    static of(version) { return new KubernetesVersion(version); }
}
exports.KubernetesVersion = KubernetesVersion;
/**
 * (experimental) Kubernetes version 1.14.
 *
 * @experimental
 */
KubernetesVersion.V1_14 = KubernetesVersion.of('1.14');
/**
 * (experimental) Kubernetes version 1.15.
 *
 * @experimental
 */
KubernetesVersion.V1_15 = KubernetesVersion.of('1.15');
/**
 * (experimental) Kubernetes version 1.16.
 *
 * @experimental
 */
KubernetesVersion.V1_16 = KubernetesVersion.of('1.16');
/**
 * (experimental) Kubernetes version 1.17.
 *
 * @experimental
 */
KubernetesVersion.V1_17 = KubernetesVersion.of('1.17');
class ClusterBase extends core_1.Resource {
    /**
     * (experimental) Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest.
     * @param manifest a list of Kubernetes resource specifications.
     * @returns a `KubernetesResource` object.
     * @experimental
     */
    addManifest(id, ...manifest) {
        return new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * (experimental) Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` construct
     * @experimental
     */
    addHelmChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
}
/**
 * (experimental) A Cluster represents a managed Kubernetes Service (EKS).
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 *
 * @experimental
 */
class Cluster extends ClusterBase {
    /**
     * (experimental) Initiates an EKS Cluster with the supplied arguments.
     *
     * @param scope a Construct, most likely a cdk.Stack created.
     * @param props properties in the IClusterProps interface.
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * If the cluster has one (or more) FargateProfiles associated, this array
         * will hold a reference to each.
         */
        this._fargateProfiles = [];
        if (props.kubectlEnabled === false) {
            throw new Error('The "eks.Cluster" class no longer allows disabling kubectl support. ' +
                'As a temporary workaround, you can use the drop-in replacement class `eks.LegacyCluster`, ' +
                'but bear in mind that this class will soon be removed and will no longer receive additional ' +
                'features or bugfixes. See https://github.com/aws/aws-cdk/issues/9332 for more details');
        }
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        // this is the role used by EKS when interacting with AWS resources
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.vpcSubnets = (_a = props.vpcSubnets) !== null && _a !== void 0 ? _a : [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        // Get subnetIds for all selected subnets
        const subnetIds = [...new Set(Array().concat(...this.vpcSubnets.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        this.endpointAccess = (_b = props.endpointAccess) !== null && _b !== void 0 ? _b : EndpointAccess.PUBLIC_AND_PRIVATE;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlLayer = props.kubectlLayer;
        const privateSubents = this.selectPrivateSubnets().slice(0, 16);
        const publicAccessDisabled = !this.endpointAccess._config.publicAccess;
        const publicAccessRestricted = !publicAccessDisabled
            && this.endpointAccess._config.publicCidrs
            && this.endpointAccess._config.publicCidrs.length !== 0;
        // validate endpoint access configuration
        if (privateSubents.length === 0 && publicAccessDisabled) {
            // no private subnets and no public access at all, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is disabled');
        }
        if (privateSubents.length === 0 && publicAccessRestricted) {
            // no private subents and public access is restricted, no good.
            throw new Error('Vpc must contain private subnets when public endpoint access is restricted');
        }
        const resource = this._clusterResource = new cluster_resource_1.ClusterResource(this, 'Resource', {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
            ...(props.secretsEncryptionKey ? {
                encryptionConfig: [{
                        provider: {
                            keyArn: props.secretsEncryptionKey.keyArn,
                        },
                        resources: ['secrets'],
                    }],
            } : {}),
            endpointPrivateAccess: this.endpointAccess._config.privateAccess,
            endpointPublicAccess: this.endpointAccess._config.publicAccess,
            publicAccessCidrs: this.endpointAccess._config.publicCidrs,
            secretsEncryptionKey: props.secretsEncryptionKey,
            vpc: this.vpc,
        });
        if (this.endpointAccess._config.privateAccess && privateSubents.length !== 0) {
            // when private access is enabled and the vpc has private subnets, lets connect
            // the provider to the vpc so that it will work even when restricting public access.
            // validate VPC properties according to: https://docs.aws.amazon.com/eks/latest/userguide/cluster-endpoint.html
            if (this.vpc instanceof ec2.Vpc && !(this.vpc.dnsHostnamesEnabled && this.vpc.dnsSupportEnabled)) {
                throw new Error('Private endpoint access requires the VPC to have DNS support and DNS hostnames enabled. Use `enableDnsHostnames: true` and `enableDnsSupport: true` when creating the VPC.');
            }
            this.kubectlPrivateSubnets = privateSubents;
            // the vpc must exist in order to properly delete the cluster (since we run `kubectl delete`).
            // this ensures that.
            this._clusterResource.node.addDependency(this.vpc);
        }
        this.adminRole = resource.adminRole;
        // we use an SSM parameter as a barrier because it's free and fast.
        this._kubectlReadyBarrier = new core_1.CfnResource(this, 'KubectlReadyBarrier', {
            type: 'AWS::SSM::Parameter',
            properties: {
                Type: 'String',
                Value: 'aws:cdk:eks:kubectl-ready',
            },
        });
        // add the cluster resource itself as a dependency of the barrier
        this._kubectlReadyBarrier.node.addDependency(this._clusterResource);
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, cluster_resource_1.clusterArnComponents(this.physicalName));
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        this.clusterSecurityGroupId = resource.attrClusterSecurityGroupId;
        this.clusterEncryptionConfigKeyArn = resource.attrEncryptionConfigKeyArn;
        const clusterSecurityGroup = ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', this.clusterSecurityGroupId);
        this.connections = new ec2.Connections({
            securityGroups: [clusterSecurityGroup, securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // we can use the cluster security group since its already attached to the cluster
        // and configured to allow connections from itself.
        this.kubectlSecurityGroup = clusterSecurityGroup;
        // use the cluster creation role to issue kubectl commands against the cluster because when the
        // cluster is first created, that's the only role that has "system:masters" permissions
        this.kubectlRole = this.adminRole;
        this._kubectlResourceProvider = this.defineKubectlProvider();
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // if an explicit role is not configured, define a masters role that can
        // be assumed by anyone in the account (with sts:AssumeRole permissions of
        // course)
        const mastersRole = (_c = props.mastersRole) !== null && _c !== void 0 ? _c : new iam.Role(this, 'MastersRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // map the IAM role to the `system:masters` group.
        this.awsAuth.addMastersRole(mastersRole);
        if (props.outputMastersRoleArn) {
            new core_1.CfnOutput(this, 'MastersRoleArn', { value: mastersRole.roleArn });
        }
        commonCommandOptions.push(`--role-arn ${mastersRole.roleArn}`);
        // allocate default capacity if non-zero (or default).
        const minCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (minCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = props.defaultCapacityType === DefaultCapacityType.EC2 ?
                this.addAutoScalingGroupCapacity('DefaultCapacity', { instanceType, minCapacity }) : undefined;
            this.defaultNodegroup = props.defaultCapacityType !== DefaultCapacityType.EC2 ?
                this.addNodegroupCapacity('DefaultCapacity', { instanceType, minSize: minCapacity }) : undefined;
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
        this.defineCoreDnsComputeType((_d = props.coreDnsComputeType) !== null && _d !== void 0 ? _d : CoreDnsComputeType.EC2);
    }
    /**
     * (experimental) Import an existing cluster.
     *
     * @param scope the construct scope, in most cases 'this'.
     * @param id the id or name to import as.
     * @param attrs the cluster properties to use for importing information.
     * @experimental
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * (experimental) Fetch the load balancer address of a service of type 'LoadBalancer'.
     *
     * @param serviceName The name of the service.
     * @param options Additional operation options.
     * @experimental
     */
    getServiceLoadBalancerAddress(serviceName, options = {}) {
        const loadBalancerAddress = new k8s_object_value_1.KubernetesObjectValue(this, `${serviceName}LoadBalancerAddress`, {
            cluster: this,
            objectType: 'service',
            objectName: serviceName,
            objectNamespace: options.namespace,
            jsonPath: '.status.loadBalancer.ingress[0].hostname',
            timeout: options.timeout,
        });
        return loadBalancerAddress.value;
    }
    /**
     * (experimental) Add nodes to this EKS cluster.
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Note that if you specify `updateType: RollingUpdate` or `updateType: ReplacingUpdate`, your nodes might be replaced at deploy
     * time without notice in case the recommended AMI for your machine image type has been updated by AWS.
     * The default behavior for `updateType` is `None`, which means only new instances will be launched using the new AMI.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * In addition, the [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * @experimental
     */
    addAutoScalingGroupCapacity(id, options) {
        if (options.machineImageType === MachineImageType.BOTTLEROCKET && options.bootstrapOptions !== undefined) {
            throw new Error('bootstrapOptions is not supported for Bottlerocket');
        }
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: options.machineImageType === MachineImageType.BOTTLEROCKET ?
                new bottlerocket_1.BottleRocketImage({
                    kubernetesVersion: this.version.version,
                }) :
                new EksOptimizedImage({
                    nodeType: nodeTypeForInstanceType(options.instanceType),
                    cpuArch: cpuArchForInstanceType(options.instanceType),
                    kubernetesVersion: this.version.version,
                }),
            updateType: options.updateType,
            instanceType: options.instanceType,
        });
        this.connectAutoScalingGroupCapacity(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
            machineImageType: options.machineImageType,
        });
        if (nodeTypeForInstanceType(options.instanceType) === NodeType.INFERENTIA) {
            this.addNeuronDevicePlugin();
        }
        return asg;
    }
    /**
     * (experimental) Add managed nodegroup to this Amazon EKS cluster.
     *
     * This method will create a new managed nodegroup and add into the capacity.
     *
     * @param id The ID of the nodegroup.
     * @param options options for creating a new nodegroup.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/managed-node-groups.html
     * @experimental
     */
    addNodegroupCapacity(id, options) {
        return new managed_nodegroup_1.Nodegroup(this, `Nodegroup${id}`, {
            cluster: this,
            ...options,
        });
    }
    /**
     * (experimental) Connect capacity in the form of an existing AutoScalingGroup to the EKS cluster.
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addAutoScalingGroupCapacity` if possible.
     *
     * @param autoScalingGroup [disable-awslint:ref-via-interface].
     * @param options options for adding auto scaling groups, like customizing the bootstrap script.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @experimental
     */
    connectAutoScalingGroupCapacity(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = options.machineImageType === MachineImageType.BOTTLEROCKET ?
                user_data_1.renderBottlerocketUserData(this) :
                user_data_1.renderAmazonLinuxUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice) {
            this.addSpotInterruptHandler();
        }
    }
    /**
     * (experimental) Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     *
     * @experimental
     */
    get awsAuth() {
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * (experimental) If this cluster is kubectl-enabled, returns the OpenID Connect issuer url.
     *
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     *
     * @experimental
     * @attribute true
     */
    get clusterOpenIdConnectIssuerUrl() {
        return this._clusterResource.attrOpenIdConnectIssuerUrl;
    }
    /**
     * (experimental) If this cluster is kubectl-enabled, returns the OpenID Connect issuer.
     *
     * This is because the values is only be retrieved by the API and not exposed
     * by CloudFormation. If this cluster is not kubectl-enabled (i.e. uses the
     * stock `CfnCluster`), this is `undefined`.
     *
     * @experimental
     * @attribute true
     */
    get clusterOpenIdConnectIssuer() {
        return this._clusterResource.attrOpenIdConnectIssuer;
    }
    /**
     * (experimental) An `OpenIdConnectProvider` resource associated with this cluster, and which can be used to link this cluster to AWS IAM.
     *
     * A provider will only be defined if this property is accessed (lazy initialization).
     *
     * @experimental
     */
    get openIdConnectProvider() {
        if (!this._openIdConnectProvider) {
            this._openIdConnectProvider = new iam.OpenIdConnectProvider(this, 'OpenIdConnectProvider', {
                url: this.clusterOpenIdConnectIssuerUrl,
                clientIds: ['sts.amazonaws.com'],
                /**
                 * For some reason EKS isn't validating the root certificate but a intermediat certificate
                 * which is one level up in the tree. Because of the a constant thumbprint value has to be
                 * stated with this OpenID Connect provider. The certificate thumbprint is the same for all the regions.
                 */
                thumbprints: ['9e99a48a9960b14926bb7f3b02e22da2b0ab7280'],
            });
        }
        return this._openIdConnectProvider;
    }
    /**
     * (experimental) Adds a Fargate profile to this cluster.
     *
     * @param id the id of this profile.
     * @param options profile options.
     * @see https://docs.aws.amazon.com/eks/latest/userguide/fargate-profile.html
     * @experimental
     */
    addFargateProfile(id, options) {
        return new fargate_profile_1.FargateProfile(this, `fargate-profile-${id}`, {
            ...options,
            cluster: this,
        });
    }
    /**
     * (experimental) Adds a service account to this cluster.
     *
     * @param id the id of this service account.
     * @param options service account options.
     * @experimental
     */
    addServiceAccount(id, options = {}) {
        return new service_account_1.ServiceAccount(this, id, {
            ...options,
            cluster: this,
        });
    }
    /**
     * Internal API used by `FargateProfile` to keep inventory of Fargate profiles associated with
     * this cluster, for the sake of ensuring the profiles are created sequentially.
     *
     * @returns the list of FargateProfiles attached to this cluster, including the one just attached.
     * @internal
     */
    _attachFargateProfile(fargateProfile) {
        this._fargateProfiles.push(fargateProfile);
        // add all profiles as a dependency of the "kubectl-ready" barrier because all kubectl-
        // resources can only be deployed after all fargate profiles are created.
        this._kubectlReadyBarrier.node.addDependency(fargateProfile);
        return this._fargateProfiles;
    }
    /**
     * Adds a resource scope that requires `kubectl` to this cluster and returns
     * the `KubectlProvider` which is the custom resource provider that should be
     * used as the resource provider.
     *
     * Called from `HelmResource` and `KubernetesResource`
     *
     * @param resourceScope the construct scope in which kubectl resources are defined.
     *
     * @internal
     */
    _attachKubectlResourceScope(resourceScope) {
        constructs_1.Node.of(resourceScope).addDependency(this._kubectlReadyBarrier);
        return this._kubectlResourceProvider;
    }
    defineKubectlProvider() {
        const uid = '@aws-cdk/aws-eks.KubectlProvider';
        // since we can't have the provider connect to multiple networks, and we
        // wanted to avoid resource tear down, we decided for now that we will only
        // support a single EKS cluster per CFN stack.
        if (this.stack.node.tryFindChild(uid)) {
            throw new Error('Only a single EKS cluster can be defined within a CloudFormation stack');
        }
        return new kubectl_provider_1.KubectlProvider(this.stack, uid, { cluster: this });
    }
    selectPrivateSubnets() {
        const privateSubnets = [];
        for (const placement of this.vpcSubnets) {
            for (const subnet of this.vpc.selectSubnets(placement).subnets) {
                if (this.vpc.privateSubnets.includes(subnet)) {
                    // definitely private, take it.
                    privateSubnets.push(subnet);
                    continue;
                }
                if (this.vpc.publicSubnets.includes(subnet)) {
                    // definitely public, skip it.
                    continue;
                }
                // neither public and nor private - what is it then? this means its a subnet instance that was explicitly passed
                // in the subnet selection. since ISubnet doesn't contain information on type, we have to assume its private and let it
                // fail at deploy time :\ (its better than filtering it out and preventing a possibly successful deployment)
                privateSubnets.push(subnet);
            }
        }
        return privateSubnets;
    }
    /**
     * Installs the AWS spot instance interrupt handler on the cluster if it's not
     * already added.
     */
    addSpotInterruptHandler() {
        if (!this._spotInterruptHandler) {
            this._spotInterruptHandler = this.addHelmChart('spot-interrupt-handler', {
                chart: 'aws-node-termination-handler',
                version: '0.9.5',
                repository: 'https://aws.github.io/eks-charts',
                namespace: 'kube-system',
                values: {
                    'nodeSelector.lifecycle': user_data_1.LifecycleLabel.SPOT,
                },
            });
        }
        return this._spotInterruptHandler;
    }
    /**
     * Installs the Neuron device plugin on the cluster if it's not
     * already added.
     */
    addNeuronDevicePlugin() {
        if (!this._neuronDevicePlugin) {
            const fileContents = fs.readFileSync(path.join(__dirname, 'addons/neuron-device-plugin.yaml'), 'utf8');
            const sanitized = YAML.parse(fileContents);
            this._neuronDevicePlugin = this.addManifest('NeuronDevicePlugin', sanitized);
        }
        return this._neuronDevicePlugin;
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    core_1.Annotations.of(this).addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
    /**
     * Patches the CoreDNS deployment configuration and sets the "eks.amazonaws.com/compute-type"
     * annotation to either "ec2" or "fargate". Note that if "ec2" is selected, the resource is
     * omitted/removed, since the cluster is created with the "ec2" compute type by default.
     */
    defineCoreDnsComputeType(type) {
        // ec2 is the "built in" compute type of the cluster so if this is the
        // requested type we can simply omit the resource. since the resource's
        // `restorePatch` is configured to restore the value to "ec2" this means
        // that deletion of the resource will change to "ec2" as well.
        if (type === CoreDnsComputeType.EC2) {
            return;
        }
        // this is the json patch we merge into the resource based off of:
        // https://docs.aws.amazon.com/eks/latest/userguide/fargate-getting-started.html#fargate-gs-coredns
        const renderPatch = (computeType) => ({
            spec: {
                template: {
                    metadata: {
                        annotations: {
                            'eks.amazonaws.com/compute-type': computeType,
                        },
                    },
                },
            },
        });
        new k8s_patch_1.KubernetesPatch(this, 'CoreDnsComputeTypePatch', {
            cluster: this,
            resourceName: 'deployment/coredns',
            resourceNamespace: 'kube-system',
            applyPatch: renderPatch(CoreDnsComputeType.FARGATE),
            restorePatch: renderPatch(CoreDnsComputeType.EC2),
        });
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends ClusterBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.props = props;
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.clusterArn = this.stack.formatArn(cluster_resource_1.clusterArnComponents(props.clusterName));
        this.kubectlRole = props.kubectlRoleArn ? iam.Role.fromRoleArn(this, 'KubectlRole', props.kubectlRoleArn) : undefined;
        this.kubectlSecurityGroup = props.kubectlSecurityGroupId ? ec2.SecurityGroup.fromSecurityGroupId(this, 'KubectlSecurityGroup', props.kubectlSecurityGroupId) : undefined;
        this.kubectlEnvironment = props.kubectlEnvironment;
        this.kubectlPrivateSubnets = props.kubectlPrivateSubnetIds ? props.kubectlPrivateSubnetIds.map((subnetid, index) => ec2.Subnet.fromSubnetId(this, `KubectlSubnet${index}`, subnetid)) : undefined;
        this.kubectlLayer = props.kubectlLayer;
        let i = 1;
        for (const sgid of (_a = props.securityGroupIds) !== null && _a !== void 0 ? _a : []) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgid));
            i++;
        }
        if (props.clusterSecurityGroupId) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, 'ClusterSecurityGroup', props.clusterSecurityGroupId));
        }
    }
    get vpc() {
        if (!this.props.vpc) {
            throw new Error('"vpc" is not defined for this imported cluster');
        }
        return this.props.vpc;
    }
    get clusterSecurityGroupId() {
        if (!this.props.clusterSecurityGroupId) {
            throw new Error('"clusterSecurityGroupId" is not defined for this imported cluster');
        }
        return this.props.clusterSecurityGroupId;
    }
    get clusterEndpoint() {
        if (!this.props.clusterEndpoint) {
            throw new Error('"clusterEndpoint" is not defined for this imported cluster');
        }
        return this.props.clusterEndpoint;
    }
    get clusterCertificateAuthorityData() {
        if (!this.props.clusterCertificateAuthorityData) {
            throw new Error('"clusterCertificateAuthorityData" is not defined for this imported cluster');
        }
        return this.props.clusterCertificateAuthorityData;
    }
    get clusterEncryptionConfigKeyArn() {
        if (!this.props.clusterEncryptionConfigKeyArn) {
            throw new Error('"clusterEncryptionConfigKeyArn" is not defined for this imported cluster');
        }
        return this.props.clusterEncryptionConfigKeyArn;
    }
}
/**
 * (experimental) Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM.
 *
 * @experimental
 */
class EksOptimizedImage {
    /**
     * (experimental) Constructs a new instance of the EcsOptimizedAmi class.
     *
     * @experimental
     */
    constructor(props = {}) {
        var _a, _b, _c;
        this.nodeType = (_a = props.nodeType) !== null && _a !== void 0 ? _a : NodeType.STANDARD;
        this.cpuArch = (_b = props.cpuArch) !== null && _b !== void 0 ? _b : CpuArch.X86_64;
        this.kubernetesVersion = (_c = props.kubernetesVersion) !== null && _c !== void 0 ? _c : LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? this.cpuArch === CpuArch.X86_64 ?
                'amazon-linux-2/' : 'amazon-linux-2-arm64/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux-2-gpu/' : '')
            + (this.nodeType === NodeType.INFERENTIA ? 'amazon-linux-2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * (experimental) Return the correct image.
     *
     * @experimental
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * (experimental) Whether the worker nodes should support GPU or just standard instances.
 *
 * @experimental
 */
var NodeType;
(function (NodeType) {
    NodeType["STANDARD"] = "Standard";
    NodeType["GPU"] = "GPU";
    NodeType["INFERENTIA"] = "INFERENTIA";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
/**
 * (experimental) CPU architecture.
 *
 * @experimental
 */
var CpuArch;
(function (CpuArch) {
    CpuArch["ARM_64"] = "arm64";
    CpuArch["X86_64"] = "x86_64";
})(CpuArch = exports.CpuArch || (exports.CpuArch = {}));
/**
 * (experimental) The type of compute resources to use for CoreDNS.
 *
 * @experimental
 */
var CoreDnsComputeType;
(function (CoreDnsComputeType) {
    CoreDnsComputeType["EC2"] = "ec2";
    CoreDnsComputeType["FARGATE"] = "fargate";
})(CoreDnsComputeType = exports.CoreDnsComputeType || (exports.CoreDnsComputeType = {}));
/**
 * (experimental) The default capacity type for the cluster.
 *
 * @experimental
 */
var DefaultCapacityType;
(function (DefaultCapacityType) {
    DefaultCapacityType[DefaultCapacityType["NODEGROUP"] = 0] = "NODEGROUP";
    DefaultCapacityType[DefaultCapacityType["EC2"] = 1] = "EC2";
})(DefaultCapacityType = exports.DefaultCapacityType || (exports.DefaultCapacityType = {}));
/**
 * (experimental) The machine image type.
 *
 * @experimental
 */
var MachineImageType;
(function (MachineImageType) {
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
function nodeTypeForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.gpu.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU :
        instance_types_1.INSTANCE_TYPES.inferentia.includes(instanceType.toString().substring(0, 4)) ? NodeType.INFERENTIA :
            NodeType.STANDARD;
}
function cpuArchForInstanceType(instanceType) {
    return instance_types_1.INSTANCE_TYPES.graviton2.includes(instanceType.toString().substring(0, 3)) ? CpuArch.ARM_64 :
        instance_types_1.INSTANCE_TYPES.graviton.includes(instanceType.toString().substring(0, 2)) ? CpuArch.ARM_64 :
            CpuArch.X86_64;
}
//# sourceMappingURL=data:application/json;base64,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