"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterArnComponents = exports.ClusterResource = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_2.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        this.adminRole = this.createAdminRole(props);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: this.adminRole,
        });
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                // the structure of config needs to be that of 'aws.EKS.CreateClusterRequest' since its passed as is
                // to the eks.createCluster sdk invocation.
                Config: {
                    name: props.name,
                    version: props.version,
                    roleArn: props.roleArn,
                    encryptionConfig: props.encryptionConfig,
                    resourcesVpcConfig: {
                        subnetIds: props.resourcesVpcConfig.subnetIds,
                        securityGroupIds: props.resourcesVpcConfig.securityGroupIds,
                        endpointPublicAccess: props.endpointPublicAccess,
                        endpointPrivateAccess: props.endpointPrivateAccess,
                        publicAccessCidrs: props.publicAccessCidrs,
                    },
                },
                AssumeRoleArn: this.adminRole.roleArn,
                // IMPORTANT: increment this number when you add new attributes to the
                // resource. Otherwise, CloudFormation will error with "Vendor response
                // doesn't contain XXX key in object" (see #8276) by incrementing this
                // number, you will effectively cause a "no-op update" to the cluster
                // which will return the new set of attribute.
                AttributesRevision: 2,
            },
        });
        resource.node.addDependency(this.adminRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrClusterSecurityGroupId = core_1.Token.asString(resource.getAtt('ClusterSecurityGroupId'));
        this.attrEncryptionConfigKeyArn = core_1.Token.asString(resource.getAtt('EncryptionConfigKeyArn'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    createAdminRole(props) {
        const stack = core_1.Stack.of(this);
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        const creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole', 'iam:listAttachedRolePolicies'],
            resources: ['*'],
        }));
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        // see https://github.com/aws/aws-cdk/issues/9027
        creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['ec2:DescribeVpcs'],
            resources: [stack.formatArn({
                    service: 'ec2',
                    resource: 'vpc',
                    resourceName: props.vpc.vpcId,
                })],
        }));
        // grant cluster creation role sufficient permission to access the specified key
        // see https://docs.aws.amazon.com/eks/latest/userguide/create-cluster.html
        if (props.secretsEncryptionKey) {
            creationRole.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'kms:Encrypt',
                    'kms:Decrypt',
                    'kms:DescribeKey',
                    'kms:CreateGrant',
                ],
                resources: [props.secretsEncryptionKey.keyArn],
            }));
        }
        return creationRole;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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