"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HelmChart = void 0;
const path = require("path");
const aws_cloudformation_1 = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kubectl_layer_1 = require("./kubectl-layer");
/**
 * (experimental) Represents a helm chart within the Kubernetes system.
 *
 * Applies/deletes the resources using `kubectl` in sync with the resource.
 *
 * @experimental
 */
class HelmChart extends core_1.Construct {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = core_1.Stack.of(this);
        // we maintain a single manifest custom resource handler for each cluster
        const handler = this.getOrCreateHelmChartHandler(props.cluster);
        if (!handler) {
            throw new Error('Cannot define a Helm chart on a cluster with kubectl disabled');
        }
        new aws_cloudformation_1.CustomResource(this, 'Resource', {
            provider: aws_cloudformation_1.CustomResourceProvider.lambda(handler),
            resourceType: HelmChart.RESOURCE_TYPE,
            properties: {
                Release: props.release || this.node.uniqueId.slice(-63).toLowerCase(),
                Chart: props.chart,
                Version: props.version,
                Values: (props.values ? stack.toJsonString(props.values) : undefined),
                Namespace: props.namespace || 'default',
                Repository: props.repository,
            },
        });
    }
    getOrCreateHelmChartHandler(cluster) {
        if (!cluster.kubectlEnabled) {
            return undefined;
        }
        let handler = cluster.node.tryFindChild('HelmChartHandler');
        if (!handler) {
            handler = new lambda.Function(cluster, 'HelmChartHandler', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'helm-chart')),
                runtime: lambda.Runtime.PYTHON_3_7,
                handler: 'index.handler',
                timeout: core_1.Duration.minutes(15),
                layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this, { version: '2.0.0-beta1' })],
                memorySize: 256,
                environment: {
                    CLUSTER_NAME: cluster.clusterName,
                },
                // NOTE: we must use the default IAM role that's mapped to "system:masters"
                // as the execution role of this custom resource handler. This is the only
                // way to be able to interact with the cluster after it's been created.
                role: cluster._defaultMastersRole,
            });
        }
        return handler;
    }
}
exports.HelmChart = HelmChart;
/**
 * (experimental) The CloudFormation reosurce type.
 *
 * @experimental
 */
HelmChart.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-HelmChart';
//# sourceMappingURL=data:application/json;base64,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