"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BuiltInAttributes = exports.Ec2Service = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_service_1 = require("../base/base-service");
const from_service_attributes_1 = require("../base/from-service-attributes");
const task_definition_1 = require("../base/task-definition");
/**
 * (experimental) This creates a service using the EC2 launch type on an ECS cluster.
 *
 * @experimental
 * @resource AWS::ECS::Service
 */
class Ec2Service extends base_service_1.BaseService {
    /**
     * (experimental) Constructs a new instance of the Ec2Service class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        if (props.daemon && props.desiredCount !== undefined) {
            throw new Error('Daemon mode launches one task on every instance. Don\'t supply desiredCount.');
        }
        if (props.daemon && props.maxHealthyPercent !== undefined && props.maxHealthyPercent !== 100) {
            throw new Error('Maximum percent must be 100 for daemon mode.');
        }
        if (props.minHealthyPercent !== undefined && props.maxHealthyPercent !== undefined && props.minHealthyPercent >= props.maxHealthyPercent) {
            throw new Error('Minimum healthy percent must be less than maximum healthy percent.');
        }
        if (!props.taskDefinition.isEc2Compatible) {
            throw new Error('Supplied TaskDefinition is not configured for compatibility with EC2');
        }
        if (props.propagateTags && props.propagateTaskTagsFrom) {
            throw new Error('You can only specify either propagateTags or propagateTaskTagsFrom. Alternatively, you can leave both blank');
        }
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        const propagateTagsFromSource = props.propagateTaskTagsFrom !== undefined ? props.propagateTaskTagsFrom
            : (props.propagateTags !== undefined ? props.propagateTags : base_service_1.PropagatedTagSource.NONE);
        super(scope, id, {
            ...props,
            // If daemon, desiredCount must be undefined and that's what we want. Otherwise, default to 1.
            desiredCount: props.daemon || props.desiredCount !== undefined ? props.desiredCount : 1,
            maxHealthyPercent: props.daemon && props.maxHealthyPercent === undefined ? 100 : props.maxHealthyPercent,
            minHealthyPercent: props.daemon && props.minHealthyPercent === undefined ? 0 : props.minHealthyPercent,
            launchType: base_service_1.LaunchType.EC2,
            propagateTags: propagateTagsFromSource,
            enableECSManagedTags: props.enableECSManagedTags,
        }, {
            cluster: props.cluster.clusterName,
            taskDefinition: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === base_service_1.DeploymentControllerType.EXTERNAL ? undefined : props.taskDefinition.taskDefinitionArn,
            placementConstraints: core_1.Lazy.anyValue({ produce: () => this.constraints }, { omitEmptyArray: true }),
            placementStrategies: core_1.Lazy.anyValue({ produce: () => this.strategies }, { omitEmptyArray: true }),
            schedulingStrategy: props.daemon ? 'DAEMON' : 'REPLICA',
        }, props.taskDefinition);
        this.constraints = [];
        this.strategies = [];
        this.daemon = props.daemon || false;
        let securityGroups;
        if (props.securityGroup !== undefined) {
            securityGroups = [props.securityGroup];
        }
        else if (props.securityGroups !== undefined) {
            securityGroups = props.securityGroups;
        }
        if (props.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            this.configureAwsVpcNetworkingWithSecurityGroups(props.cluster.vpc, props.assignPublicIp, props.vpcSubnets, securityGroups);
        }
        else {
            // Either None, Bridge or Host networking. Copy SecurityGroups from ASG.
            // We have to be smart here -- by default future Security Group rules would be created
            // in the Cluster stack. However, if the Cluster is in a different stack than us,
            // that will lead to a cyclic reference (we point to that stack for the cluster name,
            // but that stack will point to the ALB probably created right next to us).
            //
            // In that case, reference the same security groups but make sure new rules are
            // created in the current scope (i.e., this stack)
            validateNoNetworkingProps(props);
            this.connections.addSecurityGroup(...securityGroupsInThisStack(this, props.cluster.connections.securityGroups));
        }
        this.addPlacementConstraints(...props.placementConstraints || []);
        this.addPlacementStrategies(...props.placementStrategies || []);
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('A TaskDefinition must have at least one essential container');
        }
    }
    /**
     * (experimental) Imports from the specified service ARN.
     *
     * @experimental
     */
    static fromEc2ServiceArn(scope, id, ec2ServiceArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.serviceArn = ec2ServiceArn;
                this.serviceName = core_1.Stack.of(scope).parseArn(ec2ServiceArn).resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Imports from the specified service attrributes.
     *
     * @experimental
     */
    static fromEc2ServiceAttributes(scope, id, attrs) {
        return from_service_attributes_1.fromServiceAtrributes(scope, id, attrs);
    }
    /**
     * (experimental) Adds one or more placement strategies to use for tasks in the service.
     *
     * For more information, see
     * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
     *
     * @experimental
     */
    addPlacementStrategies(...strategies) {
        if (strategies.length > 0 && this.daemon) {
            throw new Error("Can't configure placement strategies when daemon=true");
        }
        for (const strategy of strategies) {
            this.strategies.push(...strategy.toJson());
        }
    }
    /**
     * (experimental) Adds one or more placement strategies to use for tasks in the service.
     *
     * For more information, see
     * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
     *
     * @experimental
     */
    addPlacementConstraints(...constraints) {
        for (const constraint of constraints) {
            this.constraints.push(...constraint.toJson());
        }
    }
    /**
     * (experimental) Validates this Ec2Service.
     *
     * @experimental
     */
    validate() {
        const ret = super.validate();
        if (!this.cluster.hasEc2Capacity) {
            ret.push('Cluster for this service needs Ec2 capacity. Call addXxxCapacity() on the cluster.');
        }
        return ret;
    }
}
exports.Ec2Service = Ec2Service;
/**
 * Validate combinations of networking arguments.
 */
function validateNoNetworkingProps(props) {
    if (props.vpcSubnets !== undefined
        || props.securityGroup !== undefined
        || props.securityGroups !== undefined
        || props.assignPublicIp) {
        throw new Error('vpcSubnets, securityGroup(s) and assignPublicIp can only be used in AwsVpc networking mode');
    }
}
/**
 * Force security group rules to be created in this stack.
 *
 * For every security group, if the scope and the group are in different stacks, return
 * a fake "imported" security group instead. This will behave as the original security group,
 * but new Ingress and Egress rule resources will be added in the current stack instead of the
 * other one.
 */
function securityGroupsInThisStack(scope, groups) {
    const thisStack = core_1.Stack.of(scope);
    let i = 1;
    return groups.map(group => {
        if (thisStack === core_1.Stack.of(group)) {
            return group;
        } // Simple case, just return the original one
        return ec2.SecurityGroup.fromSecurityGroupId(scope, `SecurityGroup${i++}`, group.securityGroupId, {
            allowAllOutbound: group.allowAllOutbound,
            mutable: true,
        });
    });
}
/**
 * (experimental) The built-in container instance attributes.
 *
 * @experimental
 */
class BuiltInAttributes {
}
exports.BuiltInAttributes = BuiltInAttributes;
/**
 * (experimental) The id of the instance.
 *
 * @experimental
 */
BuiltInAttributes.INSTANCE_ID = 'instanceId';
/**
 * (experimental) The AvailabilityZone where the instance is running in.
 *
 * @experimental
 */
BuiltInAttributes.AVAILABILITY_ZONE = 'attribute:ecs.availability-zone';
/**
 * (experimental) The AMI id the instance is using.
 *
 * @experimental
 */
BuiltInAttributes.AMI_ID = 'attribute:ecs.ami-id';
/**
 * (experimental) The EC2 instance type.
 *
 * @experimental
 */
BuiltInAttributes.INSTANCE_TYPE = 'attribute:ecs.instance-type';
/**
 * (experimental) The operating system of the instance.
 *
 * Either 'linux' or 'windows'.
 *
 * @experimental
 */
BuiltInAttributes.OS_TYPE = 'attribute:ecs.os-type';
//# sourceMappingURL=data:application/json;base64,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