"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.ContainerDependencyCondition = exports.UlimitName = exports.ContainerDefinition = exports.Secret = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("./base/task-definition");
/**
 * (experimental) A secret environment variable.
 *
 * @experimental
 */
class Secret {
    /**
     * (experimental) Creates an environment variable value from a parameter stored in AWS Systems Manager Parameter Store.
     *
     * @experimental
     */
    static fromSsmParameter(parameter) {
        return {
            arn: parameter.parameterArn,
            grantRead: grantee => parameter.grantRead(grantee),
        };
    }
    /**
     * (experimental) Creates a environment variable value from a secret stored in AWS Secrets Manager.
     *
     * @param secret the secret stored in AWS Secrets Manager.
     * @param field the name of the field with the value that you want to set as the environment variable value.
     * @experimental
     */
    static fromSecretsManager(secret, field) {
        return {
            arn: field ? `${secret.secretArn}:${field}::` : secret.secretArn,
            hasField: !!field,
            grantRead: grantee => secret.grantRead(grantee),
        };
    }
}
exports.Secret = Secret;
/**
 * (experimental) A container definition is used in a task definition to describe the containers that are launched as part of a task.
 *
 * @experimental
 */
class ContainerDefinition extends cdk.Construct {
    /**
     * (experimental) Constructs a new instance of the ContainerDefinition class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        /**
         * (experimental) The mount points for data volumes in your container.
         *
         * @experimental
         */
        this.mountPoints = new Array();
        /**
         * (experimental) The list of port mappings for the container.
         *
         * Port mappings allow containers to access ports
         * on the host container instance to send or receive traffic.
         *
         * @experimental
         */
        this.portMappings = new Array();
        /**
         * (experimental) The data volumes to mount from another container in the same task definition.
         *
         * @experimental
         */
        this.volumesFrom = new Array();
        /**
         * (experimental) An array of ulimits to set in the container.
         *
         * @experimental
         */
        this.ulimits = new Array();
        /**
         * (experimental) An array dependencies defined for container startup and shutdown.
         *
         * @experimental
         */
        this.containerDependencies = new Array();
        /**
         * The configured container links
         */
        this.links = new Array();
        if (props.memoryLimitMiB !== undefined && props.memoryReservationMiB !== undefined) {
            if (props.memoryLimitMiB < props.memoryReservationMiB) {
                throw new Error('MemoryLimitMiB should not be less than MemoryReservationMiB.');
            }
        }
        this.essential = props.essential !== undefined ? props.essential : true;
        this.taskDefinition = props.taskDefinition;
        this.memoryLimitSpecified = props.memoryLimitMiB !== undefined || props.memoryReservationMiB !== undefined;
        this.linuxParameters = props.linuxParameters;
        this.containerName = this.node.id;
        this.imageConfig = props.image.bind(this, this);
        if (props.logging) {
            this.logDriverConfig = props.logging.bind(this, this);
        }
        props.taskDefinition._linkContainer(this);
        if (props.secrets) {
            this.secrets = [];
            for (const [name, secret] of Object.entries(props.secrets)) {
                if (this.taskDefinition.isFargateCompatible && secret.hasField) {
                    throw new Error(`Cannot specify secret JSON field for a task using the FARGATE launch type: '${name}' in container '${this.node.id}'`);
                }
                secret.grantRead(this.taskDefinition.obtainExecutionRole());
                this.secrets.push({
                    name,
                    valueFrom: secret.arn,
                });
            }
        }
    }
    /**
     * (experimental) This method adds a link which allows containers to communicate with each other without the need for port mappings.
     *
     * This parameter is only supported if the task definition is using the bridge network mode.
     * Warning: The --link flag is a legacy feature of Docker. It may eventually be removed.
     *
     * @experimental
     */
    addLink(container, alias) {
        if (this.taskDefinition.networkMode !== task_definition_1.NetworkMode.BRIDGE) {
            throw new Error('You must use network mode Bridge to add container links.');
        }
        if (alias !== undefined) {
            this.links.push(`${container.containerName}:${alias}`);
        }
        else {
            this.links.push(`${container.containerName}`);
        }
    }
    /**
     * (experimental) This method adds one or more mount points for data volumes to the container.
     *
     * @experimental
     */
    addMountPoints(...mountPoints) {
        this.mountPoints.push(...mountPoints);
    }
    /**
     * (experimental) This method mounts temporary disk space to the container.
     *
     * This adds the correct container mountPoint and task definition volume.
     *
     * @experimental
     */
    addScratch(scratch) {
        const mountPoint = {
            containerPath: scratch.containerPath,
            readOnly: scratch.readOnly,
            sourceVolume: scratch.name,
        };
        const volume = {
            host: {
                sourcePath: scratch.sourcePath,
            },
            name: scratch.name,
        };
        this.taskDefinition.addVolume(volume);
        this.addMountPoints(mountPoint);
    }
    /**
     * (experimental) This method adds one or more port mappings to the container.
     *
     * @experimental
     */
    addPortMappings(...portMappings) {
        this.portMappings.push(...portMappings.map(pm => {
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC || this.taskDefinition.networkMode === task_definition_1.NetworkMode.HOST) {
                if (pm.containerPort !== pm.hostPort && pm.hostPort !== undefined) {
                    throw new Error(`Host port (${pm.hostPort}) must be left out or equal to container port ${pm.containerPort} for network mode ${this.taskDefinition.networkMode}`);
                }
            }
            if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
                if (pm.hostPort === undefined) {
                    pm = {
                        ...pm,
                        hostPort: 0,
                    };
                }
            }
            return pm;
        }));
    }
    /**
     * (experimental) This method adds one or more ulimits to the container.
     *
     * @experimental
     */
    addUlimits(...ulimits) {
        this.ulimits.push(...ulimits);
    }
    /**
     * (experimental) This method adds one or more container dependencies to the container.
     *
     * @experimental
     */
    addContainerDependencies(...containerDependencies) {
        this.containerDependencies.push(...containerDependencies);
    }
    /**
     * (experimental) This method adds one or more volumes to the container.
     *
     * @experimental
     */
    addVolumesFrom(...volumesFrom) {
        this.volumesFrom.push(...volumesFrom);
    }
    /**
     * (experimental) This method adds the specified statement to the IAM task execution policy in the task definition.
     *
     * @experimental
     */
    addToExecutionPolicy(statement) {
        this.taskDefinition.addToExecutionRolePolicy(statement);
    }
    /**
     * (experimental) Returns the host port for the requested container port if it exists.
     *
     * @experimental
     */
    findPortMapping(containerPort, protocol) {
        for (const portMapping of this.portMappings) {
            const p = portMapping.protocol || Protocol.TCP;
            const c = portMapping.containerPort;
            if (c === containerPort && p === protocol) {
                return portMapping;
            }
        }
        return undefined;
    }
    /**
     * (experimental) The inbound rules associated with the security group the task or service will use.
     *
     * This property is only used for tasks that use the awsvpc network mode.
     *
     * @experimental
     */
    get ingressPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        if (defaultPortMapping.hostPort !== undefined && defaultPortMapping.hostPort !== 0) {
            return defaultPortMapping.hostPort;
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.BRIDGE) {
            return 0;
        }
        return defaultPortMapping.containerPort;
    }
    /**
     * (experimental) The port the container will listen on.
     *
     * @experimental
     */
    get containerPort() {
        if (this.portMappings.length === 0) {
            throw new Error(`Container ${this.containerName} hasn't defined any ports. Call addPortMappings().`);
        }
        const defaultPortMapping = this.portMappings[0];
        return defaultPortMapping.containerPort;
    }
    /**
     * (experimental) Render this container definition to a CloudFormation object.
     *
     * @param _taskDefinition [disable-awslint:ref-via-interface] (unused but kept to avoid breaking change).
     * @experimental
     */
    renderContainerDefinition(_taskDefinition) {
        return {
            command: this.props.command,
            cpu: this.props.cpu,
            disableNetworking: this.props.disableNetworking,
            dependsOn: cdk.Lazy.anyValue({ produce: () => this.containerDependencies.map(renderContainerDependency) }, { omitEmptyArray: true }),
            dnsSearchDomains: this.props.dnsSearchDomains,
            dnsServers: this.props.dnsServers,
            dockerLabels: this.props.dockerLabels,
            dockerSecurityOptions: this.props.dockerSecurityOptions,
            entryPoint: this.props.entryPoint,
            essential: this.essential,
            hostname: this.props.hostname,
            image: this.imageConfig.imageName,
            memory: this.props.memoryLimitMiB,
            memoryReservation: this.props.memoryReservationMiB,
            mountPoints: cdk.Lazy.anyValue({ produce: () => this.mountPoints.map(renderMountPoint) }, { omitEmptyArray: true }),
            name: this.containerName,
            portMappings: cdk.Lazy.anyValue({ produce: () => this.portMappings.map(renderPortMapping) }, { omitEmptyArray: true }),
            privileged: this.props.privileged,
            readonlyRootFilesystem: this.props.readonlyRootFilesystem,
            repositoryCredentials: this.imageConfig.repositoryCredentials,
            startTimeout: this.props.startTimeout && this.props.startTimeout.toSeconds(),
            stopTimeout: this.props.stopTimeout && this.props.stopTimeout.toSeconds(),
            ulimits: cdk.Lazy.anyValue({ produce: () => this.ulimits.map(renderUlimit) }, { omitEmptyArray: true }),
            user: this.props.user,
            volumesFrom: cdk.Lazy.anyValue({ produce: () => this.volumesFrom.map(renderVolumeFrom) }, { omitEmptyArray: true }),
            workingDirectory: this.props.workingDirectory,
            logConfiguration: this.logDriverConfig,
            environment: this.props.environment && renderKV(this.props.environment, 'name', 'value'),
            secrets: this.secrets,
            extraHosts: this.props.extraHosts && renderKV(this.props.extraHosts, 'hostname', 'ipAddress'),
            healthCheck: this.props.healthCheck && renderHealthCheck(this.props.healthCheck),
            links: cdk.Lazy.listValue({ produce: () => this.links }, { omitEmpty: true }),
            linuxParameters: this.linuxParameters && this.linuxParameters.renderLinuxParameters(),
            resourceRequirements: (this.props.gpuCount !== undefined) ? renderResourceRequirements(this.props.gpuCount) : undefined,
        };
    }
}
exports.ContainerDefinition = ContainerDefinition;
function renderKV(env, keyName, valueName) {
    const ret = new Array();
    for (const [key, value] of Object.entries(env)) {
        ret.push({ [keyName]: key, [valueName]: value });
    }
    return ret;
}
function renderHealthCheck(hc) {
    return {
        command: getHealthCheckCommand(hc),
        interval: hc.interval != null ? hc.interval.toSeconds() : 30,
        retries: hc.retries !== undefined ? hc.retries : 3,
        startPeriod: hc.startPeriod && hc.startPeriod.toSeconds(),
        timeout: hc.timeout !== undefined ? hc.timeout.toSeconds() : 5,
    };
}
function getHealthCheckCommand(hc) {
    const cmd = hc.command;
    const hcCommand = new Array();
    if (cmd.length === 0) {
        throw new Error('At least one argument must be supplied for health check command.');
    }
    if (cmd.length === 1) {
        hcCommand.push('CMD-SHELL', cmd[0]);
        return hcCommand;
    }
    if (cmd[0] !== 'CMD' && cmd[0] !== 'CMD-SHELL') {
        hcCommand.push('CMD');
    }
    return hcCommand.concat(cmd);
}
function renderResourceRequirements(gpuCount) {
    if (gpuCount === 0) {
        return undefined;
    }
    return [{
            type: 'GPU',
            value: gpuCount.toString(),
        }];
}
/**
 * (experimental) Type of resource to set a limit on.
 *
 * @experimental
 */
var UlimitName;
(function (UlimitName) {
    UlimitName["CORE"] = "core";
    UlimitName["CPU"] = "cpu";
    UlimitName["DATA"] = "data";
    UlimitName["FSIZE"] = "fsize";
    UlimitName["LOCKS"] = "locks";
    UlimitName["MEMLOCK"] = "memlock";
    UlimitName["MSGQUEUE"] = "msgqueue";
    UlimitName["NICE"] = "nice";
    UlimitName["NOFILE"] = "nofile";
    UlimitName["NPROC"] = "nproc";
    UlimitName["RSS"] = "rss";
    UlimitName["RTPRIO"] = "rtprio";
    UlimitName["RTTIME"] = "rttime";
    UlimitName["SIGPENDING"] = "sigpending";
    UlimitName["STACK"] = "stack";
})(UlimitName = exports.UlimitName || (exports.UlimitName = {}));
function renderUlimit(ulimit) {
    return {
        name: ulimit.name,
        softLimit: ulimit.softLimit,
        hardLimit: ulimit.hardLimit,
    };
}
/**
 * @experimental
 */
var ContainerDependencyCondition;
(function (ContainerDependencyCondition) {
    ContainerDependencyCondition["START"] = "START";
    ContainerDependencyCondition["COMPLETE"] = "COMPLETE";
    ContainerDependencyCondition["SUCCESS"] = "SUCCESS";
    ContainerDependencyCondition["HEALTHY"] = "HEALTHY";
})(ContainerDependencyCondition = exports.ContainerDependencyCondition || (exports.ContainerDependencyCondition = {}));
function renderContainerDependency(containerDependency) {
    return {
        containerName: containerDependency.container.containerName,
        condition: containerDependency.condition || ContainerDependencyCondition.HEALTHY,
    };
}
/**
 * (experimental) Network protocol.
 *
 * @experimental
 */
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "tcp";
    Protocol["UDP"] = "udp";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
function renderPortMapping(pm) {
    return {
        containerPort: pm.containerPort,
        hostPort: pm.hostPort,
        protocol: pm.protocol || Protocol.TCP,
    };
}
function renderMountPoint(mp) {
    return {
        containerPath: mp.containerPath,
        readOnly: mp.readOnly,
        sourceVolume: mp.sourceVolume,
    };
}
function renderVolumeFrom(vf) {
    return {
        sourceContainer: vf.sourceContainer,
        readOnly: vf.readOnly,
    };
}
//# sourceMappingURL=data:application/json;base64,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