"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiHardwareType = exports.BottleRocketImage = exports.BottlerocketEcsVariant = exports.EcsOptimizedImage = exports.EcsOptimizedAmi = exports.WindowsOptimizedVersion = exports.Cluster = exports.MachineImageType = void 0;
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_generated_1 = require("./ecs.generated");
/**
 * (experimental) The machine image type.
 *
 * @experimental
 */
var MachineImageType;
(function (MachineImageType) {
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
/**
 * (experimental) A regional grouping of one or more container instances on which you can run tasks and services.
 *
 * @experimental
 */
class Cluster extends core_1.Resource {
    /**
     * (experimental) Constructs a new instance of the Cluster class.
     *
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * (experimental) Manage the allowed network connections for the cluster with Security Groups.
         *
         * @experimental
         */
        this.connections = new ec2.Connections();
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        const containerInsights = props.containerInsights !== undefined ? props.containerInsights : false;
        const clusterSettings = containerInsights ? [{ name: 'containerInsights', value: 'enabled' }] : undefined;
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
            clusterSettings,
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity('DefaultAutoScalingGroup', props.capacity)
            : undefined;
    }
    /**
     * (experimental) Import an existing cluster to the stack from its attributes.
     *
     * @experimental
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * (experimental) Add an AWS Cloud Map DNS namespace for this cluster.
     *
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     *
     * @experimental
     */
    addDefaultCloudMapNamespace(options) {
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error('Can only add default namespace once.');
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc,
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * (experimental) Getter for namespace added to cluster.
     *
     * @experimental
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * (experimental) This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     *
     * @experimental
     */
    addCapacity(id, options) {
        var _a;
        if (options.machineImage && options.machineImageType) {
            throw new Error('You can only specify either machineImage or machineImageType, not both.');
        }
        const machineImage = ((_a = options.machineImage) !== null && _a !== void 0 ? _a : options.machineImageType === MachineImageType.BOTTLEROCKET) ?
            new BottleRocketImage() : new EcsOptimizedAmi();
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, {
            vpc: this.vpc,
            machineImage,
            updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE,
            ...options,
        });
        this.addAutoScalingGroup(autoScalingGroup, {
            machineImageType: options.machineImageType,
            ...options,
        });
        return autoScalingGroup;
    }
    /**
     * (experimental) This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * @experimental
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        // Tie instances to cluster
        switch (options.machineImageType) {
            // Bottlerocket AMI
            case MachineImageType.BOTTLEROCKET: {
                autoScalingGroup.addUserData(
                // Connect to the cluster
                // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#connecting-to-your-cluster
                '[settings.ecs]', `cluster = "${this.clusterName}"`);
                // Enabling SSM
                // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#enabling-ssm
                autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
                // required managed policy
                autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'));
                break;
            }
            default:
                // Amazon ECS-optimized AMI for Amazon Linux 2
                autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
                if (!options.canContainersAccessInstanceRole) {
                    // Deny containers access to instance metadata service
                    // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
                    autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
                    autoScalingGroup.addUserData('sudo service iptables save');
                    // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
                    autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
                }
                if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
                    autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
                }
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        // But, scoped down to minimal permissions required.
        //  Notes:
        //   - 'ecs:CreateCluster' removed. The cluster already exists.
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DeregisterContainerInstance',
                'ecs:RegisterContainerInstance',
                'ecs:Submit*',
            ],
            resources: [
                this.clusterArn,
            ],
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These act on a cluster instance, and the instance doesn't exist until the service starts.
                // Thus, scope to the cluster using a condition.
                // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
                'ecs:Poll',
                'ecs:StartTelemetrySession',
            ],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': this.clusterArn },
            },
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These do not support resource constraints, and must be resource '*'
                'ecs:DiscoverPollEndpoint',
                'ecr:GetAuthorizationToken',
                // Preserved for backwards compatibility.
                // Users are able to enable cloudwatch agent using CDK. Existing
                // customers might be installing CW agent as part of user-data so if we
                // remove these permissions we will break that customer use cases.
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime,
                topicEncryptionKey: options.topicEncryptionKey,
            });
        }
    }
    /**
     * (experimental) Getter for autoscaling group added to cluster.
     *
     * @experimental
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * (experimental) Whether the cluster has EC2 capacity associated with it.
     *
     * @experimental
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * (experimental) This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     * @experimental
     */
    metricCpuReservation(props) {
        return this.metric('CPUReservation', props);
    }
    /**
     * (experimental) This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     * @experimental
     */
    metricMemoryReservation(props) {
        return this.metric('MemoryReservation', props);
    }
    /**
     * (experimental) This method returns the specifed CloudWatch metric for this cluster.
     *
     * @experimental
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.clusterName },
            ...props,
        }).attachTo(this);
    }
}
exports.Cluster = Cluster;
/**
 * (experimental) ECS-optimized Windows version list.
 *
 * @experimental
 */
var WindowsOptimizedVersion;
(function (WindowsOptimizedVersion) {
    WindowsOptimizedVersion["SERVER_2019"] = "2019";
    WindowsOptimizedVersion["SERVER_2016"] = "2016";
})(WindowsOptimizedVersion = exports.WindowsOptimizedVersion || (exports.WindowsOptimizedVersion = {}));
/**
 * (deprecated) Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM.
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
class EcsOptimizedAmi {
    /**
     * (deprecated) Constructs a new instance of the EcsOptimizedAmi class.
     *
     * @deprecated
     */
    constructor(props) {
        this.hwType = (props && props.hardwareType) || AmiHardwareType.STANDARD;
        if (props && props.generation) { // generation defined in the props object
            if (props.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX && this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Amazon Linux does not support special hardware type. Use Amazon Linux 2 instead');
            }
            else if (props.windowsVersion) {
                throw new Error('"windowsVersion" and Linux image "generation" cannot be both set');
            }
            else {
                this.generation = props.generation;
            }
        }
        else if (props && props.windowsVersion) {
            if (this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Windows Server does not support special hardware type');
            }
            else {
                this.windowsVersion = props.windowsVersion;
            }
        }
        else { // generation not defined in props object
            // always default to Amazon Linux v2 regardless of HW
            this.generation = ec2.AmazonLinuxGeneration.AMAZON_LINUX_2;
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * (deprecated) Return the correct image.
     *
     * @deprecated
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedAmi = EcsOptimizedAmi;
/**
 * (experimental) Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM.
 *
 * @experimental
 */
class EcsOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = props && props.hardwareType;
        if (props.windowsVersion) {
            this.windowsVersion = props.windowsVersion;
        }
        else if (props.generation) {
            this.generation = props.generation;
        }
        else {
            throw new Error('This error should never be thrown');
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * (experimental) Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM.
     *
     * @param hardwareType ECS-optimized AMI variant to use.
     * @experimental
     */
    static amazonLinux2(hardwareType = AmiHardwareType.STANDARD) {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2, hardwareType });
    }
    /**
     * (experimental) Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM.
     *
     * @experimental
     */
    static amazonLinux() {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX });
    }
    /**
     * (experimental) Construct a Windows image from the latest ECS Optimized AMI published in SSM.
     *
     * @param windowsVersion Windows Version to use.
     * @experimental
     */
    static windows(windowsVersion) {
        return new EcsOptimizedImage({ windowsVersion });
    }
    /**
     * (experimental) Return the correct image.
     *
     * @experimental
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedImage = EcsOptimizedImage;
/**
 * Amazon ECS variant
 */
var BottlerocketEcsVariant;
(function (BottlerocketEcsVariant) {
    BottlerocketEcsVariant["AWS_ECS_1"] = "aws-ecs-1";
})(BottlerocketEcsVariant = exports.BottlerocketEcsVariant || (exports.BottlerocketEcsVariant = {}));
/**
 * (experimental) Construct an Bottlerocket image from the latest AMI published in SSM.
 *
 * @experimental
 */
class BottleRocketImage {
    /**
     * (experimental) Constructs a new instance of the BottleRocketImage class.
     *
     * @experimental
     */
    constructor(props = {}) {
        var _a;
        this.variant = (_a = props.variant) !== null && _a !== void 0 ? _a : BottlerocketEcsVariant.AWS_ECS_1;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/${this.variant}/x86_64/latest/image_id`;
    }
    /**
     * (experimental) Return the correct image.
     *
     * @experimental
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
exports.BottleRocketImage = BottleRocketImage;
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this.clusterArn = props.clusterArn !== undefined ? props.clusterArn : core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName,
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups,
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
}
/**
 * (experimental) The ECS-optimized AMI variant to use.
 *
 * For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 *
 * @experimental
 */
var AmiHardwareType;
(function (AmiHardwareType) {
    AmiHardwareType["STANDARD"] = "Standard";
    AmiHardwareType["GPU"] = "GPU";
    AmiHardwareType["ARM"] = "ARM64";
})(AmiHardwareType = exports.AmiHardwareType || (exports.AmiHardwareType = {}));
//# sourceMappingURL=data:application/json;base64,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