"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Compatibility = exports.Scope = exports.PidMode = exports.IpcMode = exports.NetworkMode = exports.TaskDefinition = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_definition_1 = require("../container-definition");
const ecs_generated_1 = require("../ecs.generated");
const firelens_log_router_1 = require("../firelens-log-router");
const aws_log_driver_1 = require("../log-drivers/aws-log-driver");
class TaskDefinitionBase extends core_1.Resource {
    /**
     * (experimental) Return true if the task definition can be run on an EC2 cluster.
     *
     * @experimental
     */
    get isEc2Compatible() {
        return isEc2Compatible(this.compatibility);
    }
    /**
     * (experimental) Return true if the task definition can be run on a Fargate cluster.
     *
     * @experimental
     */
    get isFargateCompatible() {
        return isFargateCompatible(this.compatibility);
    }
}
/**
 * (experimental) The base class for all task definitions.
 *
 * @experimental
 */
class TaskDefinition extends TaskDefinitionBase {
    /**
     * (experimental) Constructs a new instance of the TaskDefinition class.
     *
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * (experimental) The container definitions.
         *
         * @experimental
         */
        this.containers = new Array();
        /**
         * All volumes
         */
        this.volumes = [];
        /**
         * Placement constraints for task instances
         */
        this.placementConstraints = new Array();
        this.family = props.family || this.node.uniqueId;
        this.compatibility = props.compatibility;
        if (props.volumes) {
            props.volumes.forEach(v => this.addVolume(v));
        }
        this.networkMode = props.networkMode !== undefined ? props.networkMode :
            this.isFargateCompatible ? NetworkMode.AWS_VPC : NetworkMode.BRIDGE;
        if (this.isFargateCompatible && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`Fargate tasks can only have AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.proxyConfiguration && this.networkMode !== NetworkMode.AWS_VPC) {
            throw new Error(`ProxyConfiguration can only be used with AwsVpc network mode, got: ${this.networkMode}`);
        }
        if (props.placementConstraints && props.placementConstraints.length > 0 && this.isFargateCompatible) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        if (this.isFargateCompatible && (!props.cpu || !props.memoryMiB)) {
            throw new Error(`Fargate-compatible tasks require both CPU (${props.cpu}) and memory (${props.memoryMiB}) specifications`);
        }
        this._executionRole = props.executionRole;
        this.taskRole = props.taskRole || new iam.Role(this, 'TaskRole', {
            assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
        });
        const taskDef = new ecs_generated_1.CfnTaskDefinition(this, 'Resource', {
            containerDefinitions: core_1.Lazy.anyValue({ produce: () => this.renderContainers() }, { omitEmptyArray: true }),
            volumes: core_1.Lazy.anyValue({ produce: () => this.renderVolumes() }, { omitEmptyArray: true }),
            executionRoleArn: core_1.Lazy.stringValue({ produce: () => this.executionRole && this.executionRole.roleArn }),
            family: this.family,
            taskRoleArn: this.taskRole.roleArn,
            requiresCompatibilities: [
                ...(isEc2Compatible(props.compatibility) ? ['EC2'] : []),
                ...(isFargateCompatible(props.compatibility) ? ['FARGATE'] : []),
            ],
            networkMode: this.renderNetworkMode(this.networkMode),
            placementConstraints: core_1.Lazy.anyValue({
                produce: () => !isFargateCompatible(this.compatibility) ? this.placementConstraints : undefined,
            }, { omitEmptyArray: true }),
            proxyConfiguration: props.proxyConfiguration ? props.proxyConfiguration.bind(this.stack, this) : undefined,
            cpu: props.cpu,
            memory: props.memoryMiB,
            ipcMode: props.ipcMode,
            pidMode: props.pidMode,
        });
        if (props.placementConstraints) {
            props.placementConstraints.forEach(pc => this.addPlacementConstraint(pc));
        }
        this.taskDefinitionArn = taskDef.ref;
    }
    /**
     * (experimental) Imports a task definition from the specified task definition ARN.
     *
     * The task will have a compatibility of EC2+Fargate.
     *
     * @experimental
     */
    static fromTaskDefinitionArn(scope, id, taskDefinitionArn) {
        class Import extends TaskDefinitionBase {
            constructor() {
                super(...arguments);
                this.taskDefinitionArn = taskDefinitionArn;
                this.compatibility = Compatibility.EC2_AND_FARGATE;
                this.executionRole = undefined;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Execution role for this task definition.
     *
     * @experimental
     */
    get executionRole() {
        return this._executionRole;
    }
    renderVolumes() {
        return this.volumes.map(renderVolume);
        function renderVolume(spec) {
            return {
                host: spec.host,
                name: spec.name,
                dockerVolumeConfiguration: spec.dockerVolumeConfiguration && {
                    autoprovision: spec.dockerVolumeConfiguration.autoprovision,
                    driver: spec.dockerVolumeConfiguration.driver,
                    driverOpts: spec.dockerVolumeConfiguration.driverOpts,
                    labels: spec.dockerVolumeConfiguration.labels,
                    scope: spec.dockerVolumeConfiguration.scope,
                },
                efsVolumeConfiguration: spec.efsVolumeConfiguration && {
                    fileSystemId: spec.efsVolumeConfiguration.fileSystemId,
                    authorizationConfig: spec.efsVolumeConfiguration.authorizationConfig,
                    rootDirectory: spec.efsVolumeConfiguration.rootDirectory,
                    transitEncryption: spec.efsVolumeConfiguration.transitEncryption,
                    transitEncryptionPort: spec.efsVolumeConfiguration.transitEncryptionPort,
                },
            };
        }
    }
    /**
     * Validate the existence of the input target and set default values.
     *
     * @internal
     */
    _validateTarget(options) {
        const targetContainer = this.findContainer(options.containerName);
        if (targetContainer === undefined) {
            throw new Error(`No container named '${options.containerName}'. Did you call "addContainer()"?`);
        }
        const targetProtocol = options.protocol || container_definition_1.Protocol.TCP;
        const targetContainerPort = options.containerPort || targetContainer.containerPort;
        const portMapping = targetContainer.findPortMapping(targetContainerPort, targetProtocol);
        if (portMapping === undefined) {
            // eslint-disable-next-line max-len
            throw new Error(`Container '${targetContainer}' has no mapping for port ${options.containerPort} and protocol ${targetProtocol}. Did you call "container.addPortMappings()"?`);
        }
        return {
            containerName: options.containerName,
            portMapping,
        };
    }
    /**
     * Returns the port range to be opened that match the provided container name and container port.
     *
     * @internal
     */
    _portRangeFromPortMapping(portMapping) {
        if (portMapping.hostPort !== undefined && portMapping.hostPort !== 0) {
            return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.hostPort) : ec2.Port.tcp(portMapping.hostPort);
        }
        if (this.networkMode === NetworkMode.BRIDGE || this.networkMode === NetworkMode.NAT) {
            return EPHEMERAL_PORT_RANGE;
        }
        return portMapping.protocol === container_definition_1.Protocol.UDP ? ec2.Port.udp(portMapping.containerPort) : ec2.Port.tcp(portMapping.containerPort);
    }
    /**
     * (experimental) Adds a policy statement to the task IAM role.
     *
     * @experimental
     */
    addToTaskRolePolicy(statement) {
        this.taskRole.addToPolicy(statement);
    }
    /**
     * (experimental) Adds a policy statement to the task execution IAM role.
     *
     * @experimental
     */
    addToExecutionRolePolicy(statement) {
        this.obtainExecutionRole().addToPolicy(statement);
    }
    /**
     * (experimental) Adds a new container to the task definition.
     *
     * @experimental
     */
    addContainer(id, props) {
        return new container_definition_1.ContainerDefinition(this, id, { taskDefinition: this, ...props });
    }
    /**
     * (experimental) Adds a firelens log router to the task definition.
     *
     * @experimental
     */
    addFirelensLogRouter(id, props) {
        // only one firelens log router is allowed in each task.
        if (this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
            throw new Error('Firelens log router is already added in this task.');
        }
        return new firelens_log_router_1.FirelensLogRouter(this, id, { taskDefinition: this, ...props });
    }
    /**
     * Links a container to this task definition.
     * @internal
     */
    _linkContainer(container) {
        this.containers.push(container);
        if (this.defaultContainer === undefined && container.essential) {
            this.defaultContainer = container;
        }
    }
    /**
     * (experimental) Adds a volume to the task definition.
     *
     * @experimental
     */
    addVolume(volume) {
        this.volumes.push(volume);
    }
    /**
     * (experimental) Adds the specified placement constraint to the task definition.
     *
     * @experimental
     */
    addPlacementConstraint(constraint) {
        if (isFargateCompatible(this.compatibility)) {
            throw new Error('Cannot set placement constraints on tasks that run on Fargate');
        }
        this.placementConstraints.push(...constraint.toJson());
    }
    /**
     * (experimental) Adds the specified extention to the task definition.
     *
     * Extension can be used to apply a packaged modification to
     * a task definition.
     *
     * @experimental
     */
    addExtension(extension) {
        extension.extend(this);
    }
    /**
     * (experimental) Creates the task execution IAM role if it doesn't already exist.
     *
     * @experimental
     */
    obtainExecutionRole() {
        if (!this._executionRole) {
            this._executionRole = new iam.Role(this, 'ExecutionRole', {
                assumedBy: new iam.ServicePrincipal('ecs-tasks.amazonaws.com'),
            });
        }
        return this._executionRole;
    }
    /**
     * (experimental) Validates the task definition.
     *
     * @experimental
     */
    validate() {
        const ret = super.validate();
        if (isEc2Compatible(this.compatibility)) {
            // EC2 mode validations
            // Container sizes
            for (const container of this.containers) {
                if (!container.memoryLimitSpecified) {
                    ret.push(`ECS Container ${container.containerName} must have at least one of 'memoryLimitMiB' or 'memoryReservationMiB' specified`);
                }
            }
        }
        return ret;
    }
    /**
     * Returns the container that match the provided containerName.
     */
    findContainer(containerName) {
        return this.containers.find(c => c.containerName === containerName);
    }
    renderNetworkMode(networkMode) {
        return (networkMode === NetworkMode.NAT) ? undefined : networkMode;
    }
    renderContainers() {
        // add firelens log router container if any application container is using firelens log driver,
        // also check if already created log router container
        for (const container of this.containers) {
            if (container.logDriverConfig && container.logDriverConfig.logDriver === 'awsfirelens'
                && !this.containers.find(x => x instanceof firelens_log_router_1.FirelensLogRouter)) {
                this.addFirelensLogRouter('log-router', {
                    image: firelens_log_router_1.obtainDefaultFluentBitECRImage(this, container.logDriverConfig),
                    firelensConfig: {
                        type: firelens_log_router_1.FirelensLogRouterType.FLUENTBIT,
                    },
                    logging: new aws_log_driver_1.AwsLogDriver({ streamPrefix: 'firelens' }),
                    memoryReservationMiB: 50,
                });
                break;
            }
        }
        return this.containers.map(x => x.renderContainerDefinition());
    }
}
exports.TaskDefinition = TaskDefinition;
/**
 * The port range to open up for dynamic port mapping
 */
const EPHEMERAL_PORT_RANGE = ec2.Port.tcpRange(32768, 65535);
/**
 * (experimental) The networking mode to use for the containers in the task.
 *
 * @experimental
 */
var NetworkMode;
(function (NetworkMode) {
    NetworkMode["NONE"] = "none";
    NetworkMode["BRIDGE"] = "bridge";
    NetworkMode["AWS_VPC"] = "awsvpc";
    NetworkMode["HOST"] = "host";
    NetworkMode["NAT"] = "nat";
})(NetworkMode = exports.NetworkMode || (exports.NetworkMode = {}));
/**
 * (experimental) The IPC resource namespace to use for the containers in the task.
 *
 * @experimental
 */
var IpcMode;
(function (IpcMode) {
    IpcMode["NONE"] = "none";
    IpcMode["HOST"] = "host";
    IpcMode["TASK"] = "task";
})(IpcMode = exports.IpcMode || (exports.IpcMode = {}));
/**
 * (experimental) The process namespace to use for the containers in the task.
 *
 * @experimental
 */
var PidMode;
(function (PidMode) {
    PidMode["HOST"] = "host";
    PidMode["TASK"] = "task";
})(PidMode = exports.PidMode || (exports.PidMode = {}));
/**
 * (experimental) The scope for the Docker volume that determines its lifecycle.
 *
 * Docker volumes that are scoped to a task are automatically provisioned when the task starts and destroyed when the task stops.
 * Docker volumes that are scoped as shared persist after the task stops.
 *
 * @experimental
 */
var Scope;
(function (Scope) {
    Scope["TASK"] = "task";
    Scope["SHARED"] = "shared";
})(Scope = exports.Scope || (exports.Scope = {}));
/**
 * (experimental) The task launch type compatibility requirement.
 *
 * @experimental
 */
var Compatibility;
(function (Compatibility) {
    Compatibility[Compatibility["EC2"] = 0] = "EC2";
    Compatibility[Compatibility["FARGATE"] = 1] = "FARGATE";
    Compatibility[Compatibility["EC2_AND_FARGATE"] = 2] = "EC2_AND_FARGATE";
})(Compatibility = exports.Compatibility || (exports.Compatibility = {}));
/**
 * Return true if the given task definition can be run on an EC2 cluster
 */
function isEc2Compatible(compatibility) {
    return [Compatibility.EC2, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
/**
 * Return true if the given task definition can be run on a Fargate cluster
 */
function isFargateCompatible(compatibility) {
    return [Compatibility.FARGATE, Compatibility.EC2_AND_FARGATE].includes(compatibility);
}
//# sourceMappingURL=data:application/json;base64,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