"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const net = require("net");
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * (experimental) The VPN connection type.
 *
 * @experimental
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * (experimental) The VPN Gateway that shall be added to the VPC.
 *
 * @experimental
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
/**
 * (experimental) Define a VPN Connection.
 *
 * @experimental
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    /* eslint-disable max-len */
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    /* eslint-enable max-len */
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        /* eslint-disable-next-line max-len */
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * (experimental) Return the given named metric for all VPN connections in the account/region.
     *
     * @experimental
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * (experimental) Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     * @experimental
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * (experimental) Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     * @experimental
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * (experimental) Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     * @experimental
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,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