"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const constructs_1 = require("constructs");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const subnet_1 = require("./subnet");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const core_2 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * (experimental) The type of Subnet.
 *
 * @experimental
 */
var SubnetType;
(function (SubnetType) {
    SubnetType["ISOLATED"] = "Isolated";
    SubnetType["PRIVATE"] = "Private";
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * (deprecated) Dependencies for NAT connectivity.
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * (experimental) If this is set to true, don't error out on trying to select subnets.
         *
         * @experimental
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * (experimental) Returns IDs of selected subnets.
     *
     * @experimental
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
        };
    }
    /**
     * (experimental) Adds a VPN Gateway to this VPC.
     *
     * @experimental
     */
    enableVpnGateway(options) {
        var _a;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_a = options.vpnRoutePropagation) !== null && _a !== void 0 ? _a : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            core_1.Annotations.of(this).addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * (experimental) Adds a new VPN connection to this VPC.
     *
     * @experimental
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * (experimental) Adds a new interface endpoint to this VPC.
     *
     * @experimental
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * (experimental) Adds a new gateway endpoint to this VPC.
     *
     * @experimental
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * (experimental) Adds a new flow log to this VPC.
     *
     * @experimental
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * (experimental) Returns the id of the VPN Gateway (if enabled).
     *
     * @experimental
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * (experimental) Return the subnets appropriate for the placement strategy.
     *
     * @experimental
     */
    selectSubnetObjects(selection = {}) {
        var _a;
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE;
            subnets = this.selectSubnetObjectsByType(type);
        }
        // Apply all the filters
        subnets = this.applySubnetFilters(subnets, (_a = selection.subnetFilters) !== null && _a !== void 0 ? _a : []);
        return subnets;
    }
    applySubnetFilters(subnets, filters) {
        let filtered = subnets;
        // Apply each filter in sequence
        for (const filter of filters) {
            filtered = filter.selectSubnets(filtered);
        }
        return filtered;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        var _a;
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            else {
                core_1.Annotations.of(this).addWarning('Usage of \'subnetName\' in SubnetSelection is deprecated, use \'subnetGroupName\' instead');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            let subnetType = this.privateSubnets.length ? SubnetType.PRIVATE : this.isolatedSubnets.length ? SubnetType.ISOLATED : SubnetType.PUBLIC;
            placement = { ...placement, subnetType: subnetType };
        }
        // Establish which subnet filters are going to be used
        let subnetFilters = (_a = placement.subnetFilters) !== null && _a !== void 0 ? _a : [];
        // Backwards compatibility with existing `availabilityZones` and `onePerAz` functionality
        if (placement.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnetFilters.push(subnet_1.SubnetFilter.availabilityZones(placement.availabilityZones));
        }
        if (!!placement.onePerAz) { // Ensure one per AZ if specified
            subnetFilters.push(subnet_1.SubnetFilter.onePerAz());
        }
        // Overwrite the provided placement filters and remove the availabilityZones and onePerAz properties
        placement = { ...placement, subnetFilters: subnetFilters, availabilityZones: undefined, onePerAz: undefined };
        const { availabilityZones, onePerAz, ...rest } = placement;
        return rest;
    }
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * (experimental) The default tenancy of instances launched into the VPC.
 *
 * @experimental
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    DefaultInstanceTenancy["DEFAULT"] = "default";
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * (experimental) Define an AWS Virtual Private Cloud.
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *    cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *    subnetType: ec2.SubnetType.PRIVATE
 * });
 *
 * for (const subnet of selection.subnets) {
 *    // ...
 * }
 * ```
 *
 * @experimental
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * (experimental) Vpc creates a VPC that spans a whole region.
     *
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     *
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * (experimental) List of public subnets in this VPC.
         *
         * @experimental
         */
        this.publicSubnets = [];
        /**
         * (experimental) List of private subnets in this VPC.
         *
         * @experimental
         */
        this.privateSubnets = [];
        /**
         * (experimental) List of isolated subnets in this VPC.
         *
         * @experimental
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = props.maxAzs !== undefined ? props.maxAzs : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * (experimental) Import an exported VPC.
     *
     * @experimental
     */
    static fromVpcAttributes(scope, id, attrs) {
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * (experimental) Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     *
     * @experimental
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * (deprecated) Adds a new S3 gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * (deprecated) Adds a new DynamoDB gateway endpoint to this VPC.
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaing = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaing);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            core_1.Tags.of(subnet).add(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes });
            core_1.Tags.of(subnet).add(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes });
        });
    }
}
exports.Vpc = Vpc;
/**
 * (experimental) The default CIDR range used when creating VPCs.
 *
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 *
 * @experimental
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * (experimental) The default subnet configuration.
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 *
 * @experimental
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE),
    },
];
/**
 * (experimental) The default subnet configuration if natGateways specified to be 0.
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 *
 * @experimental
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE: return 'Private';
        case SubnetType.ISOLATED: return 'Isolated';
    }
}
/**
 * (experimental) Represents a new VPC subnet resource.
 *
 * @experimental
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * (experimental) Parts of this VPC subnet.
         *
         * @experimental
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.availabilityZone = props.availabilityZone;
        this.ipv4CidrBlock = props.cidrBlock;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.stringValue({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    /**
     * @experimental
     */
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    /**
     * @experimental
     */
    static fromSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    // eslint-disable-next-line @typescript-eslint/no-shadow
    /**
     * (experimental) Import existing subnet from id.
     *
     * @experimental
     */
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * (experimental) Create a default route that points to a passed IGW, with a dependency on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC.
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency.
     * @experimental
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * (experimental) Network ACL associated with this Subnet.
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explic it ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     *
     * @experimental
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * (experimental) Adds an entry to this subnets route table that points to the passed NATGatwayId.
     *
     * @param natGatewayId The ID of the NAT gateway.
     * @experimental
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * (experimental) Adds an entry to this subnets route table.
     *
     * @experimental
     */
    addRoute(id, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    /**
     * (experimental) Associate a Network ACL with this subnet.
     *
     * @experimental
     */
    associateNetworkAcl(id, networkAcl) {
        this._networkAcl = networkAcl;
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
/**
 * (experimental) Type of router used in route.
 *
 * @experimental
 */
var RouterType;
(function (RouterType) {
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    RouterType["GATEWAY"] = "Gateway";
    RouterType["INSTANCE"] = "Instance";
    RouterType["NAT_GATEWAY"] = "NatGateway";
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * (experimental) Represents a public VPC subnet resource.
 *
 * @experimental
 */
class PublicSubnet extends Subnet {
    /**
     * @experimental
     */
    static fromPublicSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * (experimental) Creates a new managed NAT gateway attached to this public subnet.
     *
     * Also adds the EIP for the managed NAT.
     *
     * @returns A ref to the the NAT Gateway ID
     * @experimental
     */
    addNatGateway() {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
/**
 * (experimental) Represents a private VPC subnet resource.
 *
 * @experimental
 */
class PrivateSubnet extends Subnet {
    /**
     * @experimental
     */
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
}
exports.PrivateSubnet = PrivateSubnet;
function ifUndefined(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            const ref = core_1.Token.isUnresolved(attrs.subnetId)
                ? `at '${constructs_1.Node.of(scope).path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            core_1.Annotations.of(this).addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._ipv4CidrBlock = attrs.ipv4CidrBlock;
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error('You cannot reference a Subnet\'s availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()');
        }
        return this._availabilityZone;
    }
    get ipv4CidrBlock() {
        if (!this._ipv4CidrBlock) {
            // tslint:disable-next-line: max-line-length
            throw new Error('You cannot reference an imported Subnet\'s IPv4 CIDR if it was not supplied. Add the ipv4CidrBlock when importing using Subnet.fromSubnetAttributes()');
        }
        return this._ipv4CidrBlock;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_2.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_2.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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