"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlowLog = exports.FlowLogDestination = exports.FlowLogResourceType = exports.FlowLogDestinationType = exports.FlowLogTrafficType = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ec2_generated_1 = require("./ec2.generated");
/**
 * (experimental) The type of VPC traffic to log.
 *
 * @experimental
 */
var FlowLogTrafficType;
(function (FlowLogTrafficType) {
    FlowLogTrafficType["ACCEPT"] = "ACCEPT";
    FlowLogTrafficType["ALL"] = "ALL";
    FlowLogTrafficType["REJECT"] = "REJECT";
})(FlowLogTrafficType = exports.FlowLogTrafficType || (exports.FlowLogTrafficType = {}));
/**
 * (experimental) The available destination types for Flow Logs.
 *
 * @experimental
 */
var FlowLogDestinationType;
(function (FlowLogDestinationType) {
    FlowLogDestinationType["CLOUD_WATCH_LOGS"] = "cloud-watch-logs";
    FlowLogDestinationType["S3"] = "s3";
})(FlowLogDestinationType = exports.FlowLogDestinationType || (exports.FlowLogDestinationType = {}));
/**
 * (experimental) The type of resource to create the flow log for.
 *
 * @experimental
 */
class FlowLogResourceType {
    /**
     * (experimental) The subnet to attach the Flow Log to.
     *
     * @experimental
     */
    static fromSubnet(subnet) {
        return {
            resourceType: 'Subnet',
            resourceId: subnet.subnetId,
        };
    }
    /**
     * (experimental) The VPC to attach the Flow Log to.
     *
     * @experimental
     */
    static fromVpc(vpc) {
        return {
            resourceType: 'VPC',
            resourceId: vpc.vpcId,
        };
    }
    /**
     * (experimental) The Network Interface to attach the Flow Log to.
     *
     * @experimental
     */
    static fromNetworkInterfaceId(id) {
        return {
            resourceType: 'NetworkInterface',
            resourceId: id,
        };
    }
}
exports.FlowLogResourceType = FlowLogResourceType;
/**
 * (experimental) The destination type for the flow log.
 *
 * @experimental
 */
class FlowLogDestination {
    /**
     * (experimental) Use CloudWatch logs as the destination.
     *
     * @experimental
     */
    static toCloudWatchLogs(logGroup, iamRole) {
        return new CloudWatchLogsDestination({
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        });
    }
    /**
     * (experimental) Use S3 as the destination.
     *
     * @experimental
     */
    static toS3(bucket) {
        return new S3Destination({
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket: bucket,
        });
    }
}
exports.FlowLogDestination = FlowLogDestination;
/**
 * @experimental
 */
class S3Destination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let s3Bucket;
        if (this.props.s3Bucket === undefined) {
            s3Bucket = new s3.Bucket(scope, 'Bucket', {
                encryption: s3.BucketEncryption.UNENCRYPTED,
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        else {
            s3Bucket = this.props.s3Bucket;
        }
        return {
            logDestinationType: FlowLogDestinationType.S3,
            s3Bucket,
        };
    }
}
/**
 * @experimental
 */
class CloudWatchLogsDestination extends FlowLogDestination {
    constructor(props) {
        super();
        this.props = props;
    }
    bind(scope, _flowLog) {
        let iamRole;
        let logGroup;
        if (this.props.iamRole === undefined) {
            iamRole = new iam.Role(scope, 'IAMRole', {
                roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
        }
        else {
            iamRole = this.props.iamRole;
        }
        if (this.props.logGroup === undefined) {
            logGroup = new logs.LogGroup(scope, 'LogGroup');
        }
        else {
            logGroup = this.props.logGroup;
        }
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
                'logs:DescribeLogStreams',
            ],
            effect: iam.Effect.ALLOW,
            resources: [logGroup.logGroupArn],
        }));
        iamRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            effect: iam.Effect.ALLOW,
            resources: [iamRole.roleArn],
        }));
        return {
            logDestinationType: FlowLogDestinationType.CLOUD_WATCH_LOGS,
            logGroup,
            iamRole,
        };
    }
}
/**
 * The base class for a Flow Log
 *
 * @experimental
 */
class FlowLogBase extends core_1.Resource {
}
/**
 * (experimental) A VPC flow log.
 *
 * @experimental
 * @resource AWS::EC2::FlowLog
 */
class FlowLog extends FlowLogBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.flowLogName,
        });
        const destination = props.destination || FlowLogDestination.toCloudWatchLogs();
        const destinationConfig = destination.bind(this, this);
        this.logGroup = destinationConfig.logGroup;
        this.bucket = destinationConfig.s3Bucket;
        this.iamRole = destinationConfig.iamRole;
        const flowLog = new ec2_generated_1.CfnFlowLog(this, 'FlowLog', {
            deliverLogsPermissionArn: this.iamRole ? this.iamRole.roleArn : undefined,
            logDestinationType: destinationConfig.logDestinationType,
            logGroupName: this.logGroup ? this.logGroup.logGroupName : undefined,
            resourceId: props.resourceType.resourceId,
            resourceType: props.resourceType.resourceType,
            trafficType: props.trafficType
                ? props.trafficType
                : FlowLogTrafficType.ALL,
            logDestination: this.bucket ? this.bucket.bucketArn : undefined,
        });
        this.flowLogId = flowLog.ref;
    }
    /**
     * (experimental) Import a Flow Log by it's Id.
     *
     * @experimental
     */
    static fromFlowLogId(scope, id, flowLogId) {
        class Import extends FlowLogBase {
            constructor() {
                super(...arguments);
                this.flowLogId = flowLogId;
            }
        }
        return new Import(scope, id);
    }
}
exports.FlowLog = FlowLog;
//# sourceMappingURL=data:application/json;base64,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