"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InterfaceVpcEndpoint = exports.InterfaceVpcEndpointAwsService = exports.InterfaceVpcEndpointService = exports.GatewayVpcEndpoint = exports.GatewayVpcEndpointAwsService = exports.VpcEndpointType = exports.VpcEndpoint = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
/**
 * @experimental
 */
class VpcEndpoint extends core_1.Resource {
    /**
     * (experimental) Adds a statement to the policy document of the VPC endpoint. The statement must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add.
     * @experimental
     */
    addToPolicy(statement) {
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
/**
 * (experimental) The type of VPC endpoint.
 *
 * @experimental
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    VpcEndpointType["INTERFACE"] = "Interface";
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * (experimental) An AWS service for a gateway VPC endpoint.
 *
 * @experimental
 */
class GatewayVpcEndpointAwsService {
    /**
     * @experimental
     */
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
/**
 * @experimental
 */
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
/**
 * @experimental
 */
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * (experimental) A gateway VPC endpoint.
 *
 * @experimental
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * @experimental
     */
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
/**
 * (experimental) A custom-hosted service for an interface VPC endpoint.
 *
 * @experimental
 */
class InterfaceVpcEndpointService {
    /**
     * @experimental
     */
    constructor(name, port) {
        /**
         * (experimental) Whether Private DNS is supported by default.
         *
         * @experimental
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
/**
 * (experimental) An AWS service for an interface VPC endpoint.
 *
 * @experimental
 */
class InterfaceVpcEndpointAwsService {
    /**
     * @experimental
     */
    constructor(name, prefix, port) {
        /**
         * (experimental) Whether Private DNS is supported by default.
         *
         * @experimental
         */
        this.privateDnsDefault = true;
        const region = core_1.Lazy.stringValue({
            produce: (context) => core_1.Stack.of(context.scope).region,
        });
        this.name = `${prefix || 'com.amazonaws'}.${region}.${name}`;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
/**
 * @experimental
 */
InterfaceVpcEndpointAwsService.STEP_FUNCTIONS = new InterfaceVpcEndpointAwsService('states');
/**
 * (experimental) A interface VPC endpoint.
 *
 * @experimental
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        // Determine which subnets to place the endpoint in
        const subnetIds = this.endpointSubnets(props);
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_b = (_a = props.privateDnsEnabled) !== null && _a !== void 0 ? _a : props.service.privateDnsDefault) !== null && _b !== void 0 ? _b : true,
            policyDocument: core_1.Lazy.anyValue({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * (experimental) Imports an existing interface VPC endpoint.
     *
     * @experimental
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        if (!attrs.securityGroups && !attrs.securityGroupId) {
            throw new Error('Either `securityGroups` or `securityGroupId` must be specified.');
        }
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Determine which subnets to place the endpoint in. This is in its own function
     * because there's a lot of code.
     */
    endpointSubnets(props) {
        var _a;
        const lookupSupportedAzs = (_a = props.lookupSupportedAzs) !== null && _a !== void 0 ? _a : false;
        const subnetSelection = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        // If we don't have an account/region, we will not be able to do filtering on AZs since
        // they will be undefined
        const agnosticAcct = core_1.Token.isUnresolved(this.stack.account);
        const agnosticRegion = core_1.Token.isUnresolved(this.stack.region);
        // Some service names, such as AWS service name references, use Tokens to automatically
        // fill in the region
        // If it is an InterfaceVpcEndpointAwsService, then the reference will be resolvable since
        // only references the region
        const isAwsService = core_1.Token.isUnresolved(props.service.name) && props.service instanceof InterfaceVpcEndpointAwsService;
        // Determine what name we pass to the context provider, either the verbatim name
        // or a resolved version if it is an AWS service reference
        let lookupServiceName = props.service.name;
        if (isAwsService && !agnosticRegion) {
            lookupServiceName = core_1.Stack.of(this).resolve(props.service.name);
        }
        else {
            // It's an agnostic service and we don't know how to resolve it.
            // This is ok if the stack is region agnostic and we're not looking up
            // AZs
            lookupServiceName = props.service.name;
        }
        // Check if lookup is impossible and throw an appropriate error
        // Context provider cannot make an AWS call without an account/region
        if ((agnosticAcct || agnosticRegion) && lookupSupportedAzs) {
            throw new Error('Cannot look up VPC endpoint availability zones if account/region are not specified');
        }
        // Context provider doesn't know the name of the service if there is a Token
        // in the name
        const agnosticService = core_1.Token.isUnresolved(lookupServiceName);
        if (agnosticService && lookupSupportedAzs) {
            throw new Error(`Cannot lookup AZs for a service name with a Token: ${props.service.name}`);
        }
        // Here we do the actual lookup for AZs, if told to do so
        let subnets;
        if (lookupSupportedAzs) {
            const availableAZs = this.availableAvailabilityZones(lookupServiceName);
            subnets = subnetSelection.subnets.filter(s => availableAZs.includes(s.availabilityZone));
        }
        else {
            subnets = subnetSelection.subnets;
        }
        const subnetIds = subnets.map(s => s.subnetId);
        return subnetIds;
    }
    availableAvailabilityZones(serviceName) {
        // Here we check what AZs the endpoint service is available in
        // If for whatever reason we can't retrieve the AZs, and no context is set,
        // we will fall back to all AZs
        const availableAZs = core_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
            dummyValue: this.stack.availabilityZones,
            props: { serviceName },
        }).value;
        if (!Array.isArray(availableAZs)) {
            throw new Error(`Discovered AZs for endpoint service ${serviceName} must be an array`);
        }
        return availableAZs;
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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