"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Volume = exports.EbsDeviceVolumeType = exports.synthesizeBlockDeviceMappings = exports.BlockDeviceVolume = void 0;
const crypto = require("crypto");
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const aws_kms_1 = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const constructs_1 = require("constructs");
const ec2_generated_1 = require("./ec2.generated");
/**
 * (experimental) Describes a block device mapping for an EC2 instance or Auto Scaling group.
 *
 * @experimental
 */
class BlockDeviceVolume {
    /**
     * @param ebsDevice EBS device info.
     * @param virtualName Virtual device name.
     * @experimental
     */
    constructor(ebsDevice, virtualName) {
        this.ebsDevice = ebsDevice;
        this.virtualName = virtualName;
    }
    /**
     * (experimental) Creates a new Elastic Block Storage device.
     *
     * @param volumeSize The volume size, in Gibibytes (GiB).
     * @param options additional device options.
     * @experimental
     */
    static ebs(volumeSize, options = {}) {
        return new this({ ...options, volumeSize });
    }
    /**
     * (experimental) Creates a new Elastic Block Storage device from an existing snapshot.
     *
     * @param snapshotId The snapshot ID of the volume to use.
     * @param options additional device options.
     * @experimental
     */
    static ebsFromSnapshot(snapshotId, options = {}) {
        return new this({ ...options, snapshotId });
    }
    /**
     * (experimental) Creates a virtual, ephemeral device.
     *
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index.
     * @experimental
     */
    static ephemeral(volumeIndex) {
        if (volumeIndex < 0) {
            throw new Error(`volumeIndex must be a number starting from 0, got "${volumeIndex}"`);
        }
        return new this(undefined, `ephemeral${volumeIndex}`);
    }
}
exports.BlockDeviceVolume = BlockDeviceVolume;
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName,
            ebs,
            virtualName,
            noDevice: mappingEnabled === false ? {} : undefined,
        };
    });
}
exports.synthesizeBlockDeviceMappings = synthesizeBlockDeviceMappings;
/**
 * (experimental) Supported EBS volume types for blockDevices.
 *
 * @experimental
 */
var EbsDeviceVolumeType;
(function (EbsDeviceVolumeType) {
    EbsDeviceVolumeType["STANDARD"] = "standard";
    EbsDeviceVolumeType["IO1"] = "io1";
    EbsDeviceVolumeType["GP2"] = "gp2";
    EbsDeviceVolumeType["ST1"] = "st1";
    EbsDeviceVolumeType["SC1"] = "sc1";
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    EbsDeviceVolumeType["GENERAL_PURPOSE_SSD"] = "gp2";
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    EbsDeviceVolumeType["PROVISIONED_IOPS_SSD"] = "io1";
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    EbsDeviceVolumeType["THROUGHPUT_OPTIMIZED_HDD"] = "st1";
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    EbsDeviceVolumeType["COLD_HDD"] = "sc1";
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    EbsDeviceVolumeType["MAGNETIC"] = "standard";
})(EbsDeviceVolumeType = exports.EbsDeviceVolumeType || (exports.EbsDeviceVolumeType = {}));
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
class VolumeBase extends core_1.Resource {
    /**
     * (experimental) Grants permission to attach this Volume to an instance.
     *
     * CAUTION: Granting an instance permission to attach to itself using this method will lead to
     * an unresolvable circular reference between the instance role and the instance.
     * Use {@link IVolume.grantAttachVolumeToSelf} to grant an instance permission to attach this
     * volume to itself.
     *
     * @experimental
     */
    grantAttachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:AttachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        if (this.encryptionKey) {
            // When attaching a volume, the EC2 Service will need to grant to itself permission
            // to be able to decrypt the encryption key. We restrict the CreateGrant for principle
            // of least privilege, in accordance with best practices.
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-permissions
            const kmsGrant = this.encryptionKey.grant(grantee, 'kms:CreateGrant');
            kmsGrant.principalStatement.addConditions({
                Bool: { 'kms:GrantIsForAWSResource': true },
                StringEquals: {
                    'kms:ViaService': `ec2.${core_1.Stack.of(this).region}.amazonaws.com`,
                    'kms:GrantConstraintType': 'EncryptionContextSubset',
                },
            });
        }
        return result;
    }
    /**
     * (experimental) Grants permission to attach the Volume by a ResourceTag condition.
     *
     * If you are looking to
     * grant an Instance, AutoScalingGroup, EC2-Fleet, SpotFleet, ECS host, etc the ability to attach
     * this volume to **itself** then this is the method you want to use.
     *
     * This is implemented by adding a Tag with key `VolumeGrantAttach-<suffix>` to the given
     * constructs and this Volume, and then conditioning the Grant such that the grantee is only
     * given the ability to AttachVolume if both the Volume and the destination Instance have that
     * tag applied to them.
     *
     * @experimental
     */
    grantAttachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantAttach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantAttachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    /**
     * (experimental) Grants permission to detach this Volume from an instance CAUTION: Granting an instance permission to detach from itself using this method will lead to an unresolvable circular reference between the instance role and the instance.
     *
     * Use {@link IVolume.grantDetachVolumeFromSelf} to grant an instance permission to detach this
     * volume from itself.
     *
     * @experimental
     */
    grantDetachVolume(grantee, instances) {
        const result = aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['ec2:DetachVolume'],
            resourceArns: this.collectGrantResourceArns(instances),
        });
        // Note: No encryption key permissions are required to detach an encrypted volume.
        return result;
    }
    /**
     * (experimental) Grants permission to detach the Volume by a ResourceTag condition.
     *
     * This is implemented via the same mechanism as {@link IVolume.grantAttachVolumeByResourceTag},
     * and is subject to the same conditions.
     *
     * @experimental
     */
    grantDetachVolumeByResourceTag(grantee, constructs, tagKeySuffix) {
        const tagValue = this.calculateResourceTagValue([this, ...constructs]);
        const tagKey = `VolumeGrantDetach-${tagKeySuffix !== null && tagKeySuffix !== void 0 ? tagKeySuffix : tagValue.slice(0, 10).toUpperCase()}`;
        const grantCondition = {};
        grantCondition[`ec2:ResourceTag/${tagKey}`] = tagValue;
        const result = this.grantDetachVolume(grantee);
        result.principalStatement.addCondition('ForAnyValue:StringEquals', grantCondition);
        // The ResourceTag condition requires that all resources involved in the operation have
        // the given tag, so we tag this and all constructs given.
        core_1.Tags.of(this).add(tagKey, tagValue);
        constructs.forEach(construct => core_1.Tags.of(construct).add(tagKey, tagValue));
        return result;
    }
    collectGrantResourceArns(instances) {
        const stack = core_1.Stack.of(this);
        const resourceArns = [
            `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:volume/${this.volumeId}`,
        ];
        const instanceArnPrefix = `arn:${stack.partition}:ec2:${stack.region}:${stack.account}:instance`;
        if (instances) {
            instances.forEach(instance => resourceArns.push(`${instanceArnPrefix}/${instance === null || instance === void 0 ? void 0 : instance.instanceId}`));
        }
        else {
            resourceArns.push(`${instanceArnPrefix}/*`);
        }
        return resourceArns;
    }
    calculateResourceTagValue(constructs) {
        const md5 = crypto.createHash('md5');
        constructs.forEach(construct => md5.update(constructs_1.Node.of(construct).uniqueId));
        return md5.digest('hex');
    }
}
/**
 * (experimental) Creates a new EBS Volume in AWS EC2.
 *
 * @experimental
 */
class Volume extends VolumeBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id, {
            physicalName: props.volumeName,
        });
        this.validateProps(props);
        const resource = new ec2_generated_1.CfnVolume(this, 'Resource', {
            availabilityZone: props.availabilityZone,
            autoEnableIo: props.autoEnableIo,
            encrypted: props.encrypted,
            kmsKeyId: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
            iops: props.iops,
            multiAttachEnabled: (_b = props.enableMultiAttach) !== null && _b !== void 0 ? _b : false,
            size: (_c = props.size) === null || _c === void 0 ? void 0 : _c.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }),
            snapshotId: props.snapshotId,
            volumeType: (_d = props.volumeType) !== null && _d !== void 0 ? _d : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD,
        });
        this.volumeId = resource.ref;
        this.availabilityZone = props.availabilityZone;
        this.encryptionKey = props.encryptionKey;
        if (this.encryptionKey) {
            // Per: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#ebs-encryption-requirements
            const principal = new aws_kms_1.ViaServicePrincipal(`ec2.${core_1.Stack.of(this).region}.amazonaws.com`, new aws_iam_1.AccountRootPrincipal()).withConditions({
                StringEquals: {
                    'kms:CallerAccount': core_1.Stack.of(this).account,
                },
            });
            const grant = this.encryptionKey.grant(principal, 
            // Describe & Generate are required to be able to create the CMK-encrypted Volume.
            'kms:DescribeKey', 'kms:GenerateDataKeyWithoutPlainText');
            if (props.snapshotId) {
                // ReEncrypt is required for when re-encrypting from an encrypted snapshot.
                (_e = grant.principalStatement) === null || _e === void 0 ? void 0 : _e.addActions('kms:ReEncrypt*');
            }
        }
    }
    /**
     * (experimental) Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume.
     * @experimental
     */
    static fromVolumeAttributes(scope, id, attrs) {
        class Import extends VolumeBase {
            constructor() {
                super(...arguments);
                this.volumeId = attrs.volumeId;
                this.availabilityZone = attrs.availabilityZone;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        // Check that the provided volumeId looks like it could be valid.
        if (!core_1.Token.isUnresolved(attrs.volumeId) && !/^vol-[0-9a-fA-F]+$/.test(attrs.volumeId)) {
            throw new Error('`volumeId` does not match expected pattern. Expected `vol-<hexadecmial value>` (ex: `vol-05abe246af`) or a Token');
        }
        return new Import(scope, id);
    }
    /**
     * @experimental
     */
    validateProps(props) {
        var _a;
        if (!(props.size || props.snapshotId)) {
            throw new Error('Must provide at least one of `size` or `snapshotId`');
        }
        if (props.snapshotId && !core_1.Token.isUnresolved(props.snapshotId) && !/^snap-[0-9a-fA-F]+$/.test(props.snapshotId)) {
            throw new Error('`snapshotId` does match expected pattern. Expected `snap-<hexadecmial value>` (ex: `snap-05abe246af`) or Token');
        }
        if (props.encryptionKey && !props.encrypted) {
            throw new Error('`encrypted` must be true when providing an `encryptionKey`.');
        }
        if (props.iops) {
            if (props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
                throw new Error('`iops` may only be specified if the `volumeType` is `PROVISIONED_IOPS_SSD`/`IO1`');
            }
            if (props.iops < 100 || props.iops > 64000) {
                throw new Error('`iops` must be in the range 100 to 64,000, inclusive.');
            }
            if (props.size && (props.iops > 50 * props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL }))) {
                throw new Error('`iops` has a maximum ratio of 50 IOPS/GiB.');
            }
        }
        if (props.enableMultiAttach && props.volumeType !== EbsDeviceVolumeType.PROVISIONED_IOPS_SSD) {
            throw new Error('multi-attach is supported exclusively on `PROVISIONED_IOPS_SSD` volumes.');
        }
        if (props.size) {
            const size = props.size.toGibibytes({ rounding: core_1.SizeRoundingBehavior.FAIL });
            // Enforce maximum volume size:
            // https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics
            const sizeRanges = {};
            sizeRanges[EbsDeviceVolumeType.GENERAL_PURPOSE_SSD] = { Min: 1, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.PROVISIONED_IOPS_SSD] = { Min: 4, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.THROUGHPUT_OPTIMIZED_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.COLD_HDD] = { Min: 500, Max: 16000 };
            sizeRanges[EbsDeviceVolumeType.MAGNETIC] = { Min: 1, Max: 1000 };
            const volumeType = (_a = props.volumeType) !== null && _a !== void 0 ? _a : EbsDeviceVolumeType.GENERAL_PURPOSE_SSD;
            const { Min, Max } = sizeRanges[volumeType];
            if (size < Min || size > Max) {
                throw new Error(`\`${volumeType}\` volumes must be between ${Min} GiB and ${Max} GiB in size.`);
            }
        }
    }
}
exports.Volume = Volume;
//# sourceMappingURL=data:application/json;base64,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