import { Grant, IGrantable } from "../../aws-iam";
import { IKey } from "../../aws-kms";
import { IResource, Resource, Size } from "../../core";
import { Construct } from 'constructs';
import { CfnInstance } from './ec2.generated';
import { IInstance } from './instance';
/**
 * (experimental) Block device.
 *
 * @experimental
 */
export interface BlockDevice {
    /**
     * (experimental) The device name exposed to the EC2 instance.
     *
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/device_naming.html
     * @experimental
     * @example
     *
     * '/dev/sdh', 'xvdh'
     */
    readonly deviceName: string;
    /**
     * (experimental) Defines the block device volume, to be either an Amazon EBS volume or an ephemeral instance store volume.
     *
     * @experimental
     * @example
     *
     * BlockDeviceVolume.ebs(15), BlockDeviceVolume.ephemeral(0)
     */
    readonly volume: BlockDeviceVolume;
    /**
     * (experimental) If false, the device mapping will be suppressed.
     *
     * If set to false for the root device, the instance might fail the Amazon EC2 health check.
     * Amazon EC2 Auto Scaling launches a replacement instance if the instance fails the health check.
     *
     * @default true - device mapping is left untouched
     * @experimental
     */
    readonly mappingEnabled?: boolean;
}
/**
 * (experimental) Base block device options for an EBS volume.
 *
 * @experimental
 */
export interface EbsDeviceOptionsBase {
    /**
     * (experimental) Indicates whether to delete the volume when the instance is terminated.
     *
     * @default - true for Amazon EC2 Auto Scaling, false otherwise (e.g. EBS)
     * @experimental
     */
    readonly deleteOnTermination?: boolean;
    /**
     * (experimental) The number of I/O operations per second (IOPS) to provision for the volume.
     *
     * Must only be set for {@link volumeType}: {@link EbsDeviceVolumeType.IO1}
     *
     * The maximum ratio of IOPS to volume size (in GiB) is 50:1, so for 5,000 provisioned IOPS,
     * you need at least 100 GiB storage on the volume.
     *
     * @default - none, required for {@link EbsDeviceVolumeType.IO1}
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
     * @experimental
     */
    readonly iops?: number;
    /**
     * (experimental) The EBS volume type.
     *
     * @default {@link EbsDeviceVolumeType.GP2}
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html
     * @experimental
     */
    readonly volumeType?: EbsDeviceVolumeType;
}
/**
 * (experimental) Block device options for an EBS volume.
 *
 * @experimental
 */
export interface EbsDeviceOptions extends EbsDeviceOptionsBase {
    /**
     * (experimental) Specifies whether the EBS volume is encrypted.
     *
     * Encrypted EBS volumes can only be attached to instances that support Amazon EBS encryption
     *
     * @default false
     * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances
     * @experimental
     */
    readonly encrypted?: boolean;
}
/**
 * (experimental) Block device options for an EBS volume created from a snapshot.
 *
 * @experimental
 */
export interface EbsDeviceSnapshotOptions extends EbsDeviceOptionsBase {
    /**
     * (experimental) The volume size, in Gibibytes (GiB).
     *
     * If you specify volumeSize, it must be equal or greater than the size of the snapshot.
     *
     * @default - The snapshot size
     * @experimental
     */
    readonly volumeSize?: number;
}
/**
 * (experimental) Properties of an EBS block device.
 *
 * @experimental
 */
export interface EbsDeviceProps extends EbsDeviceSnapshotOptions {
    /**
     * (experimental) The snapshot ID of the volume to use.
     *
     * @default - No snapshot will be used
     * @experimental
     */
    readonly snapshotId?: string;
}
/**
 * (experimental) Describes a block device mapping for an EC2 instance or Auto Scaling group.
 *
 * @experimental
 */
export declare class BlockDeviceVolume {
    readonly ebsDevice?: EbsDeviceProps | undefined;
    readonly virtualName?: string | undefined;
    /**
     * (experimental) Creates a new Elastic Block Storage device.
     *
     * @param volumeSize The volume size, in Gibibytes (GiB).
     * @param options additional device options.
     * @experimental
     */
    static ebs(volumeSize: number, options?: EbsDeviceOptions): BlockDeviceVolume;
    /**
     * (experimental) Creates a new Elastic Block Storage device from an existing snapshot.
     *
     * @param snapshotId The snapshot ID of the volume to use.
     * @param options additional device options.
     * @experimental
     */
    static ebsFromSnapshot(snapshotId: string, options?: EbsDeviceSnapshotOptions): BlockDeviceVolume;
    /**
     * (experimental) Creates a virtual, ephemeral device.
     *
     * The name will be in the form ephemeral{volumeIndex}.
     *
     * @param volumeIndex the volume index.
     * @experimental
     */
    static ephemeral(volumeIndex: number): BlockDeviceVolume;
    /**
     * @param ebsDevice EBS device info.
     * @param virtualName Virtual device name.
     * @experimental
     */
    protected constructor(ebsDevice?: EbsDeviceProps | undefined, virtualName?: string | undefined);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
export declare function synthesizeBlockDeviceMappings(construct: Construct, blockDevices: BlockDevice[]): CfnInstance.BlockDeviceMappingProperty[];
/**
 * (experimental) Supported EBS volume types for blockDevices.
 *
 * @experimental
 */
export declare enum EbsDeviceVolumeType {
    /**
     * (experimental) Magnetic.
     *
     * @experimental
     */
    STANDARD = "standard",
    /**
     * (experimental) Provisioned IOPS SSD.
     *
     * @experimental
     */
    IO1 = "io1",
    /**
     * (experimental) General Purpose SSD.
     *
     * @experimental
     */
    GP2 = "gp2",
    /**
     * (experimental) Throughput Optimized HDD.
     *
     * @experimental
     */
    ST1 = "st1",
    /**
     * (experimental) Cold HDD.
     *
     * @experimental
     */
    SC1 = "sc1",
    /**
     * General purpose SSD volume that balances price and performance for a wide variety of workloads.
     */
    GENERAL_PURPOSE_SSD = "gp2",
    /**
     * Highest-performance SSD volume for mission-critical low-latency or high-throughput workloads.
     */
    PROVISIONED_IOPS_SSD = "io1",
    /**
     * Low-cost HDD volume designed for frequently accessed, throughput-intensive workloads.
     */
    THROUGHPUT_OPTIMIZED_HDD = "st1",
    /**
     * Lowest cost HDD volume designed for less frequently accessed workloads.
     */
    COLD_HDD = "sc1",
    /**
     * Magnetic volumes are backed by magnetic drives and are suited for workloads where data is accessed infrequently, and scenarios where low-cost
     * storage for small volume sizes is important.
     */
    MAGNETIC = "standard"
}
/**
 * (experimental) An EBS Volume in AWS EC2.
 *
 * @experimental
 */
export interface IVolume extends IResource {
    /**
     * (experimental) The EBS Volume's ID.
     *
     * @experimental
     * @attribute true
     */
    readonly volumeId: string;
    /**
     * (experimental) The availability zone that the EBS Volume is contained within (ex: us-west-2a).
     *
     * @experimental
     */
    readonly availabilityZone: string;
    /**
     * (experimental) The customer-managed encryption key that is used to encrypt the Volume.
     *
     * @experimental
     * @attribute true
     */
    readonly encryptionKey?: IKey;
    /**
     * (experimental) Grants permission to attach this Volume to an instance.
     *
     * CAUTION: Granting an instance permission to attach to itself using this method will lead to
     * an unresolvable circular reference between the instance role and the instance.
     * Use {@link IVolume.grantAttachVolumeToSelf} to grant an instance permission to attach this
     * volume to itself.
     *
     * @param grantee the principal being granted permission.
     * @param instances the instances to which permission is being granted to attach this volume to.
     * @experimental
     */
    grantAttachVolume(grantee: IGrantable, instances?: IInstance[]): Grant;
    /**
     * (experimental) Grants permission to attach the Volume by a ResourceTag condition.
     *
     * If you are looking to
     * grant an Instance, AutoScalingGroup, EC2-Fleet, SpotFleet, ECS host, etc the ability to attach
     * this volume to **itself** then this is the method you want to use.
     *
     * This is implemented by adding a Tag with key `VolumeGrantAttach-<suffix>` to the given
     * constructs and this Volume, and then conditioning the Grant such that the grantee is only
     * given the ability to AttachVolume if both the Volume and the destination Instance have that
     * tag applied to them.
     *
     * @param grantee the principal being granted permission.
     * @param constructs The list of constructs that will have the generated resource tag applied to them.
     * @param tagKeySuffix A suffix to use on the generated Tag key in place of the generated hash value.
     * @experimental
     */
    grantAttachVolumeByResourceTag(grantee: IGrantable, constructs: Construct[], tagKeySuffix?: string): Grant;
    /**
     * (experimental) Grants permission to detach this Volume from an instance CAUTION: Granting an instance permission to detach from itself using this method will lead to an unresolvable circular reference between the instance role and the instance.
     *
     * Use {@link IVolume.grantDetachVolumeFromSelf} to grant an instance permission to detach this
     * volume from itself.
     *
     * @param grantee the principal being granted permission.
     * @param instances the instances to which permission is being granted to detach this volume from.
     * @experimental
     */
    grantDetachVolume(grantee: IGrantable, instances?: IInstance[]): Grant;
    /**
     * (experimental) Grants permission to detach the Volume by a ResourceTag condition.
     *
     * This is implemented via the same mechanism as {@link IVolume.grantAttachVolumeByResourceTag},
     * and is subject to the same conditions.
     *
     * @param grantee the principal being granted permission.
     * @param constructs The list of constructs that will have the generated resource tag applied to them.
     * @param tagKeySuffix A suffix to use on the generated Tag key in place of the generated hash value.
     * @experimental
     */
    grantDetachVolumeByResourceTag(grantee: IGrantable, constructs: Construct[], tagKeySuffix?: string): Grant;
}
/**
 * (experimental) Properties of an EBS Volume.
 *
 * @experimental
 */
export interface VolumeProps {
    /**
     * (experimental) The value of the physicalName property of this resource.
     *
     * @default The physical name will be allocated by CloudFormation at deployment time
     * @experimental
     */
    readonly volumeName?: string;
    /**
     * (experimental) The Availability Zone in which to create the volume.
     *
     * @experimental
     */
    readonly availabilityZone: string;
    /**
     * (experimental) The size of the volume, in GiBs.
     *
     * You must specify either a snapshot ID or a volume size.
     * See {@link https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#ebs-volume-characteristics|Volume Characteristics}
     * for details on the allowable size for each type of volume.
     *
     * @default If you're creating the volume from a snapshot and don't specify a volume size, the default is the snapshot size.
     * @experimental
     */
    readonly size?: Size;
    /**
     * (experimental) The snapshot from which to create the volume.
     *
     * You must specify either a snapshot ID or a volume size.
     *
     * @default The EBS volume is not created from a snapshot.
     * @experimental
     */
    readonly snapshotId?: string;
    /**
     * (experimental) Indicates whether Amazon EBS Multi-Attach is enabled.
     *
     * See {@link https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volumes-multi.html#considerations|Considerations and limitations}
     * for the constraints of multi-attach.
     *
     * @default false
     * @experimental
     */
    readonly enableMultiAttach?: boolean;
    /**
     * (experimental) Specifies whether the volume should be encrypted.
     *
     * The effect of setting the encryption state to true depends on the volume origin
     * (new or from a snapshot), starting encryption state, ownership, and whether encryption by default is enabled. For more information,
     * see {@link https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-by-default|Encryption by Default}
     * in the Amazon Elastic Compute Cloud User Guide.
     *
     * Encrypted Amazon EBS volumes must be attached to instances that support Amazon EBS encryption. For more information, see
     * {@link https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances|Supported Instance Types.}
     *
     * @default false
     * @experimental
     */
    readonly encrypted?: boolean;
    /**
     * (experimental) The customer-managed encryption key that is used to encrypt the Volume.
     *
     * The encrypted property must
     * be true if this is provided.
     *
     * Note: If using an {@link aws-kms.IKey} created from a {@link aws-kms.Key.fromKeyArn()} here,
     * then the KMS key **must** have the following in its Key policy; otherwise, the Volume
     * will fail to create.
     *
     *      {
     *        "Effect": "Allow",
     *        "Principal": { "AWS": "<arn for your account-user> ex: arn:aws:iam::00000000000:root" },
     *        "Resource": "*",
     *        "Action": [
     *          "kms:DescribeKey",
     *          "kms:GenerateDataKeyWithoutPlainText",
     *        ],
     *        "Condition": {
     *          "StringEquals": {
     *            "kms:ViaService": "ec2.<Region>.amazonaws.com", (eg: ec2.us-east-1.amazonaws.com)
     *            "kms:CallerAccount": "0000000000" (your account ID)
     *          }
     *        }
     *      }
     *
     * @default The default KMS key for the account, region, and EC2 service is used.
     * @experimental
     */
    readonly encryptionKey?: IKey;
    /**
     * (experimental) Indicates whether the volume is auto-enabled for I/O operations.
     *
     * By default, Amazon EBS disables I/O to the volume from attached EC2
     * instances when it determines that a volume's data is potentially inconsistent. If the consistency of the volume is not a concern, and
     * you prefer that the volume be made available immediately if it's impaired, you can configure the volume to automatically enable I/O.
     *
     * @default false
     * @experimental
     */
    readonly autoEnableIo?: boolean;
    /**
     * (experimental) The type of the volume;
     *
     * what type of storage to use to form the EBS Volume.
     *
     * @default {@link EbsDeviceVolumeType.GENERAL_PURPOSE_SSD}
     * @experimental
     */
    readonly volumeType?: EbsDeviceVolumeType;
    /**
     * (experimental) The number of I/O operations per second (IOPS) to provision for the volume, with a maximum ratio of 50 IOPS/GiB.
     *
     * See {@link https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-volume-types.html#EBSVolumeTypes_piops|Provisioned IOPS SSD (io1) volumes}
     * for more information.
     *
     * This parameter is valid only for PROVISIONED_IOPS_SSD volumes.
     *
     * @default None -- Required for {@link EbsDeviceVolumeType.PROVISIONED_IOPS_SSD}
     * @experimental
     */
    readonly iops?: number;
}
/**
 * (experimental) Attributes required to import an existing EBS Volume into the Stack.
 *
 * @experimental
 */
export interface VolumeAttributes {
    /**
     * (experimental) The EBS Volume's ID.
     *
     * @experimental
     */
    readonly volumeId: string;
    /**
     * (experimental) The availability zone that the EBS Volume is contained within (ex: us-west-2a).
     *
     * @experimental
     */
    readonly availabilityZone: string;
    /**
     * (experimental) The customer-managed encryption key that is used to encrypt the Volume.
     *
     * @default None -- The EBS Volume is not using a customer-managed KMS key for encryption.
     * @experimental
     */
    readonly encryptionKey?: IKey;
}
/**
 * Common behavior of Volumes. Users should not use this class directly, and instead use ``Volume``.
 */
declare abstract class VolumeBase extends Resource implements IVolume {
    abstract readonly volumeId: string;
    abstract readonly availabilityZone: string;
    abstract readonly encryptionKey?: IKey;
    /**
     * (experimental) Grants permission to attach this Volume to an instance.
     *
     * CAUTION: Granting an instance permission to attach to itself using this method will lead to
     * an unresolvable circular reference between the instance role and the instance.
     * Use {@link IVolume.grantAttachVolumeToSelf} to grant an instance permission to attach this
     * volume to itself.
     *
     * @experimental
     */
    grantAttachVolume(grantee: IGrantable, instances?: IInstance[]): Grant;
    /**
     * (experimental) Grants permission to attach the Volume by a ResourceTag condition.
     *
     * If you are looking to
     * grant an Instance, AutoScalingGroup, EC2-Fleet, SpotFleet, ECS host, etc the ability to attach
     * this volume to **itself** then this is the method you want to use.
     *
     * This is implemented by adding a Tag with key `VolumeGrantAttach-<suffix>` to the given
     * constructs and this Volume, and then conditioning the Grant such that the grantee is only
     * given the ability to AttachVolume if both the Volume and the destination Instance have that
     * tag applied to them.
     *
     * @experimental
     */
    grantAttachVolumeByResourceTag(grantee: IGrantable, constructs: Construct[], tagKeySuffix?: string): Grant;
    /**
     * (experimental) Grants permission to detach this Volume from an instance CAUTION: Granting an instance permission to detach from itself using this method will lead to an unresolvable circular reference between the instance role and the instance.
     *
     * Use {@link IVolume.grantDetachVolumeFromSelf} to grant an instance permission to detach this
     * volume from itself.
     *
     * @experimental
     */
    grantDetachVolume(grantee: IGrantable, instances?: IInstance[]): Grant;
    /**
     * (experimental) Grants permission to detach the Volume by a ResourceTag condition.
     *
     * This is implemented via the same mechanism as {@link IVolume.grantAttachVolumeByResourceTag},
     * and is subject to the same conditions.
     *
     * @experimental
     */
    grantDetachVolumeByResourceTag(grantee: IGrantable, constructs: Construct[], tagKeySuffix?: string): Grant;
    private collectGrantResourceArns;
    private calculateResourceTagValue;
}
/**
 * (experimental) Creates a new EBS Volume in AWS EC2.
 *
 * @experimental
 */
export declare class Volume extends VolumeBase {
    /**
     * (experimental) Import an existing EBS Volume into the Stack.
     *
     * @param scope the scope of the import.
     * @param id the ID of the imported Volume in the construct tree.
     * @param attrs the attributes of the imported Volume.
     * @experimental
     */
    static fromVolumeAttributes(scope: Construct, id: string, attrs: VolumeAttributes): IVolume;
    /**
     * (experimental) The EBS Volume's ID.
     *
     * @experimental
     */
    readonly volumeId: string;
    /**
     * (experimental) The availability zone that the EBS Volume is contained within (ex: us-west-2a).
     *
     * @experimental
     */
    readonly availabilityZone: string;
    /**
     * (experimental) The customer-managed encryption key that is used to encrypt the Volume.
     *
     * @experimental
     */
    readonly encryptionKey?: IKey;
    /**
     * @experimental
     */
    constructor(scope: Construct, id: string, props: VolumeProps);
    /**
     * @experimental
     */
    protected validateProps(props: VolumeProps): void;
}
export {};
