"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Peer = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const connections_1 = require("./connections");
/**
 * (experimental) Peer object factories (to be used in Security Group management).
 *
 * The static methods on this object can be used to create peer objects
 * which represent a connection partner in Security Group rules.
 *
 * Use this object if you need to represent connection partners using plain IP
 * addresses, or a prefix list ID.
 *
 * If you want to address a connection partner by Security Group, you can just
 * use the Security Group (or the construct that contains a Security Group)
 * directly, as it already implements `IPeer`.
 *
 * @experimental
 */
class Peer {
    /**
     * (experimental) Create an IPv4 peer from a CIDR.
     *
     * @experimental
     */
    static ipv4(cidrIp) {
        return new CidrIPv4(cidrIp);
    }
    /**
     * (experimental) Any IPv4 address.
     *
     * @experimental
     */
    static anyIpv4() {
        return new AnyIPv4();
    }
    /**
     * (experimental) Create an IPv6 peer from a CIDR.
     *
     * @experimental
     */
    static ipv6(cidrIp) {
        return new CidrIPv6(cidrIp);
    }
    /**
     * (experimental) Any IPv6 address.
     *
     * @experimental
     */
    static anyIpv6() {
        return new AnyIPv6();
    }
    /**
     * (experimental) A prefix list.
     *
     * @experimental
     */
    static prefixList(prefixListId) {
        return new PrefixList(prefixListId);
    }
    /**
     * @experimental
     */
    constructor() {
    }
}
exports.Peer = Peer;
/**
 * A connection to and from a given IP range
 */
class CidrIPv4 {
    constructor(cidrIp) {
        this.cidrIp = cidrIp;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIp)) {
            const cidrMatch = cidrIp.match(/^(\d{1,3}\.){3}\d{1,3}(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv4 CIDR: "${cidrIp}"`);
            }
            if (!cidrMatch[2]) {
                throw new Error(`CIDR mask is missing in IPv4: "${cidrIp}". Did you mean "${cidrIp}/32"?`);
            }
        }
        this.uniqueId = cidrIp;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIp: this.cidrIp };
    }
}
/**
 * Any IPv4 address
 */
class AnyIPv4 extends CidrIPv4 {
    constructor() {
        super('0.0.0.0/0');
    }
}
/**
 * A connection to a from a given IPv6 range
 */
class CidrIPv6 {
    constructor(cidrIpv6) {
        this.cidrIpv6 = cidrIpv6;
        this.canInlineRule = true;
        this.connections = new connections_1.Connections({ peer: this });
        if (!core_1.Token.isUnresolved(cidrIpv6)) {
            const cidrMatch = cidrIpv6.match(/^([\da-f]{0,4}:){2,7}([\da-f]{0,4})?(\/\d+)?$/);
            if (!cidrMatch) {
                throw new Error(`Invalid IPv6 CIDR: "${cidrIpv6}"`);
            }
            if (!cidrMatch[3]) {
                throw new Error(`CIDR mask is missing in IPv6: "${cidrIpv6}". Did you mean "${cidrIpv6}/128"?`);
            }
        }
        this.uniqueId = cidrIpv6;
    }
    /**
     * Produce the ingress rule JSON for the given connection
     */
    toIngressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
    /**
     * Produce the egress rule JSON for the given connection
     */
    toEgressRuleConfig() {
        return { cidrIpv6: this.cidrIpv6 };
    }
}
/**
 * Any IPv6 address
 */
class AnyIPv6 extends CidrIPv6 {
    constructor() {
        super('::/0');
    }
}
/**
 * A prefix list
 *
 * Prefix lists are used to allow traffic to VPC-local service endpoints.
 *
 * For more information, see this page:
 *
 * https://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-endpoints.html
 */
class PrefixList {
    constructor(prefixListId) {
        this.prefixListId = prefixListId;
        this.canInlineRule = false;
        this.connections = new connections_1.Connections({ peer: this });
        this.uniqueId = prefixListId;
    }
    toIngressRuleConfig() {
        return { sourcePrefixListId: this.prefixListId };
    }
    toEgressRuleConfig() {
        return { destinationPrefixListId: this.prefixListId };
    }
}
//# sourceMappingURL=data:application/json;base64,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