"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Connections = void 0;
const peer_1 = require("./peer");
/**
 * (experimental) Manage the allowed network connections for constructs with Security Groups.
 *
 * Security Groups can be thought of as a firewall for network-connected
 * devices. This class makes it easy to allow network connections to and
 * from security groups, and between security groups individually. When
 * establishing connectivity between security groups, it will automatically
 * add rules in both security groups
 *
 * This object can manage one or more security groups.
 *
 * @experimental
 */
class Connections {
    /**
     * @experimental
     */
    constructor(props = {}) {
        /**
         * Underlying securityGroup for this Connections object, if present
         *
         * May be empty if this Connections object is not managing a SecurityGroup,
         * but simply representing a Connectable peer.
         */
        this._securityGroups = new ReactiveList();
        /**
         * The rule that defines how to represent this peer in a security group
         */
        this._securityGroupRules = new ReactiveList();
        /**
         * When doing bidirectional grants between Connections, make sure we don't recursive infinitely
         */
        this.skip = false;
        /**
         * When doing bidirectional grants between Security Groups in different stacks, put the rule on the other SG
         */
        this.remoteRule = false;
        this.connections = this;
        this._securityGroups.push(...(props.securityGroups || []));
        this._securityGroupRules.push(...this._securityGroups.asArray());
        if (props.peer) {
            this._securityGroupRules.push(props.peer);
        }
        this.defaultPort = props.defaultPort;
    }
    /**
     * @experimental
     */
    get securityGroups() {
        return this._securityGroups.asArray();
    }
    /**
     * (experimental) Add a security group to the list of security groups managed by this object.
     *
     * @experimental
     */
    addSecurityGroup(...securityGroups) {
        for (const securityGroup of securityGroups) {
            this._securityGroups.push(securityGroup);
            this._securityGroupRules.push(securityGroup);
        }
    }
    /**
     * (experimental) Allow connections to the peer on the given port.
     *
     * @experimental
     */
    allowTo(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addEgressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowFrom(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * (experimental) Allow connections from the peer on the given port.
     *
     * @experimental
     */
    allowFrom(other, portRange, description) {
        if (this.skip) {
            return;
        }
        const remoteRule = this.remoteRule; // Capture current value into local for callback to close over
        this._securityGroups.forEachAndForever(securityGroup => {
            other.connections._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description, remoteRule);
            });
        });
        this.skip = true;
        other.connections.remoteRule = true;
        try {
            other.connections.allowTo(this, portRange, description);
        }
        finally {
            this.skip = false;
            other.connections.remoteRule = false;
        }
    }
    /**
     * (experimental) Allow hosts inside the security group to connect to each other on the given port.
     *
     * @experimental
     */
    allowInternally(portRange, description) {
        this._securityGroups.forEachAndForever(securityGroup => {
            this._securityGroupRules.forEachAndForever(rule => {
                securityGroup.addIngressRule(rule, portRange, description);
                securityGroup.addEgressRule(rule, portRange, description);
            });
        });
    }
    /**
     * (experimental) Allow to all IPv4 ranges.
     *
     * @experimental
     */
    allowToAnyIpv4(portRange, description) {
        this.allowTo(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * (experimental) Allow from any IPv4 ranges.
     *
     * @experimental
     */
    allowFromAnyIpv4(portRange, description) {
        this.allowFrom(peer_1.Peer.anyIpv4(), portRange, description);
    }
    /**
     * (experimental) Allow connections from the peer on our default port.
     *
     * Even if the peer has a default port, we will always use our default port.
     *
     * @experimental
     */
    allowDefaultPortFrom(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFrom(): this resource has no default port');
        }
        this.allowFrom(other, this.defaultPort, description);
    }
    /**
     * (experimental) Allow hosts inside the security group to connect to each other.
     *
     * @experimental
     */
    allowDefaultPortInternally(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortInternally(): this resource has no default port');
        }
        this.allowInternally(this.defaultPort, description);
    }
    /**
     * (experimental) Allow default connections from all IPv4 ranges.
     *
     * @experimental
     */
    allowDefaultPortFromAnyIpv4(description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortFromAnyIpv4(): this resource has no default port');
        }
        this.allowFromAnyIpv4(this.defaultPort, description);
    }
    /**
     * (experimental) Allow connections to the security group on their default port.
     *
     * @experimental
     */
    allowToDefaultPort(other, description) {
        if (other.connections.defaultPort === undefined) {
            throw new Error('Cannot call alloToDefaultPort(): other resource has no default port');
        }
        this.allowTo(other, other.connections.defaultPort, description);
    }
    /**
     * (experimental) Allow connections from the peer on our default port.
     *
     * Even if the peer has a default port, we will always use our default port.
     *
     * @experimental
     */
    allowDefaultPortTo(other, description) {
        if (!this.defaultPort) {
            throw new Error('Cannot call allowDefaultPortTo(): this resource has no default port');
        }
        this.allowTo(other, this.defaultPort, description);
    }
}
exports.Connections = Connections;
class ReactiveList {
    constructor() {
        this.elements = new Array();
        this.listeners = new Array();
    }
    push(...xs) {
        this.elements.push(...xs);
        for (const listener of this.listeners) {
            for (const x of xs) {
                listener(x);
            }
        }
    }
    forEachAndForever(listener) {
        for (const element of this.elements) {
            listener(element);
        }
        this.listeners.push(listener);
    }
    asArray() {
        return this.elements.slice();
    }
    get length() {
        return this.elements.length;
    }
}
//# sourceMappingURL=data:application/json;base64,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