"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const crypto = require("crypto");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * (experimental) A CloudFormation-init configuration.
 *
 * @experimental
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * (experimental) Build a new config from a set of Init Elements.
     *
     * @experimental
     */
    static fromElements(...elements) {
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * (experimental) Use an existing InitConfig object as the default and only config.
     *
     * @experimental
     */
    static fromConfig(config) {
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * (experimental) Build a CloudFormationInit from config sets.
     *
     * @experimental
     */
    static fromConfigSets(props) {
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * (experimental) Add a config with the given name to this CloudFormationInit object.
     *
     * @experimental
     */
    addConfig(configName, config) {
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * (experimental) Add a config set with the given name to this CloudFormationInit object.
     *
     * The new configset will reference the given configs in the given order.
     *
     * @experimental
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * @internal
     */
    _attach(attachedResource, attachOptions) {
        var _a, _b, _c;
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata('AWS::CloudFormation::Init', bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        const resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const configSets = ((_a = attachOptions.configSets) !== null && _a !== void 0 ? _a : ['default']).join(',');
        const printLog = (_b = attachOptions.printLog) !== null && _b !== void 0 ? _b : true;
        if ((_c = attachOptions.embedFingerprint) !== null && _c !== void 0 ? _c : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : [],
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${resourceLocator}`,
                ...printLog ? ['  cat /var/log/cfn-init.log >&2'] : [],
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
/**
 * (experimental) A collection of configuration elements.
 *
 * @experimental
 */
class InitConfig {
    /**
     * @experimental
     */
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * (experimental) Whether this configset has elements or not.
     *
     * @experimental
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * (experimental) Add one or more elements to the config.
     *
     * @experimental
     */
    add(...elements) {
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _a;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_a = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _a !== void 0 ? _a : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _a, _b;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_a = target[key]) !== null && _a !== void 0 ? _a : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_b = target[key]) !== null && _b !== void 0 ? _b : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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