"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitSource = exports.InitService = exports.InitPackage = exports.InitUser = exports.InitGroup = exports.InitFile = exports.InitCommand = exports.InitCommandWaitDuration = exports.InitElement = exports.InitServiceRestartHandle = void 0;
const fs = require("fs");
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * (experimental) An object that represents reasons to restart an InitService.
 *
 * Pass an instance of this object to the `InitFile`, `InitCommand`,
 * `InitSource` and `InitPackage` objects, and finally to an `InitService`
 * itself to cause the actions (files, commands, sources, and packages)
 * to trigger a restart of the service.
 *
 * For example, the following will run a custom command to install Nginx,
 * and trigger the nginx service to be restarted after the command has run.
 *
 * ```ts
 * const handle = new ec2.InitServiceRestartHandle();
 * ec2.CloudFormationInit.fromElements(
 *    ec2.InitCommand.shellCommand('/usr/bin/custom-nginx-install.sh', { serviceRestartHandles: [handle] }),
 *    ec2.InitService.enable('nginx', { serviceRestartHandle: handle }),
 * );
 * ```
 *
 * @experimental
 */
class InitServiceRestartHandle {
    constructor() {
        this.commands = new Array();
        this.files = new Array();
        this.sources = new Array();
        this.packages = {};
    }
    /**
     * Add a command key to the restart set
     * @internal
     */
    _addCommand(key) {
        return this.commands.push(key);
    }
    /**
     * Add a file key to the restart set
     * @internal
     */
    _addFile(key) {
        return this.files.push(key);
    }
    /**
     * Add a source key to the restart set
     * @internal
     */
    _addSource(key) {
        return this.sources.push(key);
    }
    /**
     * Add a package key to the restart set
     * @internal
     */
    _addPackage(packageType, key) {
        if (!this.packages[packageType]) {
            this.packages[packageType] = [];
        }
        this.packages[packageType].push(key);
    }
    /**
     * Render the restart handles for use in an InitService declaration
     * @internal
     */
    _renderRestartHandles() {
        const nonEmpty = (x) => x.length > 0 ? x : undefined;
        return {
            commands: nonEmpty(this.commands),
            files: nonEmpty(this.files),
            packages: Object.keys(this.packages).length > 0 ? this.packages : undefined,
            sources: nonEmpty(this.sources),
        };
    }
}
exports.InitServiceRestartHandle = InitServiceRestartHandle;
/**
 * (experimental) Base class for all CloudFormation Init elements.
 *
 * @experimental
 */
class InitElement {
}
exports.InitElement = InitElement;
/**
 * (experimental) Represents a duration to wait after a command has finished, in case of a reboot (Windows only).
 *
 * @experimental
 */
class InitCommandWaitDuration {
    /**
     * (experimental) Wait for a specified duration after a command.
     *
     * @experimental
     */
    static of(duration) {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return duration.toSeconds(); }
        }();
    }
    /**
     * (experimental) Do not wait for this command.
     *
     * @experimental
     */
    static none() {
        return InitCommandWaitDuration.of(core_1.Duration.seconds(0));
    }
    /**
     * (experimental) cfn-init will exit and resume only after a reboot.
     *
     * @experimental
     */
    static forever() {
        return new class extends InitCommandWaitDuration {
            /** @internal */
            _render() { return 'forever'; }
        }();
    }
}
exports.InitCommandWaitDuration = InitCommandWaitDuration;
/**
 * (experimental) Command to execute on the instance.
 *
 * @experimental
 */
class InitCommand extends InitElement {
    constructor(command, options) {
        super();
        this.command = command;
        this.options = options;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.COMMAND.toString();
    }
    /**
     * (experimental) Run a shell command.
     *
     * Remember that some characters like `&`, `|`, `;`, `>` etc. have special meaning in a shell and
     * need to be preceded by a `\` if you want to treat them as part of a filename.
     *
     * @experimental
     */
    static shellCommand(shellCommand, options = {}) {
        return new InitCommand([shellCommand], options);
    }
    /**
     * (experimental) Run a command from an argv array.
     *
     * You do not need to escape space characters or enclose command parameters in quotes.
     *
     * @experimental
     */
    static argvCommand(argv, options = {}) {
        if (argv.length === 0) {
            throw new Error('Cannot define argvCommand with an empty arguments');
        }
        return new InitCommand(argv, options);
    }
    /** @internal */
    _bind(options) {
        var _a, _b;
        const commandKey = this.options.key || `${options.index}`.padStart(3, '0'); // 001, 005, etc.
        if (options.platform !== cfn_init_internal_1.InitPlatform.WINDOWS && this.options.waitAfterCompletion !== undefined) {
            throw new Error(`Command '${this.command}': 'waitAfterCompletion' is only valid for Windows systems.`);
        }
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addCommand(commandKey);
        }
        return {
            config: {
                [commandKey]: {
                    command: this.command,
                    env: this.options.env,
                    cwd: this.options.cwd,
                    test: this.options.testCmd,
                    ignoreErrors: this.options.ignoreErrors,
                    waitAfterCompletion: (_b = this.options.waitAfterCompletion) === null || _b === void 0 ? void 0 : _b._render(),
                },
            },
        };
    }
}
exports.InitCommand = InitCommand;
/**
 * (experimental) Create files on the EC2 instance.
 *
 * @experimental
 */
class InitFile extends InitElement {
    /**
     * @experimental
     */
    constructor(fileName, options) {
        super();
        this.fileName = fileName;
        this.options = options;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.FILE.toString();
    }
    /**
     * (experimental) Use a literal string as the file content.
     *
     * @experimental
     */
    static fromString(fileName, content, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content,
                        encoding: this.options.base64Encoded ? 'base64' : 'plain',
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Write a symlink with the given symlink target.
     *
     * @experimental
     */
    static symlink(fileName, target, options = {}) {
        const { mode, ...otherOptions } = options;
        if (mode && mode.slice(0, 3) !== '120') {
            throw new Error('File mode for symlinks must begin with 120XXX');
        }
        return InitFile.fromString(fileName, target, { mode: (mode || '120644'), ...otherOptions });
    }
    /**
     * (experimental) Use a JSON-compatible object as the file content, write it to a JSON file.
     *
     * May contain tokens.
     *
     * @experimental
     */
    static fromObject(fileName, obj, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        content: obj,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Read a file from disk and use its contents.
     *
     * The file will be embedded in the template, so care should be taken to not
     * exceed the template size.
     *
     * If options.base64encoded is set to true, this will base64-encode the file's contents.
     *
     * @experimental
     */
    static fromFileInline(targetFileName, sourceFileName, options = {}) {
        const encoding = options.base64Encoded ? 'base64' : 'utf8';
        const fileContents = fs.readFileSync(sourceFileName).toString(encoding);
        return InitFile.fromString(targetFileName, fileContents, options);
    }
    /**
     * (experimental) Download from a URL at instance startup time.
     *
     * @experimental
     */
    static fromUrl(fileName, url, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: url,
                    }),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Download a file from an S3 bucket at instance startup time.
     *
     * @experimental
     */
    static fromS3Object(fileName, bucket, key, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: bucket.urlForObject(key),
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(fileName, options);
    }
    /**
     * (experimental) Create an asset from the given file.
     *
     * This is appropriate for files that are too large to embed into the template.
     *
     * @experimental
     */
    static fromAsset(targetFileName, path, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetFileName}Asset`, {
                    path,
                    ...options,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /**
     * (experimental) Use a file from an asset at instance startup time.
     *
     * @experimental
     */
    static fromExistingAsset(targetFileName, asset, options = {}) {
        return new class extends InitFile {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: this._standardConfig(options, bindOptions.platform, {
                        source: asset.httpUrl,
                    }),
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetFileName, options);
    }
    /** @internal */
    _bind(bindOptions) {
        var _a;
        for (const handle of (_a = this.options.serviceRestartHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addFile(this.fileName);
        }
        return this._doBind(bindOptions);
    }
    /**
     * Render the standard config block, given content vars
     * @internal
     */
    _standardConfig(fileOptions, platform, contentVars) {
        if (platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            if (fileOptions.group || fileOptions.owner || fileOptions.mode) {
                throw new Error('Owner, group, and mode options not supported for Windows.');
            }
            return {
                [this.fileName]: { ...contentVars },
            };
        }
        return {
            [this.fileName]: {
                ...contentVars,
                mode: fileOptions.mode || '000644',
                owner: fileOptions.owner || 'root',
                group: fileOptions.group || 'root',
            },
        };
    }
}
exports.InitFile = InitFile;
/**
 * (experimental) Create Linux/UNIX groups and assign group IDs.
 *
 * Not supported for Windows systems.
 *
 * @experimental
 */
class InitGroup extends InitElement {
    /**
     * @experimental
     */
    constructor(groupName, groupId) {
        super();
        this.groupName = groupName;
        this.groupId = groupId;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.GROUP.toString();
    }
    /**
     * (experimental) Create a group from its name, and optionally, group id.
     *
     * @experimental
     */
    static fromName(groupName, groupId) {
        return new InitGroup(groupName, groupId);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init groups are not supported on Windows');
        }
        return {
            config: {
                [this.groupName]: this.groupId !== undefined ? { gid: this.groupId } : {},
            },
        };
    }
}
exports.InitGroup = InitGroup;
/**
 * (experimental) Create Linux/UNIX users and to assign user IDs.
 *
 * Users are created as non-interactive system users with a shell of
 * /sbin/nologin. This is by design and cannot be modified.
 *
 * Not supported for Windows systems.
 *
 * @experimental
 */
class InitUser extends InitElement {
    /**
     * @experimental
     */
    constructor(userName, userOptions) {
        super();
        this.userName = userName;
        this.userOptions = userOptions;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.USER.toString();
    }
    /**
     * (experimental) Create a user from user name.
     *
     * @experimental
     */
    static fromName(userName, options = {}) {
        return new InitUser(userName, options);
    }
    /** @internal */
    _bind(options) {
        if (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS) {
            throw new Error('Init users are not supported on Windows');
        }
        return {
            config: {
                [this.userName]: {
                    uid: this.userOptions.userId,
                    groups: this.userOptions.groups,
                    homeDir: this.userOptions.homeDir,
                },
            },
        };
    }
}
exports.InitUser = InitUser;
/**
 * (experimental) A package to be installed during cfn-init time.
 *
 * @experimental
 */
class InitPackage extends InitElement {
    /**
     * @experimental
     */
    constructor(type, versions, packageName, serviceHandles) {
        super();
        this.type = type;
        this.versions = versions;
        this.packageName = packageName;
        this.serviceHandles = serviceHandles;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.PACKAGE.toString();
    }
    /**
     * (experimental) Install an RPM from an HTTP URL or a location on disk.
     *
     * @experimental
     */
    static rpm(location, options = {}) {
        return new InitPackage('rpm', [location], options.key, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package using Yum.
     *
     * @experimental
     */
    static yum(packageName, options = {}) {
        var _a;
        return new InitPackage('yum', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package from RubyGems.
     *
     * @experimental
     */
    static rubyGem(gemName, options = {}) {
        var _a;
        return new InitPackage('rubygems', (_a = options.version) !== null && _a !== void 0 ? _a : [], gemName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package from PyPI.
     *
     * @experimental
     */
    static python(packageName, options = {}) {
        var _a;
        return new InitPackage('python', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install a package using APT.
     *
     * @experimental
     */
    static apt(packageName, options = {}) {
        var _a;
        return new InitPackage('apt', (_a = options.version) !== null && _a !== void 0 ? _a : [], packageName, options.serviceRestartHandles);
    }
    /**
     * (experimental) Install an MSI package from an HTTP URL or a location on disk.
     *
     * @experimental
     */
    static msi(location, options = {}) {
        // The MSI package version must be a string, not an array.
        return new class extends InitPackage {
            renderPackageVersions() { return location; }
        }('msi', [location], options.key, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        if ((this.type === 'msi') !== (options.platform === cfn_init_internal_1.InitPlatform.WINDOWS)) {
            if (this.type === 'msi') {
                throw new Error('MSI installers are only supported on Windows systems.');
            }
            else {
                throw new Error('Windows only supports the MSI package type');
            }
        }
        if (!this.packageName && !['rpm', 'msi'].includes(this.type)) {
            throw new Error('Package name must be specified for all package types besides RPM and MSI.');
        }
        const packageName = this.packageName || `${options.index}`.padStart(3, '0');
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addPackage(this.type, packageName);
        }
        return {
            config: {
                [this.type]: {
                    [packageName]: this.renderPackageVersions(),
                },
            },
        };
    }
    /**
     * @experimental
     */
    renderPackageVersions() {
        return this.versions;
    }
}
exports.InitPackage = InitPackage;
/**
 * (experimental) A services that be enabled, disabled or restarted when the instance is launched.
 *
 * @experimental
 */
class InitService extends InitElement {
    constructor(serviceName, serviceOptions) {
        super();
        this.serviceName = serviceName;
        this.serviceOptions = serviceOptions;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.SERVICE.toString();
    }
    /**
     * (experimental) Enable and start the given service, optionally restarting it.
     *
     * @experimental
     */
    static enable(serviceName, options = {}) {
        var _a;
        const { enabled, ensureRunning, ...otherOptions } = options;
        return new InitService(serviceName, {
            enabled: enabled !== null && enabled !== void 0 ? enabled : true,
            ensureRunning: (_a = ensureRunning !== null && ensureRunning !== void 0 ? ensureRunning : enabled) !== null && _a !== void 0 ? _a : true,
            ...otherOptions,
        });
    }
    /**
     * (experimental) Disable and stop the given service.
     *
     * @experimental
     */
    static disable(serviceName) {
        return new InitService(serviceName, { enabled: false, ensureRunning: false });
    }
    /** @internal */
    _bind(options) {
        var _a;
        const serviceManager = options.platform === cfn_init_internal_1.InitPlatform.LINUX ? 'sysvinit' : 'windows';
        return {
            config: {
                [serviceManager]: {
                    [this.serviceName]: {
                        enabled: this.serviceOptions.enabled,
                        ensureRunning: this.serviceOptions.ensureRunning,
                        ...(_a = this.serviceOptions.serviceRestartHandle) === null || _a === void 0 ? void 0 : _a._renderRestartHandles(),
                    },
                },
            },
        };
    }
}
exports.InitService = InitService;
/**
 * (experimental) Extract an archive into a directory.
 *
 * @experimental
 */
class InitSource extends InitElement {
    /**
     * @experimental
     */
    constructor(targetDirectory, serviceHandles) {
        super();
        this.targetDirectory = targetDirectory;
        this.serviceHandles = serviceHandles;
        /**
         * (experimental) Returns the init element type for this element.
         *
         * @experimental
         */
        this.elementType = cfn_init_internal_1.InitElementType.SOURCE.toString();
    }
    /**
     * (experimental) Retrieve a URL and extract it into the given directory.
     *
     * @experimental
     */
    static fromUrl(targetDirectory, url, options = {}) {
        return new class extends InitSource {
            _doBind() {
                return {
                    config: { [this.targetDirectory]: url },
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * (experimental) Extract a GitHub branch into a given directory.
     *
     * @experimental
     */
    static fromGitHub(targetDirectory, owner, repo, refSpec, options = {}) {
        return InitSource.fromUrl(targetDirectory, `https://github.com/${owner}/${repo}/tarball/${refSpec !== null && refSpec !== void 0 ? refSpec : 'master'}`, options);
    }
    /**
     * (experimental) Extract an archive stored in an S3 bucket into the given directory.
     *
     * @experimental
     */
    static fromS3Object(targetDirectory, bucket, key, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                bucket.grantRead(bindOptions.instanceRole, key);
                return {
                    config: { [this.targetDirectory]: bucket.urlForObject(key) },
                    authentication: standardS3Auth(bindOptions.instanceRole, bucket.bucketName),
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * (experimental) Create an InitSource from an asset created from the given path.
     *
     * @experimental
     */
    static fromAsset(targetDirectory, path, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                const asset = new s3_assets.Asset(bindOptions.scope, `${targetDirectory}Asset`, {
                    path,
                    ...bindOptions,
                });
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /**
     * (experimental) Extract a directory from an existing directory asset.
     *
     * @experimental
     */
    static fromExistingAsset(targetDirectory, asset, options = {}) {
        return new class extends InitSource {
            _doBind(bindOptions) {
                asset.grantRead(bindOptions.instanceRole);
                return {
                    config: { [this.targetDirectory]: asset.httpUrl },
                    authentication: standardS3Auth(bindOptions.instanceRole, asset.s3BucketName),
                    assetHash: asset.assetHash,
                };
            }
        }(targetDirectory, options.serviceRestartHandles);
    }
    /** @internal */
    _bind(options) {
        var _a;
        for (const handle of (_a = this.serviceHandles) !== null && _a !== void 0 ? _a : []) {
            handle._addSource(this.targetDirectory);
        }
        // Delegate actual bind to subclasses
        return this._doBind(options);
    }
}
exports.InitSource = InitSource;
/**
 * Render a standard S3 auth block for use in AWS::CloudFormation::Authentication
 *
 * This block is the same every time (modulo bucket name), so it has the same
 * key every time so the blocks are merged into one in the final render.
 */
function standardS3Auth(role, bucketName) {
    return {
        S3AccessCreds: {
            type: 'S3',
            roleName: role.roleName,
            buckets: [bucketName],
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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