"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomRule = exports.ManagedRule = exports.MaximumExecutionFrequency = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const config_generated_1 = require("./config.generated");
/**
 * A new or imported rule.
 */
class RuleBase extends core_1.Resource {
    /**
     * (experimental) Defines a CloudWatch event rule which triggers for rule events.
     *
     * Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     *
     * @experimental
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.config'],
            detail: {
                configRuleName: [this.configRuleName],
            },
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * (experimental) Defines a CloudWatch event rule which triggers for rule compliance events.
     *
     * @experimental
     */
    onComplianceChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Compliance Change'],
        });
        return rule;
    }
    /**
     * (experimental) Defines a CloudWatch event rule which triggers for rule re-evaluation status events.
     *
     * @experimental
     */
    onReEvaluationStatus(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['Config Rules Re-evaluation Status'],
        });
        return rule;
    }
}
/**
 * A new managed or custom rule.
 */
class RuleNew extends RuleBase {
    /**
     * (experimental) Imports an existing rule.
     *
     * @param configRuleName the name of the rule.
     * @experimental
     */
    static fromConfigRuleName(scope, id, configRuleName) {
        class Import extends RuleBase {
            constructor() {
                super(...arguments);
                this.configRuleName = configRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Restrict scope of changes to a specific resource.
     *
     * @param type the resource type.
     * @param identifier the resource identifier.
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     * @experimental
     */
    scopeToResource(type, identifier) {
        this.scope = {
            complianceResourceId: identifier,
            complianceResourceTypes: [type],
        };
    }
    /**
     * (experimental) Restrict scope of changes to specific resource types.
     *
     * @param types resource types.
     * @see https://docs.aws.amazon.com/config/latest/developerguide/resource-config-reference.html#supported-resources
     * @experimental
     */
    scopeToResources(...types) {
        this.scope = {
            complianceResourceTypes: types,
        };
    }
    /**
     * (experimental) Restrict scope of changes to a specific tag.
     *
     * @param key the tag key.
     * @param value the tag value.
     * @experimental
     */
    scopeToTag(key, value) {
        this.scope = {
            tagKey: key,
            tagValue: value,
        };
    }
}
/**
 * (experimental) The maximum frequency at which the AWS Config rule runs evaluations.
 *
 * @experimental
 */
var MaximumExecutionFrequency;
(function (MaximumExecutionFrequency) {
    MaximumExecutionFrequency["ONE_HOUR"] = "One_Hour";
    MaximumExecutionFrequency["THREE_HOURS"] = "Three_Hours";
    MaximumExecutionFrequency["SIX_HOURS"] = "Six_Hours";
    MaximumExecutionFrequency["TWELVE_HOURS"] = "Twelve_Hours";
    MaximumExecutionFrequency["TWENTY_FOUR_HOURS"] = "TwentyFour_Hours";
})(MaximumExecutionFrequency = exports.MaximumExecutionFrequency || (exports.MaximumExecutionFrequency = {}));
/**
 * (experimental) A new managed rule.
 *
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class ManagedRule extends RuleNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'AWS',
                sourceIdentifier: props.identifier,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        this.isManaged = true;
    }
}
exports.ManagedRule = ManagedRule;
/**
 * (experimental) A new custom rule.
 *
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class CustomRule extends RuleNew {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.configRuleName,
        });
        if (!props.configurationChanges && !props.periodic) {
            throw new Error('At least one of `configurationChanges` or `periodic` must be set to true.');
        }
        const sourceDetails = [];
        if (props.configurationChanges) {
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'ConfigurationItemChangeNotification',
            });
            sourceDetails.push({
                eventSource: 'aws.config',
                messageType: 'OversizedConfigurationItemChangeNotification',
            });
        }
        if (props.periodic) {
            sourceDetails.push({
                eventSource: 'aws.config',
                maximumExecutionFrequency: props.maximumExecutionFrequency,
                messageType: 'ScheduledNotification',
            });
        }
        props.lambdaFunction.addPermission('Permission', {
            principal: new iam.ServicePrincipal('config.amazonaws.com'),
        });
        if (props.lambdaFunction.role) {
            props.lambdaFunction.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSConfigRulesExecutionRole'));
        }
        // The lambda permission must be created before the rule
        this.node.addDependency(props.lambdaFunction);
        const rule = new config_generated_1.CfnConfigRule(this, 'Resource', {
            configRuleName: this.physicalName,
            description: props.description,
            inputParameters: props.inputParameters,
            maximumExecutionFrequency: props.maximumExecutionFrequency,
            scope: core_1.Lazy.anyValue({ produce: () => this.scope }),
            source: {
                owner: 'CUSTOM_LAMBDA',
                sourceDetails,
                sourceIdentifier: props.lambdaFunction.functionArn,
            },
        });
        this.configRuleName = rule.ref;
        this.configRuleArn = rule.attrArn;
        this.configRuleId = rule.attrConfigRuleId;
        this.configRuleComplianceType = rule.attrComplianceType;
        if (props.configurationChanges) {
            this.isCustomWithChanges = true;
        }
    }
}
exports.CustomRule = CustomRule;
//# sourceMappingURL=data:application/json;base64,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