"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationStackNotificationCheck = exports.CloudFormationStackDriftDetectionCheck = exports.AccessKeysRotated = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rule_1 = require("./rule");
/**
 * (experimental) Checks whether the active access keys are rotated within the number of days specified in `maxAge`.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/access-keys-rotated.html
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class AccessKeysRotated extends rule_1.ManagedRule {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'ACCESS_KEYS_ROTATED',
            inputParameters: {
                ...props.maxAge
                    ? {
                        maxAccessKeyAge: props.maxAge.toDays(),
                    }
                    : {},
            },
        });
    }
}
exports.AccessKeysRotated = AccessKeysRotated;
/**
 * (experimental) Checks whether your CloudFormation stacks' actual configuration differs, or has drifted, from its expected configuration.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-drift-detection-check.html
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackDriftDetectionCheck extends rule_1.ManagedRule {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_DRIFT_DETECTION_CHECK',
            inputParameters: {
                cloudformationRoleArn: core_1.Lazy.stringValue({ produce: () => this.role.roleArn }),
            },
        });
        this.scopeToResource('AWS::CloudFormation::Stack', props.ownStackOnly ? core_1.Stack.of(this).stackId : undefined);
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('config.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'),
            ],
        });
    }
}
exports.CloudFormationStackDriftDetectionCheck = CloudFormationStackDriftDetectionCheck;
/**
 * (experimental) Checks whether your CloudFormation stacks are sending event notifications to a SNS topic.
 *
 * Optionally checks whether specified SNS topics are used.
 *
 * @see https://docs.aws.amazon.com/config/latest/developerguide/cloudformation-stack-notification-check.html
 * @experimental
 * @resource AWS::Config::ConfigRule
 */
class CloudFormationStackNotificationCheck extends rule_1.ManagedRule {
    /**
     * @experimental
     */
    constructor(scope, id, props = {}) {
        if (props.topics && props.topics.length > 5) {
            throw new Error('At most 5 topics can be specified.');
        }
        super(scope, id, {
            ...props,
            identifier: 'CLOUDFORMATION_STACK_NOTIFICATION_CHECK',
            inputParameters: props.topics && props.topics.reduce((params, topic, idx) => ({ ...params, [`snsTopic${idx + 1}`]: topic.topicArn }), {}),
        });
        this.scopeToResource('AWS::CloudFormation::Stack');
    }
}
exports.CloudFormationStackNotificationCheck = CloudFormationStackNotificationCheck;
//# sourceMappingURL=data:application/json;base64,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