"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RichAction = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Helper routines to work with Actions
 *
 * Can't put these on Action themselves since we only have an interface
 * and every library would need to reimplement everything (there is no
 * `ActionBase`).
 *
 * So here go the members that should have gone onto the Action class
 * but can't.
 *
 * It was probably my own idea but I don't want it anymore:
 * https://github.com/aws/aws-cdk/issues/10393
 */
class RichAction {
    constructor(action, pipeline) {
        this.action = action;
        this.pipeline = pipeline;
        this.actionProperties = action.actionProperties;
    }
    bind(scope, stage, options) {
        return this.action.bind(scope, stage, options);
    }
    onStateChange(name, target, options) {
        return this.action.onStateChange(name, target, options);
    }
    get isCrossRegion() {
        return !actionDimensionSameAsPipelineDimension(this.effectiveRegion, this.pipeline.env.region);
    }
    get isCrossAccount() {
        return !actionDimensionSameAsPipelineDimension(this.effectiveAccount, this.pipeline.env.account);
    }
    /**
     * Returns the Stack of the resource backing this action
     * if they belong to the same environment.
     * Returns `undefined` if either this action is not backed by a resource,
     * or if the resource does not belong to the same env as its Stack
     * (which can happen for imported resources).
     */
    get resourceStack() {
        const actionResource = this.actionProperties.resource;
        if (!actionResource) {
            return undefined;
        }
        const actionResourceStack = core_1.Stack.of(actionResource);
        const actionResourceStackEnv = {
            region: actionResourceStack.region,
            account: actionResourceStack.account,
        };
        return sameEnv(actionResource.env, actionResourceStackEnv) ? actionResourceStack : undefined;
    }
    /**
     * The region this action wants to execute in.
     * `undefined` means it wants to execute in the same region as the pipeline.
     */
    get effectiveRegion() {
        var _a, _b;
        return (_b = (_a = this.action.actionProperties.resource) === null || _a === void 0 ? void 0 : _a.env.region) !== null && _b !== void 0 ? _b : this.action.actionProperties.region;
    }
    /**
     * The account this action wants to execute in.
     * `undefined` means it wants to execute in the same account as the pipeline.
     */
    get effectiveAccount() {
        var _a, _b, _c, _d, _e;
        return (_e = (_b = (_a = this.action.actionProperties.role) === null || _a === void 0 ? void 0 : _a.env.account) !== null && _b !== void 0 ? _b : (_d = (_c = this.action.actionProperties) === null || _c === void 0 ? void 0 : _c.resource) === null || _d === void 0 ? void 0 : _d.env.account) !== null && _e !== void 0 ? _e : this.action.actionProperties.account;
    }
}
exports.RichAction = RichAction;
function actionDimensionSameAsPipelineDimension(actionDim, pipelineDim) {
    // if the action's dimension is `undefined`,
    // it means it is in the same region/account as the pipeline
    if (!actionDim) {
        return true;
    }
    // if the action's region/account is AWS::Region/AWS::AccountId,
    // we assume it's also in the same region/account as the pipeline
    if (core_1.Token.isUnresolved(actionDim)) {
        return true;
    }
    // here, we know the action's dimension is explicitly set;
    // in this case, it must be equal to the pipeline's dimension
    // for the action to be considered in the same region/account
    return core_1.Token.compareStrings(actionDim, pipelineDim) === core_1.TokenComparison.SAME;
}
/**
 * Whether the two envs represent the same environment
 */
function sameEnv(env1, env2) {
    return sameEnvDimension(env1.region, env2.region)
        && sameEnvDimension(env1.account, env2.account);
}
/**
 * Whether two string probably contain the same environment dimension (region or account)
 *
 * Used to compare either accounts or regions, and also returns true if both
 * are unresolved (in which case both are expted to be "current region" or "current account").
 */
function sameEnvDimension(dim1, dim2) {
    return [core_1.TokenComparison.SAME, core_1.TokenComparison.BOTH_UNRESOLVED].includes(core_1.Token.compareStrings(dim1, dim2));
}
//# sourceMappingURL=data:application/json;base64,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