"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("../../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id);
        const createKmsKey = (_a = props.createKmsKey) !== null && _a !== void 0 ? _a : true;
        let encryptionAlias;
        if (createKmsKey) {
            const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
            encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
                targetKey: encryptionKey,
                aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                removalPolicy: cdk.RemovalPolicy.DESTROY,
            });
        }
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryption: encryptionAlias ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default', {
            createKmsKey: props.createKmsKey,
        });
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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