"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * (experimental) How should the S3 Action detect changes.
 *
 * This is the type of the {@link S3SourceAction.trigger} property.
 *
 * @experimental
 */
var S3Trigger;
(function (S3Trigger) {
    S3Trigger["NONE"] = "None";
    S3Trigger["POLL"] = "Poll";
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * (experimental) Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 *
 * @experimental
 */
class S3SourceAction extends action_1.Action {
    /**
     * @experimental
     */
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /**
     * (experimental) The variables emitted by this action.
     *
     * @experimental
     */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    /**
     * (experimental) The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @experimental
     */
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = stage.pipeline.node.uniqueId + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,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