"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * (experimental) How should the CodeCommit Action detect changes.
 *
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 *
 * @experimental
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    CodeCommitTrigger["NONE"] = "None";
    CodeCommitTrigger["POLL"] = "Poll";
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * (experimental) CodePipeline Source that is provided by an AWS CodeCommit repository.
 *
 * @experimental
 */
class CodeCommitSourceAction extends action_1.Action {
    /**
     * @experimental
     */
    constructor(props) {
        var _a;
        const branch = (_a = props.branch) !== null && _a !== void 0 ? _a : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /**
     * (experimental) The variables emitted by this action.
     *
     * @experimental
     */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    /**
     * (experimental) The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @experimental
     */
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        const baseId = stage.pipeline.node.uniqueId;
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : '-${this.branch}-';
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
//# sourceMappingURL=data:application/json;base64,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