"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeleteStackAction = exports.CloudFormationCreateUpdateStackAction = exports.CloudFormationCreateReplaceChangeSetAction = exports.CloudFormationExecuteChangeSetAction = void 0;
const cloudformation = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * (experimental) CodePipeline action to execute a prepared change set.
 *
 * @experimental
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    /**
     * @experimental
     */
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    /**
     * (experimental) The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @experimental
     */
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * (experimental) Add statement to the service role assumed by CloudFormation while executing this action.
     *
     * @experimental
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    /**
     * @experimental
     */
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * (experimental) CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 *
 * @experimental
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    /**
     * @experimental
     */
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    /**
     * (experimental) The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @experimental
     */
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * (experimental) CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 *
 * @experimental
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    /**
     * @experimental
     */
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    /**
     * (experimental) The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @experimental
     */
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * (experimental) CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 *
 * @experimental
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    /**
     * @experimental
     */
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    /**
     * (experimental) The method called when an Action is attached to a Pipeline.
     *
     * This method is guaranteed to be called only once for each Action instance.
     *
     * @experimental
     */
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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