"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ValidationMethod = exports.Certificate = exports.CertificateValidation = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const certificatemanager_generated_1 = require("./certificatemanager.generated");
const util_1 = require("./util");
/**
 * (experimental) How to validate a certificate.
 *
 * @experimental
 */
class CertificateValidation {
    /** @param props Certification validation properties */
    constructor(props) {
        var _a;
        this.props = props;
        this.method = (_a = props.method) !== null && _a !== void 0 ? _a : ValidationMethod.EMAIL;
    }
    /**
     * (experimental) Validate the certifcate with DNS.
     *
     * IMPORTANT: If `hostedZone` is not specified, DNS records must be added
     * manually and the stack will not complete creating until the records are
     * added.
     *
     * @param hostedZone the hosted zone where DNS records must be created.
     * @experimental
     */
    static fromDns(hostedZone) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZone,
        });
    }
    /**
     * (experimental) Validate the certifcate with automatically created DNS records in multiple Amazon Route 53 hosted zones.
     *
     * @param hostedZones a map of hosted zones where DNS records must be created for the domains in the certificate.
     * @experimental
     */
    static fromDnsMultiZone(hostedZones) {
        return new CertificateValidation({
            method: ValidationMethod.DNS,
            hostedZones,
        });
    }
    /**
     * (experimental) Validate the certifcate with Email.
     *
     * IMPORTANT: if you are creating a certificate as part of your stack, the stack
     * will not complete creating until you read and follow the instructions in the
     * email that you will receive.
     *
     * ACM will send validation emails to the following addresses:
     *
     *   admin@domain.com
     *   administrator@domain.com
     *   hostmaster@domain.com
     *   postmaster@domain.com
     *   webmaster@domain.com
     *
     * For every domain that you register.
     *
     * @param validationDomains a map of validation domains to use for domains in the certificate.
     * @experimental
     */
    static fromEmail(validationDomains) {
        return new CertificateValidation({
            method: ValidationMethod.EMAIL,
            validationDomains,
        });
    }
}
exports.CertificateValidation = CertificateValidation;
/**
 * (experimental) A certificate managed by AWS Certificate Manager.
 *
 * @experimental
 */
class Certificate extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        let validation;
        if (props.validation) {
            validation = props.validation;
        }
        else { // Deprecated props
            if (props.validationMethod === ValidationMethod.DNS) {
                validation = CertificateValidation.fromDns();
            }
            else {
                validation = CertificateValidation.fromEmail(props.validationDomains);
            }
        }
        const allDomainNames = [props.domainName].concat(props.subjectAlternativeNames || []);
        const cert = new certificatemanager_generated_1.CfnCertificate(this, 'Resource', {
            domainName: props.domainName,
            subjectAlternativeNames: props.subjectAlternativeNames,
            domainValidationOptions: renderDomainValidation(validation, allDomainNames),
            validationMethod: validation.method,
        });
        this.certificateArn = cert.ref;
    }
    /**
     * (experimental) Import a certificate.
     *
     * @experimental
     */
    static fromCertificateArn(scope, id, certificateArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.certificateArn = certificateArn;
            }
        }
        return new Import(scope, id);
    }
}
exports.Certificate = Certificate;
/**
 * (experimental) Method used to assert ownership of the domain.
 *
 * @experimental
 */
var ValidationMethod;
(function (ValidationMethod) {
    ValidationMethod["EMAIL"] = "EMAIL";
    ValidationMethod["DNS"] = "DNS";
})(ValidationMethod = exports.ValidationMethod || (exports.ValidationMethod = {}));
// eslint-disable-next-line max-len
function renderDomainValidation(validation, domainNames) {
    var _a, _b, _c;
    const domainValidation = [];
    switch (validation.method) {
        case ValidationMethod.DNS:
            for (const domainName of getUniqueDnsDomainNames(domainNames)) {
                const hostedZone = (_b = (_a = validation.props.hostedZones) === null || _a === void 0 ? void 0 : _a[domainName]) !== null && _b !== void 0 ? _b : validation.props.hostedZone;
                if (hostedZone) {
                    domainValidation.push({ domainName, hostedZoneId: hostedZone.hostedZoneId });
                }
            }
            break;
        case ValidationMethod.EMAIL:
            for (const domainName of domainNames) {
                const validationDomain = (_c = validation.props.validationDomains) === null || _c === void 0 ? void 0 : _c[domainName];
                if (!validationDomain && core_1.Token.isUnresolved(domainName)) {
                    throw new Error('When using Tokens for domain names, \'validationDomains\' needs to be supplied');
                }
                domainValidation.push({ domainName, validationDomain: validationDomain !== null && validationDomain !== void 0 ? validationDomain : util_1.apexDomain(domainName) });
            }
            break;
        default:
            throw new Error(`Unknown validation method ${validation.method}`);
    }
    return domainValidation.length !== 0 ? domainValidation : undefined;
}
/**
 * Removes wildcard domains (*.example.com) where the base domain (example.com) is present.
 * This is because the DNS validation treats them as the same thing, and the automated CloudFormation
 * DNS validation errors out with the duplicate records.
 */
function getUniqueDnsDomainNames(domainNames) {
    return domainNames.filter(domain => {
        return core_1.Token.isUnresolved(domain) || !domain.startsWith('*.') || !domainNames.includes(domain.replace('*.', ''));
    });
}
//# sourceMappingURL=data:application/json;base64,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