"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findAlarmThresholds = exports.normalizeIntervals = void 0;
/**
 * Normalize the given interval set to cover the complete number line and make sure it has at most one gap
 */
function normalizeIntervals(intervals, changesAreAbsolute) {
    // Make intervals a complete numberline
    const full = orderAndCompleteIntervals(intervals);
    // Add 'undefined's in uncovered areas of the number line
    makeGapsUndefined(full);
    // In case of relative changes, treat 0-change also as 'undefined' (= no change action)
    if (!changesAreAbsolute) {
        makeZerosUndefined(full);
    }
    // Combine adjacent undefines and make sure there's at most one of them
    combineUndefineds(full);
    validateAtMostOneUndefined(full);
    return full;
}
exports.normalizeIntervals = normalizeIntervals;
/**
 * Completely order scaling intervals, making their lower and upper bounds concrete.
 */
function orderAndCompleteIntervals(intervals) {
    if (intervals.length < 2) {
        throw new Error('Require at least 2 intervals');
    }
    for (const interval of intervals) {
        if (interval.lower === undefined && interval.upper === undefined) {
            throw new Error(`Must supply at least one of 'upper' or 'lower', got: ${JSON.stringify(interval)}`);
        }
    }
    // Make a copy
    intervals = intervals.map(x => ({ ...x }));
    // Sort by whatever number we have for each interval
    intervals.sort(comparatorFromKey((x) => x.lower !== undefined ? x.lower : x.upper));
    // Propagate boundaries until no more change
    while (propagateBounds(intervals)) { /* Repeat */ }
    const lastIndex = intervals.length - 1;
    // Validate that no intervals have undefined bounds now, which must mean they're complete.
    if (intervals[0].lower === undefined) {
        intervals[0] = { ...intervals[0], lower: 0 };
    }
    if (intervals[lastIndex].upper === undefined) {
        intervals[lastIndex] = { ...intervals[lastIndex], upper: Infinity };
    }
    for (const interval of intervals) {
        if (interval.lower === undefined || interval.upper === undefined) {
            throw new Error(`Could not determine the lower and upper bounds for ${JSON.stringify(interval)}`);
        }
    }
    const completeIntervals = intervals;
    // Validate that we have nonoverlapping intervals now.
    for (let i = 0; i < completeIntervals.length - 1; i++) {
        if (overlap(completeIntervals[i], completeIntervals[i + 1])) {
            throw new Error(`Two intervals overlap: ${JSON.stringify(completeIntervals[i])} and ${JSON.stringify(completeIntervals[i + 1])}`);
        }
    }
    // Fill up the gaps
    return completeIntervals;
}
/**
 * Make the intervals cover the complete number line
 *
 * This entails adding intervals with an 'undefined' change to fill up the gaps.
 *
 * Since metrics have a halfopen interval, the first one will get a lower bound
 * of 0, the last one will get an upper bound of +Infinity.
 *
 * In case of absolute adjustments, the lower number of the adjacent bound will
 * be used, which means conservative change. In case of relative adjustments,
 * we'll use relative adjusment 0 (which means no change).
 */
function makeGapsUndefined(intervals) {
    // Add edge intervals if necessary, but only for relative adjustments. Since we're
    // going to make scaling intervals extend all the way out to infinity on either side,
    // the result is the same for absolute adjustments anyway.
    if (intervals[0].lower !== 0) {
        intervals.splice(0, 1, {
            lower: 0,
            upper: intervals[0].lower,
            change: undefined,
        });
    }
    if (last(intervals).upper !== Infinity) {
        intervals.push({
            lower: last(intervals).upper,
            upper: Infinity,
            change: undefined,
        });
    }
    let i = 1;
    while (i < intervals.length) {
        if (intervals[i - 1].upper < intervals[i].lower) {
            intervals.splice(i, 0, {
                lower: intervals[i - 1].upper,
                upper: intervals[i].lower,
                change: undefined,
            });
        }
        else {
            i++;
        }
    }
}
/**
 * Turn zero changes into undefined, in-place
 */
function makeZerosUndefined(intervals) {
    for (let i = 0; i < intervals.length; ++i) {
        const interval = intervals[i];
        if (interval.change === 0) {
            intervals[i] = { ...interval, change: undefined };
        }
    }
}
/**
 * If there are adjacent "undefined" intervals, combine them
 */
function combineUndefineds(intervals) {
    let i = 0;
    while (i < intervals.length - 1) {
        if (intervals[i].change === undefined && intervals[i + 1].change === undefined) {
            intervals[i] = { ...intervals[i], upper: intervals[i + 1].upper };
            intervals.splice(i + 1, 1);
        }
        else {
            i++;
        }
    }
}
function validateAtMostOneUndefined(intervals) {
    const undef = intervals.filter(x => x.change === undefined);
    if (undef.length > 1) {
        throw new Error(`Can have at most one no-change interval, got ${JSON.stringify(undef)}`);
    }
}
function comparatorFromKey(keyFn) {
    return (a, b) => {
        const keyA = keyFn(a);
        const keyB = keyFn(b);
        if (keyA < keyB) {
            return -1;
        }
        if (keyA === keyB) {
            return 0;
        }
        return 1;
    };
}
function propagateBounds(intervals) {
    let ret = false;
    // Propagate upper bounds upwards
    for (let i = 0; i < intervals.length - 1; i++) {
        if (intervals[i].upper !== undefined && intervals[i + 1].lower === undefined) {
            intervals[i + 1] = { ...intervals[i + 1], lower: intervals[i].upper };
            ret = true;
        }
    }
    // Propagate lower bounds downwards
    for (let i = intervals.length - 1; i >= 1; i--) {
        if (intervals[i].lower !== undefined && intervals[i - 1].upper === undefined) {
            intervals[i - 1] = { ...intervals[i - 1], upper: intervals[i].lower };
            ret = true;
        }
    }
    return ret;
}
/**
 * Whether two intervals overlap
 */
function overlap(a, b) {
    return a.lower < b.upper && a.upper > b.lower;
}
function last(xs) {
    return xs[xs.length - 1];
}
/**
 * Locate the intervals that should have the alarm thresholds, by index.
 *
 * Pick the intervals on either side of the singleton "undefined" interval, or
 * pick the middle interval if there's no such interval.
 */
function findAlarmThresholds(intervals) {
    const gapIndex = intervals.findIndex(x => x.change === undefined);
    if (gapIndex !== -1) {
        return {
            lowerAlarmIntervalIndex: gapIndex > 0 ? gapIndex - 1 : undefined,
            upperAlarmIntervalIndex: gapIndex < intervals.length - 1 ? gapIndex + 1 : undefined,
        };
    }
    if (intervals.length === 1) {
        return { upperAlarmIntervalIndex: 0 };
    }
    const middleIndex = Math.floor(intervals.length / 2);
    return {
        lowerAlarmIntervalIndex: middleIndex - 1,
        upperAlarmIntervalIndex: middleIndex,
    };
}
exports.findAlarmThresholds = findAlarmThresholds;
//# sourceMappingURL=data:application/json;base64,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