"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Between = exports.BinaryCondition = exports.BeginsWith = exports.BaseKeyCondition = exports.shapeAddition = exports.SchemaMode = void 0;
/**
 * Utility enum for Schema class
 */
var SchemaMode;
(function (SchemaMode) {
    SchemaMode["FILE"] = "FILE";
    SchemaMode["CODE"] = "CODE";
})(SchemaMode = exports.SchemaMode || (exports.SchemaMode = {}));
;
/**
 * Generates an addition to the schema
 *
 * @param options the options to produced a stringfied addition
 *
 * @returns the following shape:
 *
 * ```
 * prefix name interfaces directives {
 *   field
 *   field
 *   ...
 * }
 * ```
 */
function shapeAddition(options) {
    const typeName = () => { return options.name ? ` ${options.name}` : ''; };
    const interfaces = generateInterfaces(options.interfaceTypes);
    const directives = generateDirectives({
        directives: options.directives,
        modes: options.modes,
    });
    return options.fields.reduce((acc, field) => `${acc}  ${field}\n`, `${options.prefix}${typeName()}${interfaces}${directives} {\n`) + '}';
}
exports.shapeAddition = shapeAddition;
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
exports.BaseKeyCondition = BaseKeyCondition;
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BeginsWith = BeginsWith;
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
exports.BinaryCondition = BinaryCondition;
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
exports.Between = Between;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility function to generate interfaces for object types
 *
 * @param interfaceTypes the interfaces this object type implements
 */
function generateInterfaces(interfaceTypes) {
    if (!interfaceTypes || interfaceTypes.length === 0)
        return '';
    return interfaceTypes.reduce((acc, interfaceType) => `${acc} ${interfaceType.name} &`, ' implements').slice(0, -2);
}
/**
 * Utility function to generate directives
 */
function generateDirectives(options) {
    if (!options.directives || options.directives.length === 0)
        return '';
    // reduce over all directives and get string version of the directive
    // pass in the auth modes for checks to happen on compile time
    return options.directives.reduce((acc, directive) => { var _a; return `${acc}${directive._bindToAuthModes(options.modes).toString()}${(_a = options.delimiter) !== null && _a !== void 0 ? _a : ' '}`; }, ' ').slice(0, -1);
}
//# sourceMappingURL=data:application/json;base64,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