"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualRouter = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const route_1 = require("./route");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualRouterBase extends cdk.Resource {
    /**
     * (experimental) Add a single route to the router.
     *
     * @experimental
     */
    addRoute(id, props) {
        const route = new route_1.Route(this, id, {
            ...props,
            routeName: id,
            mesh: this.mesh,
            virtualRouter: this,
        });
        return route;
    }
}
/**
 * @experimental
 */
class VirtualRouter extends VirtualRouterBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.virtualRouterName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.listeners = new Array();
        this.mesh = props.mesh;
        this.addListener(props.listener || { portMapping: { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP } });
        const router = new appmesh_generated_1.CfnVirtualRouter(this, 'Resource', {
            virtualRouterName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                listeners: this.listeners,
            },
        });
        this.virtualRouterName = this.getResourceNameAttribute(router.attrVirtualRouterName);
        this.virtualRouterArn = this.getResourceArnAttribute(router.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter`,
            resourceName: this.physicalName,
        });
    }
    /**
     * (experimental) Import an existing VirtualRouter given an ARN.
     *
     * @experimental
     */
    static fromVirtualRouterArn(scope, id, virtualRouterArn) {
        return new ImportedVirtualRouter(scope, id, { virtualRouterArn });
    }
    /**
     * (experimental) Import an existing VirtualRouter given names.
     *
     * @experimental
     */
    static fromVirtualRouterName(scope, id, meshName, virtualRouterName) {
        return new ImportedVirtualRouter(scope, id, { meshName, virtualRouterName });
    }
    /**
     * (experimental) Import an existing virtual router given attributes.
     *
     * @experimental
     */
    static fromVirtualRouterAttributes(scope, id, attrs) {
        return new ImportedVirtualRouter(scope, id, attrs);
    }
    /**
     * Add port mappings to the router
     */
    addListener(listener) {
        this.listeners.push({
            portMapping: listener.portMapping,
        });
    }
}
exports.VirtualRouter = VirtualRouter;
/**
 * Used to import a VirtualRouter and perform actions or read properties from
 */
class ImportedVirtualRouter extends VirtualRouterBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.mesh) {
            this._mesh = props.mesh;
        }
        if (props.meshName) {
            if (props.mesh) {
                throw new Error('Supply either \'mesh\' or \'meshName\', but not both');
            }
            this._mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', props.meshName);
        }
        if (props.virtualRouterArn) {
            this.virtualRouterArn = props.virtualRouterArn;
            this.virtualRouterName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualRouterArn).resourceName));
        }
        else if (props.virtualRouterName && props.meshName) {
            this.virtualRouterName = props.virtualRouterName;
            this.virtualRouterArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter`,
                resourceName: this.virtualRouterName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
    /**
     * The AppMesh mesh the VirtualRouter belongs to
     */
    get mesh() {
        if (!this._mesh) {
            throw new Error('Please supply either \'mesh\' or \'meshName\' when calling \'fromVirtualRouterAttributes\'');
        }
        return this._mesh;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlydHVhbC1yb3V0ZXIuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ2aXJ0dWFsLXJvdXRlci50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxrQ0FBa0MsQ0FBQyxnREFBZ0Q7QUFFbkYsMkRBQXVEO0FBQ3ZELGlDQUFxQztBQUNyQyxtQ0FBZ0Q7QUFDaEQsMkRBQTREO0FBb0Q1RCxNQUFlLGlCQUFrQixTQUFRLEdBQUcsQ0FBQyxRQUFROzs7Ozs7SUFnQjFDLFFBQVEsQ0FBQyxFQUFVLEVBQUUsS0FBcUI7UUFDN0MsTUFBTSxLQUFLLEdBQUcsSUFBSSxhQUFLLENBQUMsSUFBSSxFQUFFLEVBQUUsRUFBRTtZQUM5QixHQUFHLEtBQUs7WUFDUixTQUFTLEVBQUUsRUFBRTtZQUNiLElBQUksRUFBRSxJQUFJLENBQUMsSUFBSTtZQUNmLGFBQWEsRUFBRSxJQUFJO1NBQ3RCLENBQUMsQ0FBQztRQUNILE9BQU8sS0FBSyxDQUFDO0lBQ2pCLENBQUM7Q0FDSjs7OztBQVVELE1BQWEsYUFBYyxTQUFRLGlCQUFpQjs7OztJQWdDaEQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF5QjtRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsaUJBQWlCLElBQUksR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUUsQ0FBQztTQUN2RyxDQUFDLENBQUM7UUFKVSxjQUFTLEdBQUcsSUFBSSxLQUFLLEVBQWtELENBQUM7UUFLckYsSUFBSSxDQUFDLElBQUksR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxXQUFXLENBQUMsS0FBSyxDQUFDLFFBQVEsSUFBSSxFQUFFLFdBQVcsRUFBRSxFQUFFLElBQUksRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLDRCQUFRLENBQUMsSUFBSSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQzdGLE1BQU0sTUFBTSxHQUFHLElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNsRCxpQkFBaUIsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUNwQyxRQUFRLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRO1lBQzVCLElBQUksRUFBRTtnQkFDRixTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVM7YUFDNUI7U0FDSixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsaUJBQWlCLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ3JGLElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtZQUM3RCxPQUFPLEVBQUUsU0FBUztZQUNsQixRQUFRLEVBQUUsUUFBUSxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsZ0JBQWdCO1lBQ3JELFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtTQUNsQyxDQUFDLENBQUM7SUFDUCxDQUFDOzs7Ozs7SUEvQ00sTUFBTSxDQUFDLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLGdCQUF3QjtRQUNyRixPQUFPLElBQUkscUJBQXFCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztJQUN0RSxDQUFDOzs7Ozs7SUFJTSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsUUFBZ0IsRUFBRSxpQkFBeUI7UUFDekcsT0FBTyxJQUFJLHFCQUFxQixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsRUFBRSxRQUFRLEVBQUUsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDO0lBQ2pGLENBQUM7Ozs7OztJQUlNLE1BQU0sQ0FBQywyQkFBMkIsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4QjtRQUNsRyxPQUFPLElBQUkscUJBQXFCLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN2RCxDQUFDO0lBa0NEOztPQUVHO0lBQ0ssV0FBVyxDQUFDLFFBQWtCO1FBQ2xDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDO1lBQ2hCLFdBQVcsRUFBRSxRQUFRLENBQUMsV0FBVztTQUNwQyxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUE1REQsc0NBNERDO0FBc0JEOztHQUVHO0FBQ0gsTUFBTSxxQkFBc0IsU0FBUSxpQkFBaUI7SUFVakQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4QjtRQUNwRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksS0FBSyxDQUFDLElBQUksRUFBRTtZQUNaLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztTQUMzQjtRQUNELElBQUksS0FBSyxDQUFDLFFBQVEsRUFBRTtZQUNoQixJQUFJLEtBQUssQ0FBQyxJQUFJLEVBQUU7Z0JBQ1osTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDO2FBQzNFO1lBQ0QsSUFBSSxDQUFDLEtBQUssR0FBRyxXQUFJLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQ2hFO1FBQ0QsSUFBSSxLQUFLLENBQUMsZ0JBQWdCLEVBQUU7WUFDeEIsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztZQUMvQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsR0FBRyxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRSxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsWUFBYSxDQUFDLENBQUMsQ0FBQztTQUNwSTthQUNJLElBQUksS0FBSyxDQUFDLGlCQUFpQixJQUFJLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDaEQsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztZQUNqRCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNqRCxPQUFPLEVBQUUsU0FBUztnQkFDbEIsUUFBUSxFQUFFLFFBQVEsS0FBSyxDQUFDLFFBQVEsZ0JBQWdCO2dCQUNoRCxZQUFZLEVBQUUsSUFBSSxDQUFDLGlCQUFpQjthQUN2QyxDQUFDLENBQUM7U0FDTjthQUNJO1lBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1NBQ3BEO0lBQ0wsQ0FBQztJQUNEOztPQUVHO0lBQ0gsSUFBVyxJQUFJO1FBQ1gsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLEVBQUU7WUFDYixNQUFNLElBQUksS0FBSyxDQUFDLDRGQUE0RixDQUFDLENBQUM7U0FDakg7UUFDRCxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUM7SUFDdEIsQ0FBQztDQUNKIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBDZm5WaXJ0dWFsUm91dGVyIH0gZnJvbSAnLi9hcHBtZXNoLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBJTWVzaCwgTWVzaCB9IGZyb20gJy4vbWVzaCc7XG5pbXBvcnQgeyBSb3V0ZSwgUm91dGVCYXNlUHJvcHMgfSBmcm9tICcuL3JvdXRlJztcbmltcG9ydCB7IFBvcnRNYXBwaW5nLCBQcm90b2NvbCB9IGZyb20gJy4vc2hhcmVkLWludGVyZmFjZXMnO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJVmlydHVhbFJvdXRlciBleHRlbmRzIGNkay5JUmVzb3VyY2Uge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2aXJ0dWFsUm91dGVyTmFtZTogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZpcnR1YWxSb3V0ZXJBcm46IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IG1lc2g6IElNZXNoO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIGFkZFJvdXRlKGlkOiBzdHJpbmcsIHByb3BzOiBSb3V0ZUJhc2VQcm9wcyk6IFJvdXRlO1xufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBWaXJ0dWFsUm91dGVyQmFzZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbGlzdGVuZXI/OiBMaXN0ZW5lcjtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2aXJ0dWFsUm91dGVyTmFtZT86IHN0cmluZztcbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIExpc3RlbmVyIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgcG9ydE1hcHBpbmc6IFBvcnRNYXBwaW5nO1xufVxuYWJzdHJhY3QgY2xhc3MgVmlydHVhbFJvdXRlckJhc2UgZXh0ZW5kcyBjZGsuUmVzb3VyY2UgaW1wbGVtZW50cyBJVmlydHVhbFJvdXRlciB7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIFZpcnR1YWxSb3V0ZXJcbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgdmlydHVhbFJvdXRlck5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgZm9yIHRoZSBWaXJ0dWFsUm91dGVyXG4gICAgICovXG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHZpcnR1YWxSb3V0ZXJBcm46IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgQXBwTWVzaCBtZXNoIHRoZSBWaXJ0dWFsUm91dGVyIGJlbG9uZ3MgdG9cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgbWVzaDogSU1lc2g7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZFJvdXRlKGlkOiBzdHJpbmcsIHByb3BzOiBSb3V0ZUJhc2VQcm9wcyk6IFJvdXRlIHtcbiAgICAgICAgY29uc3Qgcm91dGUgPSBuZXcgUm91dGUodGhpcywgaWQsIHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgcm91dGVOYW1lOiBpZCxcbiAgICAgICAgICAgIG1lc2g6IHRoaXMubWVzaCxcbiAgICAgICAgICAgIHZpcnR1YWxSb3V0ZXI6IHRoaXMsXG4gICAgICAgIH0pO1xuICAgICAgICByZXR1cm4gcm91dGU7XG4gICAgfVxufVxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBWaXJ0dWFsUm91dGVyUHJvcHMgZXh0ZW5kcyBWaXJ0dWFsUm91dGVyQmFzZVByb3BzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbWVzaDogSU1lc2g7XG59XG5leHBvcnQgY2xhc3MgVmlydHVhbFJvdXRlciBleHRlbmRzIFZpcnR1YWxSb3V0ZXJCYXNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIHN0YXRpYyBmcm9tVmlydHVhbFJvdXRlckFybihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCB2aXJ0dWFsUm91dGVyQXJuOiBzdHJpbmcpOiBJVmlydHVhbFJvdXRlciB7XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0ZWRWaXJ0dWFsUm91dGVyKHNjb3BlLCBpZCwgeyB2aXJ0dWFsUm91dGVyQXJuIH0pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIGZyb21WaXJ0dWFsUm91dGVyTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBtZXNoTmFtZTogc3RyaW5nLCB2aXJ0dWFsUm91dGVyTmFtZTogc3RyaW5nKTogSVZpcnR1YWxSb3V0ZXIge1xuICAgICAgICByZXR1cm4gbmV3IEltcG9ydGVkVmlydHVhbFJvdXRlcihzY29wZSwgaWQsIHsgbWVzaE5hbWUsIHZpcnR1YWxSb3V0ZXJOYW1lIH0pO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgc3RhdGljIGZyb21WaXJ0dWFsUm91dGVyQXR0cmlidXRlcyhzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBhdHRyczogVmlydHVhbFJvdXRlckF0dHJpYnV0ZXMpOiBJVmlydHVhbFJvdXRlciB7XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0ZWRWaXJ0dWFsUm91dGVyKHNjb3BlLCBpZCwgYXR0cnMpO1xuICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICBwdWJsaWMgcmVhZG9ubHkgdmlydHVhbFJvdXRlck5hbWU6IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSB2aXJ0dWFsUm91dGVyQXJuOiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHB1YmxpYyByZWFkb25seSBtZXNoOiBJTWVzaDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGxpc3RlbmVycyA9IG5ldyBBcnJheTxDZm5WaXJ0dWFsUm91dGVyLlZpcnR1YWxSb3V0ZXJMaXN0ZW5lclByb3BlcnR5PigpO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWaXJ0dWFsUm91dGVyUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLnZpcnR1YWxSb3V0ZXJOYW1lIHx8IGNkay5MYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5ub2RlLnVuaXF1ZUlkIH0pLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5tZXNoID0gcHJvcHMubWVzaDtcbiAgICAgICAgdGhpcy5hZGRMaXN0ZW5lcihwcm9wcy5saXN0ZW5lciB8fCB7IHBvcnRNYXBwaW5nOiB7IHBvcnQ6IDgwODAsIHByb3RvY29sOiBQcm90b2NvbC5IVFRQIH0gfSk7XG4gICAgICAgIGNvbnN0IHJvdXRlciA9IG5ldyBDZm5WaXJ0dWFsUm91dGVyKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIHZpcnR1YWxSb3V0ZXJOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgICAgICAgIG1lc2hOYW1lOiB0aGlzLm1lc2gubWVzaE5hbWUsXG4gICAgICAgICAgICBzcGVjOiB7XG4gICAgICAgICAgICAgICAgbGlzdGVuZXJzOiB0aGlzLmxpc3RlbmVycyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLnZpcnR1YWxSb3V0ZXJOYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUocm91dGVyLmF0dHJWaXJ0dWFsUm91dGVyTmFtZSk7XG4gICAgICAgIHRoaXMudmlydHVhbFJvdXRlckFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUocm91dGVyLnJlZiwge1xuICAgICAgICAgICAgc2VydmljZTogJ2FwcG1lc2gnLFxuICAgICAgICAgICAgcmVzb3VyY2U6IGBtZXNoLyR7cHJvcHMubWVzaC5tZXNoTmFtZX0vdmlydHVhbFJvdXRlcmAsXG4gICAgICAgICAgICByZXNvdXJjZU5hbWU6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIHBvcnQgbWFwcGluZ3MgdG8gdGhlIHJvdXRlclxuICAgICAqL1xuICAgIHByaXZhdGUgYWRkTGlzdGVuZXIobGlzdGVuZXI6IExpc3RlbmVyKSB7XG4gICAgICAgIHRoaXMubGlzdGVuZXJzLnB1c2goe1xuICAgICAgICAgICAgcG9ydE1hcHBpbmc6IGxpc3RlbmVyLnBvcnRNYXBwaW5nLFxuICAgICAgICB9KTtcbiAgICB9XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgaW50ZXJmYWNlIFZpcnR1YWxSb3V0ZXJBdHRyaWJ1dGVzIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSB2aXJ0dWFsUm91dGVyTmFtZT86IHN0cmluZztcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHZpcnR1YWxSb3V0ZXJBcm4/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IG1lc2g/OiBJTWVzaDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgbWVzaE5hbWU/OiBzdHJpbmc7XG59XG4vKipcbiAqIFVzZWQgdG8gaW1wb3J0IGEgVmlydHVhbFJvdXRlciBhbmQgcGVyZm9ybSBhY3Rpb25zIG9yIHJlYWQgcHJvcGVydGllcyBmcm9tXG4gKi9cbmNsYXNzIEltcG9ydGVkVmlydHVhbFJvdXRlciBleHRlbmRzIFZpcnR1YWxSb3V0ZXJCYXNlIHtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgVmlydHVhbFJvdXRlclxuICAgICAqL1xuICAgIHB1YmxpYyByZWFkb25seSB2aXJ0dWFsUm91dGVyTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBmb3IgdGhlIFZpcnR1YWxSb3V0ZXJcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgdmlydHVhbFJvdXRlckFybjogc3RyaW5nO1xuICAgIHByaXZhdGUgX21lc2g/OiBJTWVzaDtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogVmlydHVhbFJvdXRlckF0dHJpYnV0ZXMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgaWYgKHByb3BzLm1lc2gpIHtcbiAgICAgICAgICAgIHRoaXMuX21lc2ggPSBwcm9wcy5tZXNoO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5tZXNoTmFtZSkge1xuICAgICAgICAgICAgaWYgKHByb3BzLm1lc2gpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1N1cHBseSBlaXRoZXIgXFwnbWVzaFxcJyBvciBcXCdtZXNoTmFtZVxcJywgYnV0IG5vdCBib3RoJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICB0aGlzLl9tZXNoID0gTWVzaC5mcm9tTWVzaE5hbWUodGhpcywgJ01lc2gnLCBwcm9wcy5tZXNoTmFtZSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLnZpcnR1YWxSb3V0ZXJBcm4pIHtcbiAgICAgICAgICAgIHRoaXMudmlydHVhbFJvdXRlckFybiA9IHByb3BzLnZpcnR1YWxSb3V0ZXJBcm47XG4gICAgICAgICAgICB0aGlzLnZpcnR1YWxSb3V0ZXJOYW1lID0gY2RrLkZuLnNlbGVjdCgyLCBjZGsuRm4uc3BsaXQoJy8nLCBjZGsuU3RhY2sub2Yoc2NvcGUpLnBhcnNlQXJuKHByb3BzLnZpcnR1YWxSb3V0ZXJBcm4pLnJlc291cmNlTmFtZSEpKTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIGlmIChwcm9wcy52aXJ0dWFsUm91dGVyTmFtZSAmJiBwcm9wcy5tZXNoTmFtZSkge1xuICAgICAgICAgICAgdGhpcy52aXJ0dWFsUm91dGVyTmFtZSA9IHByb3BzLnZpcnR1YWxSb3V0ZXJOYW1lO1xuICAgICAgICAgICAgdGhpcy52aXJ0dWFsUm91dGVyQXJuID0gY2RrLlN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICAgICAgc2VydmljZTogJ2FwcG1lc2gnLFxuICAgICAgICAgICAgICAgIHJlc291cmNlOiBgbWVzaC8ke3Byb3BzLm1lc2hOYW1lfS92aXJ0dWFsUm91dGVyYCxcbiAgICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6IHRoaXMudmlydHVhbFJvdXRlck5hbWUsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignTmVlZCBlaXRoZXIgYXJuIG9yIGJvdGggbmFtZXMnKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgQXBwTWVzaCBtZXNoIHRoZSBWaXJ0dWFsUm91dGVyIGJlbG9uZ3MgdG9cbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0IG1lc2goKTogSU1lc2gge1xuICAgICAgICBpZiAoIXRoaXMuX21lc2gpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignUGxlYXNlIHN1cHBseSBlaXRoZXIgXFwnbWVzaFxcJyBvciBcXCdtZXNoTmFtZVxcJyB3aGVuIGNhbGxpbmcgXFwnZnJvbVZpcnR1YWxSb3V0ZXJBdHRyaWJ1dGVzXFwnJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMuX21lc2g7XG4gICAgfVxufVxuIl19