"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = exports.MethodLoggingLevel = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const access_log_1 = require("./access-log");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * @experimental
 */
var MethodLoggingLevel;
(function (MethodLoggingLevel) {
    MethodLoggingLevel["OFF"] = "OFF";
    MethodLoggingLevel["ERROR"] = "ERROR";
    MethodLoggingLevel["INFO"] = "INFO";
})(MethodLoggingLevel = exports.MethodLoggingLevel || (exports.MethodLoggingLevel = {}));
/**
 * @experimental
 */
class Stage extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.enableCacheCluster = props.cacheClusterEnabled;
        const methodSettings = this.renderMethodSettings(props); // this can mutate `this.cacheClusterEnabled`
        // custom access logging
        let accessLogSetting;
        const accessLogDestination = props.accessLogDestination;
        const accessLogFormat = props.accessLogFormat;
        if (!accessLogDestination && !accessLogFormat) {
            accessLogSetting = undefined;
        }
        else {
            if (accessLogFormat !== undefined &&
                !core_1.Token.isUnresolved(accessLogFormat.toString()) &&
                !/.*\$context.requestId.*/.test(accessLogFormat.toString())) {
                throw new Error('Access log must include at least `AccessLogFormat.contextRequestId()`');
            }
            if (accessLogFormat !== undefined && accessLogDestination === undefined) {
                throw new Error('Access log format is specified without a destination');
            }
            accessLogSetting = {
                destinationArn: accessLogDestination === null || accessLogDestination === void 0 ? void 0 : accessLogDestination.bind(this).destinationArn,
                format: (accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString()) ? accessLogFormat === null || accessLogFormat === void 0 ? void 0 : accessLogFormat.toString() : access_log_1.AccessLogFormat.clf().toString(),
            };
        }
        // enable cache cluster if cacheClusterSize is set
        if (props.cacheClusterSize !== undefined) {
            if (this.enableCacheCluster === undefined) {
                this.enableCacheCluster = true;
            }
            else if (this.enableCacheCluster === false) {
                throw new Error(`Cannot set "cacheClusterSize" to ${props.cacheClusterSize} and "cacheClusterEnabled" to "false"`);
            }
        }
        const cacheClusterSize = this.enableCacheCluster ? (props.cacheClusterSize || '0.5') : undefined;
        const resource = new apigateway_generated_1.CfnStage(this, 'Resource', {
            stageName: props.stageName || 'prod',
            accessLogSetting,
            cacheClusterEnabled: this.enableCacheCluster,
            cacheClusterSize,
            clientCertificateId: props.clientCertificateId,
            deploymentId: props.deployment.deploymentId,
            restApiId: props.deployment.api.restApiId,
            description: props.description,
            documentationVersion: props.documentationVersion,
            variables: props.variables,
            tracingEnabled: props.tracingEnabled,
            methodSettings,
        });
        this.stageName = resource.ref;
        this.restApi = props.deployment.api;
    }
    /**
     * (experimental) Returns the invoke URL for a certain path.
     *
     * @param path The resource path.
     * @experimental
     */
    urlForPath(path = '/') {
        if (!path.startsWith('/')) {
            throw new Error(`Path must begin with "/": ${path}`);
        }
        return `https://${this.restApi.restApiId}.execute-api.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}/${this.stageName}${path}`;
    }
    renderMethodSettings(props) {
        const settings = new Array();
        const self = this;
        // extract common method options from the stage props
        const commonMethodOptions = {
            metricsEnabled: props.metricsEnabled,
            loggingLevel: props.loggingLevel,
            dataTraceEnabled: props.dataTraceEnabled,
            throttlingBurstLimit: props.throttlingBurstLimit,
            throttlingRateLimit: props.throttlingRateLimit,
            cachingEnabled: props.cachingEnabled,
            cacheTtl: props.cacheTtl,
            cacheDataEncrypted: props.cacheDataEncrypted,
        };
        // if any of them are defined, add an entry for '/*/*'.
        const hasCommonOptions = Object.keys(commonMethodOptions).map(v => commonMethodOptions[v]).filter(x => x).length > 0;
        if (hasCommonOptions) {
            settings.push(renderEntry('/*/*', commonMethodOptions));
        }
        if (props.methodOptions) {
            for (const path of Object.keys(props.methodOptions)) {
                settings.push(renderEntry(path, props.methodOptions[path]));
            }
        }
        return settings.length === 0 ? undefined : settings;
        function renderEntry(path, options) {
            if (options.cachingEnabled) {
                if (self.enableCacheCluster === undefined) {
                    self.enableCacheCluster = true;
                }
                else if (self.enableCacheCluster === false) {
                    throw new Error(`Cannot enable caching for method ${path} since cache cluster is disabled on stage`);
                }
            }
            const { httpMethod, resourcePath } = util_1.parseMethodOptionsPath(path);
            return {
                httpMethod,
                resourcePath,
                cacheDataEncrypted: options.cacheDataEncrypted,
                cacheTtlInSeconds: options.cacheTtl && options.cacheTtl.toSeconds(),
                cachingEnabled: options.cachingEnabled,
                dataTraceEnabled: options.dataTraceEnabled,
                loggingLevel: options.loggingLevel,
                metricsEnabled: options.metricsEnabled,
                throttlingBurstLimit: options.throttlingBurstLimit,
                throttlingRateLimit: options.throttlingRateLimit,
            };
        }
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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