"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProxyResource = exports.Resource = exports.ResourceBase = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const cors_1 = require("./cors");
const integrations_1 = require("./integrations");
const method_1 = require("./method");
/**
 * @experimental
 */
class ResourceBase extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id) {
        super(scope, id);
        this.children = {};
    }
    /**
     * (experimental) Defines a new child resource where this resource is the parent.
     *
     * @experimental
     */
    addResource(pathPart, options) {
        return new Resource(this, pathPart, { parent: this, pathPart, ...options });
    }
    /**
     * (experimental) Defines a new method for this resource.
     *
     * @experimental
     */
    addMethod(httpMethod, integration, options) {
        return new method_1.Method(this, httpMethod, { resource: this, httpMethod, integration, options });
    }
    /**
     * (experimental) Adds a greedy proxy resource ("{proxy+}") and an ANY method to this route.
     *
     * @experimental
     */
    addProxy(options) {
        return new ProxyResource(this, '{proxy+}', { parent: this, ...options });
    }
    /**
     * (experimental) Adds an OPTIONS method to this resource which responds to Cross-Origin Resource Sharing (CORS) preflight requests.
     *
     * Cross-Origin Resource Sharing (CORS) is a mechanism that uses additional
     * HTTP headers to tell browsers to give a web application running at one
     * origin, access to selected resources from a different origin. A web
     * application executes a cross-origin HTTP request when it requests a
     * resource that has a different origin (domain, protocol, or port) from its
     * own.
     *
     * @experimental
     */
    addCorsPreflight(options) {
        const headers = {};
        //
        // Access-Control-Allow-Headers
        const allowHeaders = options.allowHeaders || cors_1.Cors.DEFAULT_HEADERS;
        headers['Access-Control-Allow-Headers'] = `'${allowHeaders.join(',')}'`;
        //
        // Access-Control-Allow-Origin
        if (options.allowOrigins.length === 0) {
            throw new Error('allowOrigins must contain at least one origin');
        }
        if (options.allowOrigins.includes('*') && options.allowOrigins.length > 1) {
            throw new Error(`Invalid "allowOrigins" - cannot mix "*" with specific origins: ${options.allowOrigins.join(',')}`);
        }
        // we use the first origin here and if there are more origins in the list, we
        // will match against them in the response velocity template
        const initialOrigin = options.allowOrigins[0];
        headers['Access-Control-Allow-Origin'] = `'${initialOrigin}'`;
        // the "Vary" header is required if we allow a specific origin
        // https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin#CORS_and_caching
        if (initialOrigin !== '*') {
            headers.Vary = '\'Origin\'';
        }
        //
        // Access-Control-Allow-Methods
        let allowMethods = options.allowMethods || cors_1.Cors.ALL_METHODS;
        if (allowMethods.includes('ANY')) {
            if (allowMethods.length > 1) {
                throw new Error(`ANY cannot be used with any other method. Received: ${allowMethods.join(',')}`);
            }
            allowMethods = cors_1.Cors.ALL_METHODS;
        }
        headers['Access-Control-Allow-Methods'] = `'${allowMethods.join(',')}'`;
        //
        // Access-Control-Allow-Credentials
        if (options.allowCredentials) {
            headers['Access-Control-Allow-Credentials'] = '\'true\'';
        }
        //
        // Access-Control-Max-Age
        let maxAgeSeconds;
        if (options.maxAge && options.disableCache) {
            throw new Error('The options "maxAge" and "disableCache" are mutually exclusive');
        }
        if (options.maxAge) {
            maxAgeSeconds = options.maxAge.toSeconds();
        }
        if (options.disableCache) {
            maxAgeSeconds = -1;
        }
        if (maxAgeSeconds) {
            headers['Access-Control-Max-Age'] = `'${maxAgeSeconds}'`;
        }
        //
        // Access-Control-Expose-Headers
        //
        if (options.exposeHeaders) {
            headers['Access-Control-Expose-Headers'] = `'${options.exposeHeaders.join(',')}'`;
        }
        //
        // statusCode
        const statusCode = options.statusCode !== undefined ? options.statusCode : 204;
        //
        // prepare responseParams
        const integrationResponseParams = {};
        const methodReponseParams = {};
        for (const [name, value] of Object.entries(headers)) {
            const key = `method.response.header.${name}`;
            integrationResponseParams[key] = value;
            methodReponseParams[key] = true;
        }
        return this.addMethod('OPTIONS', new integrations_1.MockIntegration({
            requestTemplates: { 'application/json': '{ statusCode: 200 }' },
            integrationResponses: [
                { statusCode: `${statusCode}`, responseParameters: integrationResponseParams, responseTemplates: renderResponseTemplate() },
            ],
        }), {
            methodResponses: [
                { statusCode: `${statusCode}`, responseParameters: methodReponseParams },
            ],
        });
        // renders the response template to match all possible origins (if we have more than one)
        function renderResponseTemplate() {
            const origins = options.allowOrigins.slice(1);
            if (origins.length === 0) {
                return undefined;
            }
            const template = new Array();
            template.push('#set($origin = $input.params("Origin"))');
            template.push('#if($origin == "") #set($origin = $input.params("origin")) #end');
            const condition = origins.map(o => `$origin.matches("${o}")`).join(' || ');
            template.push(`#if(${condition})`);
            template.push('  #set($context.responseOverride.header.Access-Control-Allow-Origin = $origin)');
            template.push('#end');
            return {
                'application/json': template.join('\n'),
            };
        }
    }
    /**
     * (experimental) Retrieves a child resource by path part.
     *
     * @experimental
     */
    getResource(pathPart) {
        return this.children[pathPart];
    }
    /**
     * @internal
     */
    _trackChild(pathPart, resource) {
        this.children[pathPart] = resource;
    }
    /**
     * (experimental) Gets or create all resources leading up to the specified path.
     *
     * - Path may only start with "/" if this method is called on the root resource.
     * - All resources are created using default options.
     *
     * @experimental
     */
    resourceForPath(path) {
        if (!path) {
            return this;
        }
        if (path.startsWith('/')) {
            if (this.path !== '/') {
                throw new Error(`Path may start with "/" only for the resource, but we are at: ${this.path}`);
            }
            // trim trailing "/"
            return this.resourceForPath(path.substr(1));
        }
        const parts = path.split('/');
        const next = parts.shift();
        if (!next || next === '') {
            throw new Error('resourceForPath cannot be called with an empty path');
        }
        let resource = this.getResource(next);
        if (!resource) {
            resource = this.addResource(next);
        }
        return resource.resourceForPath(parts.join('/'));
    }
    /**
     * @deprecated - Throws error in some use cases that have been enabled since this deprecation notice. Use `RestApi.urlForPath()` instead.
     */
    get url() {
        return this.restApi.urlForPath(this.path);
    }
}
exports.ResourceBase = ResourceBase;
/**
 * @experimental
 */
class Resource extends ResourceBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id);
        validateResourcePathPart(props.pathPart);
        this.parentResource = props.parent;
        if (props.parent instanceof ResourceBase) {
            props.parent._trackChild(props.pathPart, this);
        }
        const resourceProps = {
            restApiId: props.parent.api.restApiId,
            parentId: props.parent.resourceId,
            pathPart: props.pathPart,
        };
        const resource = new apigateway_generated_1.CfnResource(this, 'Resource', resourceProps);
        this.resourceId = resource.ref;
        this.api = props.parent.api;
        // render resource path (special case for root)
        this.path = props.parent.path;
        if (!this.path.endsWith('/')) {
            this.path += '/';
        }
        this.path += props.pathPart;
        const deployment = props.parent.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({ resource: resourceProps });
        }
        // setup defaults based on properties and inherit from parent. method defaults
        // are inherited per property, so children can override piecemeal.
        this.defaultIntegration = props.defaultIntegration || props.parent.defaultIntegration;
        this.defaultMethodOptions = {
            ...props.parent.defaultMethodOptions,
            ...props.defaultMethodOptions,
        };
        this.defaultCorsPreflightOptions = props.defaultCorsPreflightOptions || props.parent.defaultCorsPreflightOptions;
        if (this.defaultCorsPreflightOptions) {
            this.addCorsPreflight(this.defaultCorsPreflightOptions);
        }
    }
    /**
     * (deprecated) The RestApi associated with this Resource.
     *
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        if (!this.parentResource) {
            throw new Error('parentResource was unexpectedly not defined');
        }
        return this.parentResource.restApi;
    }
}
exports.Resource = Resource;
/**
 * (experimental) Defines a {proxy+} greedy resource and an ANY method on a route.
 *
 * @see https://docs.aws.amazon.com/apigateway/latest/developerguide/api-gateway-set-up-simple-proxy.html
 * @experimental
 */
class ProxyResource extends Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            parent: props.parent,
            pathPart: '{proxy+}',
            defaultIntegration: props.defaultIntegration,
            defaultMethodOptions: props.defaultMethodOptions,
        });
        const anyMethod = props.anyMethod !== undefined ? props.anyMethod : true;
        if (anyMethod) {
            this.anyMethod = this.addMethod('ANY');
        }
    }
    /**
     * (experimental) Defines a new method for this resource.
     *
     * @experimental
     */
    addMethod(httpMethod, integration, options) {
        // In case this proxy is mounted under the root, also add this method to
        // the root so that empty paths are proxied as well.
        if (this.parentResource && this.parentResource.path === '/') {
            // skip if the root resource already has this method defined
            if (!(this.parentResource.node.tryFindChild(httpMethod) instanceof method_1.Method)) {
                this.parentResource.addMethod(httpMethod, integration, options);
            }
        }
        return super.addMethod(httpMethod, integration, options);
    }
}
exports.ProxyResource = ProxyResource;
function validateResourcePathPart(part) {
    // strip {} which indicate this is a parameter
    if (part.startsWith('{') && part.endsWith('}')) {
        part = part.substr(1, part.length - 2);
        // proxy resources are allowed to end with a '+'
        if (part.endsWith('+')) {
            part = part.substr(0, part.length - 1);
        }
    }
    if (!/^[a-zA-Z0-9\.\_\-]+$/.test(part)) {
        throw new Error(`Resource's path part only allow [a-zA-Z0-9._-], an optional trailing '+'
      and curly braces at the beginning and the end: ${part}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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