"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
/**
 * @experimental
 */
class Method extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const integration = (_b = (_a = props.integration) !== null && _a !== void 0 ? _a : this.resource.defaultIntegration) !== null && _b !== void 0 ? _b : new mock_1.MockIntegration();
        const bindResult = integration.bind(this);
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(bindResult),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_c = options.authorizationScopes) !== null && _c !== void 0 ? _c : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                method: {
                    ...methodProps,
                    integrationToken: bindResult === null || bindResult === void 0 ? void 0 : bindResult.deploymentToken,
                },
            });
        }
    }
    /**
     * (deprecated) The RestApi associated with this Method.
     *
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * (experimental) Returns an execute-api ARN for this method:.
     *
     * arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled, or can be explicitly assigned.
     * When not configured, {stage} will be set to '*', as a shorthand for 'all stages'.
     *
     * @experimental
     * @attribute true
     */
    get methodArn() {
        var _a;
        const stage = (_a = this.api.deploymentStage) === null || _a === void 0 ? void 0 : _a.stageName;
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * (experimental) Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     *
     * This stage is used by the AWS Console UI when testing the method.
     *
     * @experimental
     */
    get testMethodArn() {
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(bindResult) {
        var _a;
        const options = (_a = bindResult.options) !== null && _a !== void 0 ? _a : {};
        let credentials;
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // eslint-disable-next-line max-len
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: bindResult.type,
            uri: bindResult.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: bindResult.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
/**
 * @experimental
 */
var AuthorizationType;
(function (AuthorizationType) {
    AuthorizationType["NONE"] = "NONE";
    AuthorizationType["IAM"] = "AWS_IAM";
    AuthorizationType["CUSTOM"] = "CUSTOM";
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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