"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RequestAuthorizer = exports.TokenAuthorizer = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("../apigateway.generated");
const authorizer_1 = require("../authorizer");
class LambdaAuthorizer extends authorizer_1.Authorizer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.handler = props.handler;
        this.role = props.assumeRole;
        if (props.resultsCacheTtl && ((_a = props.resultsCacheTtl) === null || _a === void 0 ? void 0 : _a.toSeconds()) > 3600) {
            throw new Error('Lambda authorizer property \'resultsCacheTtl\' must not be greater than 3600 seconds (1 hour)');
        }
    }
    /**
     * Attaches this authorizer to a specific REST API.
     * @internal
     */
    _attachToApi(restApi) {
        if (this.restApiId && this.restApiId !== restApi.restApiId) {
            throw new Error('Cannot attach authorizer to two different rest APIs');
        }
        this.restApiId = restApi.restApiId;
    }
    /**
     * (experimental) Sets up the permissions necessary for the API Gateway service to invoke the Lambda function.
     *
     * @experimental
     */
    setupPermissions() {
        if (!this.role) {
            this.handler.addPermission(`${this.node.uniqueId}:Permissions`, {
                principal: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                sourceArn: this.authorizerArn,
            });
        }
        else if (this.role instanceof iam.Role) { // i.e. not imported
            this.role.attachInlinePolicy(new iam.Policy(this, 'authorizerInvokePolicy', {
                statements: [
                    new iam.PolicyStatement({
                        resources: [this.handler.functionArn],
                        actions: ['lambda:InvokeFunction'],
                    }),
                ],
            }));
        }
    }
    /**
     * (experimental) Returns a token that resolves to the Rest Api Id at the time of synthesis.
     *
     * Throws an error, during token resolution, if no RestApi is attached to this authorizer.
     *
     * @experimental
     */
    lazyRestApiId() {
        return core_1.Lazy.stringValue({
            produce: () => {
                if (!this.restApiId) {
                    throw new Error(`Authorizer (${this.node.path}) must be attached to a RestApi`);
                }
                return this.restApiId;
            },
        });
    }
}
/**
 * (experimental) Token based lambda authorizer that recognizes the caller's identity as a bearer token, such as a JSON Web Token (JWT) or an OAuth token.
 *
 * Based on the token, authorization is performed by a lambda function.
 *
 * @experimental
 * @resource AWS::ApiGateway::Authorizer
 */
class TokenAuthorizer extends LambdaAuthorizer {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'TOKEN',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySource || 'method.request.header.Authorization',
            identityValidationExpression: props.validationRegex,
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.TokenAuthorizer = TokenAuthorizer;
/**
 * (experimental) Request-based lambda authorizer that recognizes the caller's identity via request parameters, such as headers, paths, query strings, stage variables, or context variables.
 *
 * Based on the request, authorization is performed by a lambda function.
 *
 * @experimental
 * @resource AWS::ApiGateway::Authorizer
 */
class RequestAuthorizer extends LambdaAuthorizer {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id, props);
        if ((props.resultsCacheTtl === undefined || props.resultsCacheTtl.toSeconds() !== 0) && props.identitySources.length === 0) {
            throw new Error('At least one Identity Source is required for a REQUEST-based Lambda authorizer if caching is enabled.');
        }
        const restApiId = this.lazyRestApiId();
        const resource = new apigateway_generated_1.CfnAuthorizer(this, 'Resource', {
            name: (_a = props.authorizerName) !== null && _a !== void 0 ? _a : this.node.uniqueId,
            restApiId,
            type: 'REQUEST',
            authorizerUri: lambdaAuthorizerArn(props.handler),
            authorizerCredentials: (_b = props.assumeRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            authorizerResultTtlInSeconds: (_c = props.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            identitySource: props.identitySources.map(is => is.toString()).join(','),
        });
        this.authorizerId = resource.ref;
        this.authorizerArn = core_1.Stack.of(this).formatArn({
            service: 'execute-api',
            resource: restApiId,
            resourceName: `authorizers/${this.authorizerId}`,
        });
        this.setupPermissions();
    }
}
exports.RequestAuthorizer = RequestAuthorizer;
/**
 * constructs the authorizerURIArn.
 */
function lambdaAuthorizerArn(handler) {
    return `arn:${core_1.Stack.of(handler).partition}:apigateway:${core_1.Stack.of(handler).region}:lambda:path/2015-03-31/functions/${handler.functionArn}/invocations`;
}
//# sourceMappingURL=data:application/json;base64,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