"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PipelineDeployStackAction = void 0;
const cfn = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const cpactions = require("../../aws-codepipeline-actions"); // Automatically re-written from '@aws-cdk/aws-codepipeline-actions'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * (experimental) A class to deploy a stack that is part of a CDK App, using CodePipeline.
 *
 * This composite Action takes care of preparing and executing a CloudFormation ChangeSet.
 *
 * It currently does *not* support stacks that make use of ``Asset``s, and
 * requires the deployed stack is in the same account and region where the
 * CodePipeline is hosted.
 *
 * @experimental
 */
class PipelineDeployStackAction {
    /**
     * @experimental
     */
    constructor(props) {
        var _a, _b;
        this.stack = props.stack;
        const assets = this.stack.node.metadata.filter(md => md.type === cxschema.ArtifactMetadataEntryType.ASSET);
        if (assets.length > 0) {
            // FIXME: Implement the necessary actions to publish assets
            throw new Error(`Cannot deploy the stack ${this.stack.stackName} because it references ${assets.length} asset(s)`);
        }
        const createChangeSetRunOrder = props.createChangeSetRunOrder || 1;
        const executeChangeSetRunOrder = props.executeChangeSetRunOrder || (createChangeSetRunOrder + 1);
        if (createChangeSetRunOrder >= executeChangeSetRunOrder) {
            throw new Error(`createChangeSetRunOrder (${createChangeSetRunOrder}) must be < executeChangeSetRunOrder (${executeChangeSetRunOrder})`);
        }
        const changeSetName = props.changeSetName || 'CDK-CodePipeline-ChangeSet';
        const capabilities = cfnCapabilities(props.adminPermissions, props.capabilities);
        this.prepareChangeSetAction = new cpactions.CloudFormationCreateReplaceChangeSetAction({
            actionName: (_a = props.createChangeSetActionName) !== null && _a !== void 0 ? _a : 'ChangeSet',
            changeSetName,
            runOrder: createChangeSetRunOrder,
            stackName: props.stack.stackName,
            templatePath: props.input.atPath(props.stack.templateFile),
            adminPermissions: props.adminPermissions,
            deploymentRole: props.role,
            capabilities,
        });
        this.executeChangeSetAction = new cpactions.CloudFormationExecuteChangeSetAction({
            actionName: (_b = props.executeChangeSetActionName) !== null && _b !== void 0 ? _b : 'Execute',
            changeSetName,
            runOrder: executeChangeSetRunOrder,
            stackName: this.stack.stackName,
        });
    }
    /**
     * @experimental
     */
    bind(scope, stage, options) {
        if (this.stack.environment !== cdk.Stack.of(scope).environment) {
            // FIXME: Add the necessary to extend to stacks in a different account
            throw new Error('Cross-environment deployment is not supported');
        }
        stage.addAction(this.prepareChangeSetAction);
        this._deploymentRole = this.prepareChangeSetAction.deploymentRole;
        return this.executeChangeSetAction.bind(scope, stage, options);
    }
    /**
     * @experimental
     */
    get deploymentRole() {
        if (!this._deploymentRole) {
            throw new Error('Use this action in a pipeline first before accessing \'deploymentRole\'');
        }
        return this._deploymentRole;
    }
    /**
     * (experimental) Add policy statements to the role deploying the stack.
     *
     * This role is passed to CloudFormation and must have the IAM permissions
     * necessary to deploy the stack or you can grant this role `adminPermissions`
     * by using that option during creation. If you do not grant
     * `adminPermissions` you need to identify the proper statements to add to
     * this role based on the CloudFormation Resources in your stack.
     *
     * @experimental
     */
    addToDeploymentRolePolicy(statement) {
        this.deploymentRole.addToPolicy(statement);
    }
    /**
     * @experimental
     */
    onStateChange(name, target, options) {
        return this.executeChangeSetAction.onStateChange(name, target, options);
    }
    /**
     * @experimental
     */
    get actionProperties() {
        return this.executeChangeSetAction.actionProperties;
    }
}
exports.PipelineDeployStackAction = PipelineDeployStackAction;
function cfnCapabilities(adminPermissions, capabilities) {
    if (adminPermissions && capabilities === undefined) {
        // admin true default capability to NamedIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.NAMED_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else if (capabilities === undefined) {
        // else capabilities are undefined set AnonymousIAM and AutoExpand
        return [cfn.CloudFormationCapabilities.ANONYMOUS_IAM, cfn.CloudFormationCapabilities.AUTO_EXPAND];
    }
    else {
        // else capabilities are defined use them
        return capabilities;
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicGlwZWxpbmUtZGVwbG95LXN0YWNrLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInBpcGVsaW5lLWRlcGxveS1zdGFjay1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsZ0RBQWdELENBQUMsOERBQThEO0FBRS9HLDREQUE0RCxDQUFDLG9FQUFvRTtBQUdqSSx3REFBd0QsQ0FBQyxpRUFBaUU7QUFDMUgsa0NBQWtDLENBQUMsZ0RBQWdEOzs7Ozs7Ozs7Ozs7QUF3Rm5GLE1BQWEseUJBQXlCOzs7O0lBUWxDLFlBQVksS0FBcUM7O1FBQzdDLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQztRQUN6QixNQUFNLE1BQU0sR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxFQUFFLENBQUMsRUFBRSxDQUFDLElBQUksS0FBSyxRQUFRLENBQUMseUJBQXlCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0csSUFBSSxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNuQiwyREFBMkQ7WUFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLDBCQUEwQixNQUFNLENBQUMsTUFBTSxXQUFXLENBQUMsQ0FBQztTQUN0SDtRQUNELE1BQU0sdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixJQUFJLENBQUMsQ0FBQztRQUNuRSxNQUFNLHdCQUF3QixHQUFHLEtBQUssQ0FBQyx3QkFBd0IsSUFBSSxDQUFDLHVCQUF1QixHQUFHLENBQUMsQ0FBQyxDQUFDO1FBQ2pHLElBQUksdUJBQXVCLElBQUksd0JBQXdCLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0QkFBNEIsdUJBQXVCLHlDQUF5Qyx3QkFBd0IsR0FBRyxDQUFDLENBQUM7U0FDNUk7UUFDRCxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxJQUFJLDRCQUE0QixDQUFDO1FBQzFFLE1BQU0sWUFBWSxHQUFHLGVBQWUsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ2pGLElBQUksQ0FBQyxzQkFBc0IsR0FBRyxJQUFJLFNBQVMsQ0FBQywwQ0FBMEMsQ0FBQztZQUNuRixVQUFVLFFBQUUsS0FBSyxDQUFDLHlCQUF5QixtQ0FBSSxXQUFXO1lBQzFELGFBQWE7WUFDYixRQUFRLEVBQUUsdUJBQXVCO1lBQ2pDLFNBQVMsRUFBRSxLQUFLLENBQUMsS0FBSyxDQUFDLFNBQVM7WUFDaEMsWUFBWSxFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO1lBQzFELGdCQUFnQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7WUFDeEMsY0FBYyxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQzFCLFlBQVk7U0FDZixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsc0JBQXNCLEdBQUcsSUFBSSxTQUFTLENBQUMsb0NBQW9DLENBQUM7WUFDN0UsVUFBVSxRQUFFLEtBQUssQ0FBQywwQkFBMEIsbUNBQUksU0FBUztZQUN6RCxhQUFhO1lBQ2IsUUFBUSxFQUFFLHdCQUF3QjtZQUNsQyxTQUFTLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTO1NBQ2xDLENBQUMsQ0FBQztJQUNQLENBQUM7Ozs7SUFDTSxJQUFJLENBQUMsS0FBb0IsRUFBRSxLQUEwQixFQUFFLE9BQXVDO1FBQ2pHLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxXQUFXLEtBQUssR0FBRyxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUMsV0FBVyxFQUFFO1lBQzVELHNFQUFzRTtZQUN0RSxNQUFNLElBQUksS0FBSyxDQUFDLCtDQUErQyxDQUFDLENBQUM7U0FDcEU7UUFDRCxLQUFLLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxDQUFDO1FBQzdDLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLGNBQWMsQ0FBQztRQUNsRSxPQUFPLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNuRSxDQUFDOzs7O0lBQ0QsSUFBVyxjQUFjO1FBQ3JCLElBQUksQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFO1lBQ3ZCLE1BQU0sSUFBSSxLQUFLLENBQUMseUVBQXlFLENBQUMsQ0FBQztTQUM5RjtRQUNELE9BQU8sSUFBSSxDQUFDLGVBQWUsQ0FBQztJQUNoQyxDQUFDOzs7Ozs7Ozs7Ozs7SUFVTSx5QkFBeUIsQ0FBQyxTQUE4QjtRQUMzRCxJQUFJLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUMvQyxDQUFDOzs7O0lBQ00sYUFBYSxDQUFDLElBQVksRUFBRSxNQUEyQixFQUFFLE9BQTBCO1FBQ3RGLE9BQU8sSUFBSSxDQUFDLHNCQUFzQixDQUFDLGFBQWEsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQzVFLENBQUM7Ozs7SUFDRCxJQUFXLGdCQUFnQjtRQUN2QixPQUFPLElBQUksQ0FBQyxzQkFBc0IsQ0FBQyxnQkFBZ0IsQ0FBQztJQUN4RCxDQUFDO0NBQ0o7QUF4RUQsOERBd0VDO0FBQ0QsU0FBUyxlQUFlLENBQUMsZ0JBQXlCLEVBQUUsWUFBK0M7SUFDL0YsSUFBSSxnQkFBZ0IsSUFBSSxZQUFZLEtBQUssU0FBUyxFQUFFO1FBQ2hELDJEQUEyRDtRQUMzRCxPQUFPLENBQUMsR0FBRyxDQUFDLDBCQUEwQixDQUFDLFNBQVMsRUFBRSxHQUFHLENBQUMsMEJBQTBCLENBQUMsV0FBVyxDQUFDLENBQUM7S0FDakc7U0FDSSxJQUFJLFlBQVksS0FBSyxTQUFTLEVBQUU7UUFDakMsa0VBQWtFO1FBQ2xFLE9BQU8sQ0FBQyxHQUFHLENBQUMsMEJBQTBCLENBQUMsYUFBYSxFQUFFLEdBQUcsQ0FBQywwQkFBMEIsQ0FBQyxXQUFXLENBQUMsQ0FBQztLQUNyRztTQUNJO1FBQ0QseUNBQXlDO1FBQ3pDLE9BQU8sWUFBWSxDQUFDO0tBQ3ZCO0FBQ0wsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNmbiBmcm9tIFwiLi4vLi4vYXdzLWNsb3VkZm9ybWF0aW9uXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWRmb3JtYXRpb24nXG5pbXBvcnQgKiBhcyBjb2RlcGlwZWxpbmUgZnJvbSBcIi4uLy4uL2F3cy1jb2RlcGlwZWxpbmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnXG5pbXBvcnQgKiBhcyBjcGFjdGlvbnMgZnJvbSBcIi4uLy4uL2F3cy1jb2RlcGlwZWxpbmUtYWN0aW9uc1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZS1hY3Rpb25zJ1xuaW1wb3J0ICogYXMgZXZlbnRzIGZyb20gXCIuLi8uLi9hd3MtZXZlbnRzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzJ1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0ICogYXMgY3hzY2hlbWEgZnJvbSBcIi4uLy4uL2Nsb3VkLWFzc2VtYmx5LXNjaGVtYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY2xvdWQtYXNzZW1ibHktc2NoZW1hJ1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuZXhwb3J0IGludGVyZmFjZSBQaXBlbGluZURlcGxveVN0YWNrQWN0aW9uUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IHN0YWNrOiBjZGsuU3RhY2s7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgaW5wdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGNoYW5nZVNldE5hbWU/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgY3JlYXRlQ2hhbmdlU2V0UnVuT3JkZXI/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBjcmVhdGVDaGFuZ2VTZXRBY3Rpb25OYW1lPzogc3RyaW5nO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBleGVjdXRlQ2hhbmdlU2V0UnVuT3JkZXI/OiBudW1iZXI7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcmVhZG9ubHkgZXhlY3V0ZUNoYW5nZVNldEFjdGlvbk5hbWU/OiBzdHJpbmc7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgICByZWFkb25seSBjYXBhYmlsaXRpZXM/OiBjZm4uQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXNbXTtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGFkbWluUGVybWlzc2lvbnM6IGJvb2xlYW47XG59XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBQaXBlbGluZURlcGxveVN0YWNrQWN0aW9uIGltcGxlbWVudHMgY29kZXBpcGVsaW5lLklBY3Rpb24ge1xuICAgIC8qKlxuICAgICAqIFRoZSByb2xlIHVzZWQgYnkgQ2xvdWRGb3JtYXRpb24gZm9yIHRoZSBkZXBsb3kgYWN0aW9uXG4gICAgICovXG4gICAgcHJpdmF0ZSBfZGVwbG95bWVudFJvbGU/OiBpYW0uSVJvbGU7XG4gICAgcHJpdmF0ZSByZWFkb25seSBzdGFjazogY2RrLlN0YWNrO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcHJlcGFyZUNoYW5nZVNldEFjdGlvbjogY3BhY3Rpb25zLkNsb3VkRm9ybWF0aW9uQ3JlYXRlUmVwbGFjZUNoYW5nZVNldEFjdGlvbjtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb246IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb247XG4gICAgY29uc3RydWN0b3IocHJvcHM6IFBpcGVsaW5lRGVwbG95U3RhY2tBY3Rpb25Qcm9wcykge1xuICAgICAgICB0aGlzLnN0YWNrID0gcHJvcHMuc3RhY2s7XG4gICAgICAgIGNvbnN0IGFzc2V0cyA9IHRoaXMuc3RhY2subm9kZS5tZXRhZGF0YS5maWx0ZXIobWQgPT4gbWQudHlwZSA9PT0gY3hzY2hlbWEuQXJ0aWZhY3RNZXRhZGF0YUVudHJ5VHlwZS5BU1NFVCk7XG4gICAgICAgIGlmIChhc3NldHMubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgLy8gRklYTUU6IEltcGxlbWVudCB0aGUgbmVjZXNzYXJ5IGFjdGlvbnMgdG8gcHVibGlzaCBhc3NldHNcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGRlcGxveSB0aGUgc3RhY2sgJHt0aGlzLnN0YWNrLnN0YWNrTmFtZX0gYmVjYXVzZSBpdCByZWZlcmVuY2VzICR7YXNzZXRzLmxlbmd0aH0gYXNzZXQocylgKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBjcmVhdGVDaGFuZ2VTZXRSdW5PcmRlciA9IHByb3BzLmNyZWF0ZUNoYW5nZVNldFJ1bk9yZGVyIHx8IDE7XG4gICAgICAgIGNvbnN0IGV4ZWN1dGVDaGFuZ2VTZXRSdW5PcmRlciA9IHByb3BzLmV4ZWN1dGVDaGFuZ2VTZXRSdW5PcmRlciB8fCAoY3JlYXRlQ2hhbmdlU2V0UnVuT3JkZXIgKyAxKTtcbiAgICAgICAgaWYgKGNyZWF0ZUNoYW5nZVNldFJ1bk9yZGVyID49IGV4ZWN1dGVDaGFuZ2VTZXRSdW5PcmRlcikge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBjcmVhdGVDaGFuZ2VTZXRSdW5PcmRlciAoJHtjcmVhdGVDaGFuZ2VTZXRSdW5PcmRlcn0pIG11c3QgYmUgPCBleGVjdXRlQ2hhbmdlU2V0UnVuT3JkZXIgKCR7ZXhlY3V0ZUNoYW5nZVNldFJ1bk9yZGVyfSlgKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBjaGFuZ2VTZXROYW1lID0gcHJvcHMuY2hhbmdlU2V0TmFtZSB8fCAnQ0RLLUNvZGVQaXBlbGluZS1DaGFuZ2VTZXQnO1xuICAgICAgICBjb25zdCBjYXBhYmlsaXRpZXMgPSBjZm5DYXBhYmlsaXRpZXMocHJvcHMuYWRtaW5QZXJtaXNzaW9ucywgcHJvcHMuY2FwYWJpbGl0aWVzKTtcbiAgICAgICAgdGhpcy5wcmVwYXJlQ2hhbmdlU2V0QWN0aW9uID0gbmV3IGNwYWN0aW9ucy5DbG91ZEZvcm1hdGlvbkNyZWF0ZVJlcGxhY2VDaGFuZ2VTZXRBY3Rpb24oe1xuICAgICAgICAgICAgYWN0aW9uTmFtZTogcHJvcHMuY3JlYXRlQ2hhbmdlU2V0QWN0aW9uTmFtZSA/PyAnQ2hhbmdlU2V0JyxcbiAgICAgICAgICAgIGNoYW5nZVNldE5hbWUsXG4gICAgICAgICAgICBydW5PcmRlcjogY3JlYXRlQ2hhbmdlU2V0UnVuT3JkZXIsXG4gICAgICAgICAgICBzdGFja05hbWU6IHByb3BzLnN0YWNrLnN0YWNrTmFtZSxcbiAgICAgICAgICAgIHRlbXBsYXRlUGF0aDogcHJvcHMuaW5wdXQuYXRQYXRoKHByb3BzLnN0YWNrLnRlbXBsYXRlRmlsZSksXG4gICAgICAgICAgICBhZG1pblBlcm1pc3Npb25zOiBwcm9wcy5hZG1pblBlcm1pc3Npb25zLFxuICAgICAgICAgICAgZGVwbG95bWVudFJvbGU6IHByb3BzLnJvbGUsXG4gICAgICAgICAgICBjYXBhYmlsaXRpZXMsXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLmV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb24gPSBuZXcgY3BhY3Rpb25zLkNsb3VkRm9ybWF0aW9uRXhlY3V0ZUNoYW5nZVNldEFjdGlvbih7XG4gICAgICAgICAgICBhY3Rpb25OYW1lOiBwcm9wcy5leGVjdXRlQ2hhbmdlU2V0QWN0aW9uTmFtZSA/PyAnRXhlY3V0ZScsXG4gICAgICAgICAgICBjaGFuZ2VTZXROYW1lLFxuICAgICAgICAgICAgcnVuT3JkZXI6IGV4ZWN1dGVDaGFuZ2VTZXRSdW5PcmRlcixcbiAgICAgICAgICAgIHN0YWNrTmFtZTogdGhpcy5zdGFjay5zdGFja05hbWUsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICBwdWJsaWMgYmluZChzY29wZTogY2RrLkNvbnN0cnVjdCwgc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UsIG9wdGlvbnM6IGNvZGVwaXBlbGluZS5BY3Rpb25CaW5kT3B0aW9ucyk6IGNvZGVwaXBlbGluZS5BY3Rpb25Db25maWcge1xuICAgICAgICBpZiAodGhpcy5zdGFjay5lbnZpcm9ubWVudCAhPT0gY2RrLlN0YWNrLm9mKHNjb3BlKS5lbnZpcm9ubWVudCkge1xuICAgICAgICAgICAgLy8gRklYTUU6IEFkZCB0aGUgbmVjZXNzYXJ5IHRvIGV4dGVuZCB0byBzdGFja3MgaW4gYSBkaWZmZXJlbnQgYWNjb3VudFxuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdDcm9zcy1lbnZpcm9ubWVudCBkZXBsb3ltZW50IGlzIG5vdCBzdXBwb3J0ZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBzdGFnZS5hZGRBY3Rpb24odGhpcy5wcmVwYXJlQ2hhbmdlU2V0QWN0aW9uKTtcbiAgICAgICAgdGhpcy5fZGVwbG95bWVudFJvbGUgPSB0aGlzLnByZXBhcmVDaGFuZ2VTZXRBY3Rpb24uZGVwbG95bWVudFJvbGU7XG4gICAgICAgIHJldHVybiB0aGlzLmV4ZWN1dGVDaGFuZ2VTZXRBY3Rpb24uYmluZChzY29wZSwgc3RhZ2UsIG9wdGlvbnMpO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IGRlcGxveW1lbnRSb2xlKCk6IGlhbS5JUm9sZSB7XG4gICAgICAgIGlmICghdGhpcy5fZGVwbG95bWVudFJvbGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVXNlIHRoaXMgYWN0aW9uIGluIGEgcGlwZWxpbmUgZmlyc3QgYmVmb3JlIGFjY2Vzc2luZyBcXCdkZXBsb3ltZW50Um9sZVxcJycpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLl9kZXBsb3ltZW50Um9sZTtcbiAgICB9XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gICAgcHVibGljIGFkZFRvRGVwbG95bWVudFJvbGVQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KSB7XG4gICAgICAgIHRoaXMuZGVwbG95bWVudFJvbGUuYWRkVG9Qb2xpY3koc3RhdGVtZW50KTtcbiAgICB9XG4gICAgcHVibGljIG9uU3RhdGVDaGFuZ2UobmFtZTogc3RyaW5nLCB0YXJnZXQ/OiBldmVudHMuSVJ1bGVUYXJnZXQsIG9wdGlvbnM/OiBldmVudHMuUnVsZVByb3BzKTogZXZlbnRzLlJ1bGUge1xuICAgICAgICByZXR1cm4gdGhpcy5leGVjdXRlQ2hhbmdlU2V0QWN0aW9uLm9uU3RhdGVDaGFuZ2UobmFtZSwgdGFyZ2V0LCBvcHRpb25zKTtcbiAgICB9XG4gICAgcHVibGljIGdldCBhY3Rpb25Qcm9wZXJ0aWVzKCk6IGNvZGVwaXBlbGluZS5BY3Rpb25Qcm9wZXJ0aWVzIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZXhlY3V0ZUNoYW5nZVNldEFjdGlvbi5hY3Rpb25Qcm9wZXJ0aWVzO1xuICAgIH1cbn1cbmZ1bmN0aW9uIGNmbkNhcGFiaWxpdGllcyhhZG1pblBlcm1pc3Npb25zOiBib29sZWFuLCBjYXBhYmlsaXRpZXM/OiBjZm4uQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXNbXSk6IGNmbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllc1tdIHtcbiAgICBpZiAoYWRtaW5QZXJtaXNzaW9ucyAmJiBjYXBhYmlsaXRpZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAvLyBhZG1pbiB0cnVlIGRlZmF1bHQgY2FwYWJpbGl0eSB0byBOYW1lZElBTSBhbmQgQXV0b0V4cGFuZFxuICAgICAgICByZXR1cm4gW2Nmbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllcy5OQU1FRF9JQU0sIGNmbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllcy5BVVRPX0VYUEFORF07XG4gICAgfVxuICAgIGVsc2UgaWYgKGNhcGFiaWxpdGllcyA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIC8vIGVsc2UgY2FwYWJpbGl0aWVzIGFyZSB1bmRlZmluZWQgc2V0IEFub255bW91c0lBTSBhbmQgQXV0b0V4cGFuZFxuICAgICAgICByZXR1cm4gW2Nmbi5DbG91ZEZvcm1hdGlvbkNhcGFiaWxpdGllcy5BTk9OWU1PVVNfSUFNLCBjZm4uQ2xvdWRGb3JtYXRpb25DYXBhYmlsaXRpZXMuQVVUT19FWFBBTkRdO1xuICAgIH1cbiAgICBlbHNlIHtcbiAgICAgICAgLy8gZWxzZSBjYXBhYmlsaXRpZXMgYXJlIGRlZmluZWQgdXNlIHRoZW1cbiAgICAgICAgcmV0dXJuIGNhcGFiaWxpdGllcztcbiAgICB9XG59XG4iXX0=