# Copyright (C) 2019-2021 HERE Europe B.V.
# SPDX-License-Identifier: Apache-2.0

"""This module defines all the configs which will be required as inputs to Geocoding and Search
APIs."""

from .base_config import Bunch


class PlacesCategories(Bunch):
    """A class to define constant values for categories.

    The HERE places category system provides three levels of granularity:

    Level 1 - Categories that are organized into logical high-level groupings.
    For example, "100" represents the top level category for places associated with eating and
    drinking establishments (100="Eat and Drink").

    Level 2 - Categories organized by logical sub-groups or domains.
    For example, "100-1000" represents the mid-level category for places associated with eating
    and drinking establishments that are restaurants (100-1000="Eat and Drink / Restaurant").

    Level 3 - Categories that provide the greatest level of granularity about place categorization.
    For example, "100-1000-0001" is the complete category code for places associated with eating
    and drinking establishments that are casual dining restaurants (100-1000-0001="Eat and Drink
    / Restaurant / Casual Dining").

    Detail information about each category is given `here <https://developer.here.com/documentation/geocoding-search-api/dev_guide/topics-places/places-category-system-full.html>`_.
    """  # noqa E501


places_categories = {
    "restaurant": "100-1000-0000",
    "casual_dining": "100-1000-0001",
    "fine_dining": "100-1000-0002",
    "take_out_and_delivery_only": "100-1000-0003",
    "food_market-stall": "100-1000-0004",
    "taqueria": "100-1000-0005",
    "deli": "100-1000-0006",
    "cafeteria": "100-1000-0007",
    "bistro": "100-1000-0008",
    "fast_food": "100-1000-0009",
    "coffee_tea": "100-1100-0000",
    "coffee_shop": "100-1100-0010",
    "tea_house": "100-1100-0331",
    "nightlife_entertainment": "200-2000-0000",
    "bar_or_pub": "200-2000-0011",
    "night_club": "200-2000-0012",
    "dancing": "200-2000-0013",
    "karaoke": "200-2000-0014",
    "live_entertainment-music": "200-2000-0015",
    "billiards_pool_hall": "200-2000-0016",
    "video_arcade_game_room": "200-2000-0017",
    "jazz_club": "200-2000-0018",
    "beer_garden": "200-2000-0019",
    "adult_entertainment": "200-2000-0306",
    "cocktail_lounge": "200-2000-0368",
    "cinema": "200-2100-0019",
    "theatre_music_and_culture": "200-2200-0000",
    "performing_arts": "200-2200-0020",
    "gambling_lottery_betting": "200-2300-0000",
    "casino": "200-2300-0021",
    "lottery_booth": "200-2300-0022",
    "landmark_attraction": "300-3000-0000",
    "tourist_attraction": "300-3000-0023",
    "gallery": "300-3000-0024",
    "historical_monument": "300-3000-0025",
    "castle": "300-3000-0030",
    "winery": "300-3000-0065",
    "named_intersection-chowk": "300-3000-0232",
    "brewery": "300-3000-0350",
    "distillery": "300-3000-0351",
    "museum": "300-3100-0000",
    "science_museum": "300-3100-0026",
    "childrens_museum": "300-3100-0027",
    "history_museum": "300-3100-0028",
    "art_museum": "300-3100-0029",
    "religious_place": "300-3200-0000",
    "church": "300-3200-0030",
    "temple": "300-3200-0031",
    "synagogue": "300-3200-0032",
    "ashram": "300-3200-0033",
    "mosque": "300-3200-0034",
    "other_place_of_worship": "300-3200-0309",
    "gurdwara": "300-3200-0375",
    "pagoda": "300-3200-0376",
    "body_of_water": "350-3500-0233",
    "reservoir": "350-3500-0234",
    "waterfall": "350-3500-0235",
    "bay_harbor": "350-3500-0300",
    "river": "350-3500-0302",
    "canal": "350-3500-0303",
    "lake": "350-3500-0304",
    "mountain_or_hill": "350-3510-0236",
    "mountain_passes": "350-3510-0237",
    "mountain_peaks": "350-3510-0238",
    "undersea_feature": "350-3520-0224",
    "forest,_heath_or_other_vegetation": "350-3522-0239",
    "natural_and_geographical": "350-3550-0336",
    "public_sports_airport": "400-4000-4580",
    "airport": "400-4000-4581",
    "airport_terminal": "400-4000-4582",
    "train_station": "400-4100-0035",
    "bus_station": "400-4100-0036",
    "underground_train-subway": "400-4100-0037",
    "commuter_rail_station": "400-4100-0038",
    "commuter_train": "400-4100-0039",
    "public_transit_access": "400-4100-0040",
    "transportation_service": "400-4100-0041",
    "bus_stop": "400-4100-0042",
    "local_transit": "400-4100-0043",
    "ferry_terminal": "400-4100-0044",
    "boat_ferry": "400-4100-0045",
    "rail_ferry": "400-4100-0046",
    "taxi_stand": "400-4100-0047",
    "highway_exit": "400-4100-0226",
    "tollbooth": "400-4100-0326",
    "lightrail": "400-4100-0337",
    "water_transit": "400-4100-0338",
    "monorail": "400-4100-0339",
    "aerial_tramway": "400-4100-0340",
    "bus_rapid_transit": "400-4100-0341",
    "inclined_rail": "400-4100-0342",
    "bicycle_sharing_location": "400-4100-0347",
    "bicycle_parking": "400-4100-0348",
    "weigh_station": "400-4200-0048",
    "cargo_center": "400-4200-0049",
    "rail_yard": "400-4200-0050",
    "seaport_harbour": "400-4200-0051",
    "airport_cargo": "400-4200-0052",
    "couriers": "400-4200-0240",
    "cargo_transportation": "400-4200-0241",
    "delivery_entrance": "400-4200-0311",
    "loading_dock": "400-4200-0312",
    "loading_zone": "400-4200-0313",
    "rest_area": "400-4300-0000",
    "complete_rest_area": "400-4300-0199",
    "parking_and_restroom_only_rest_area": "400-4300-0200",
    "parking_only_rest_area": "400-4300-0201",
    "motorway_service_rest_area": "400-4300-0202",
    "scenic_overlook_rest_area": "400-4300-0308",
    "hotel_or_motel": "500-5000-0000",
    "hotel": "500-5000-0053",
    "motel": "500-5000-0054",
    "lodging": "500-5100-0000",
    "hostel": "500-5100-0055",
    "campground": "500-5100-0056",
    "guest_house": "500-5100-0057",
    "bed_and_breakfast": "500-5100-0058",
    "holiday_park": "500-5100-0059",
    "short_time_motel": "500-5100-0060",
    "ryokan": "500-5100-0061",
    "outdoor_recreation": "550-5510-0000",
    "park_recreation_area": "550-5510-0202",
    "sports_field": "550-5510-0203",
    "garden": "550-5510-0204",
    "beach": "550-5510-0205",
    "recreation_center": "550-5510-0206",
    "ski_lift": "550-5510-0227",
    "scenic_point": "550-5510-0242",
    "off_road_trailhead": "550-5510-0358",
    "trailhead": "550-5510-0359",
    "off_road_vehicle_area": "550-5510-0374",
    "campsite": "550-5510-0378",
    "outdoor_service": "550-5510-0379",
    "ranger_station": "550-5510-0380",
    "bicycle_service": "550-5510-0387",
    "leisure": "550-5520-0000",
    "amusement_park": "550-5520-0207",
    "zoo": "550-5520-0208",
    "wild_animal_park": "550-5520-0209",
    "wildlife_refuge": "550-5520-0210",
    "aquarium": "550-5520-0211",
    "ski_resort": "550-5520-0212",
    "animal_park": "550-5520-0228",
    "water_park": "550-5520-0357",
    "convenience_store": "600-6000-0061",
    "shopping_mall": "600-6100-0062",
    "department_store": "600-6200-0063",
    "food_beverage_specialty_store": "600-6300-0064",
    "grocery": "600-6300-0066",
    "specialty_food_store": "600-6300-0067",
    "wine_and_liquor": "600-6300-0068",
    "bakery_and_baked_goods_store": "600-6300-0244",
    "sweet_shop": "600-6300-0245",
    "doughnut_shop": "600-6300-0246",
    "butcher": "600-6300-0363",
    "dairy_goods": "600-6300-0364",
    "drugstore_or_pharmacy": "600-6400-0000",
    "drugstore": "600-6400-0069",
    "pharmacy": "600-6400-0070",
    "consumer_electronics_store": "600-6500-0072",
    "mobile_retailer": "600-6500-0073",
    "mobile_service_center": "600-6500-0074",
    "computer_and_software": "600-6500-0075",
    "entertainment_electronics": "600-6500-0076",
    "hardware,_house_and_garden": "600-6600-0000",
    "home_improvement": "600-6600-0077",
    "home_specialty_store": "600-6600-0078",
    "floor_and_carpet": "600-6600-0079",
    "furniture_store": "600-6600-0080",
    "garden_center": "600-6600-0082",
    "glass_and_window": "600-6600-0083",
    "lumber": "600-6600-0084",
    "major_appliance": "600-6600-0085",
    "power_equipment_dealer": "600-6600-0310",
    "paint_store": "600-6600-0319",
    "other_bookshop": "600-6700-0000",
    "bookstore": "600-6700-0087",
    "clothing_and_accessories": "600-6800-0000",
    "mens_apparel": "600-6800-0089",
    "womens_apparel": "600-6800-0090",
    "childrens_apparel": "600-6800-0091",
    "shoes_footwear": "600-6800-0092",
    "specialty_clothing_store": "600-6800-0093",
    "consumer_goods": "600-6900-0000",
    "sporting_goods_store": "600-6900-0094",
    "office_supply_and_services_store": "600-6900-0095",
    "specialty_store": "600-6900-0096",
    "pet_supply": "600-6900-0097",
    "warehouse_wholesale_store": "600-6900-0098",
    "general_merchandise": "600-6900-0099",
    "discount_store": "600-6900-0100",
    "flowers_and_jewelry": "600-6900-0101",
    "variety_store": "600-6900-0102",
    "gift,_antique_and_art": "600-6900-0103",
    "record,_cd_and_video": "600-6900-0105",
    "video_and_game_rental": "600-6900-0106",
    "cigar_and_tobacco_shop": "600-6900-0107",
    "vaping_store": "600-6900-0108",
    "bicycle_and_bicycle_accessories_shop": "600-6900-0246",
    "market": "600-6900-0247",
    "motorcycle_accessories": "600-6900-0248",
    "non_store_retailers": "600-6900-0249",
    "pawnshop": "600-6900-0250",
    "used_second_hand_merchandise_stores": "600-6900-0251",
    "adult_shop": "600-6900-0305",
    "arts_and_crafts_supplies": "600-6900-0307",
    "florist": "600-6900-0355",
    "jeweler": "600-6900-0356",
    "toy_store": "600-6900-0358",
    "hunting_fishing_shop": "600-6900-0388",
    "running_walking_shop": "600-6900-0389",
    "skate_shop": "600-6900-0390",
    "ski_shop": "600-6900-0391",
    "snowboard_shop": "600-6900-0392",
    "surf_shop": "600-6900-0393",
    "bmx_shop": "600-6900-0394",
    "camping_hiking_shop": "600-6900-0395",
    "canoe_kayak_shop": "600-6900-0396",
    "cross_country_ski_shop": "600-6900-0397",
    "tack_shop": "600-6900-0398",
    "hair_and_beauty": "600-6950-0000",
    "barber": "600-6950-0399",
    "nail_salon": "600-6950-0400",
    "hair_salon": "600-6950-0401",
    "bank": "700-7000-0107",
    "atm": "700-7010-0108",
    "money_transferring_service": "700-7050-0109",
    "check_cashing_service_currency_exchange": "700-7050-0110",
    "communication_media": "700-7100-0000",
    "telephone_service": "700-7100-0134",
    "commercial_services": "700-7200-0000",
    "advertising_marketing,_pr_and_market_research": "700-7200-0252",
    "catering_and_other_food_services": "700-7200-0253",
    "construction": "700-7200-0254",
    "customer_care_service_center": "700-7200-0255",
    "engineering_and_scientific_services": "700-7200-0256",
    "farming": "700-7200-0257",
    "food_production": "700-7200-0258",
    "human_resources_and_recruiting_services": "700-7200-0259",
    "investigation_services": "700-7200-0260",
    "it_and_office_equipment_services": "700-7200-0261",
    "landscaping_services": "700-7200-0262",
    "locksmiths_and_security_systems_services": "700-7200-0263",
    "management_and_consulting_services": "700-7200-0264",
    "manufacturing": "700-7200-0265",
    "mining": "700-7200-0266",
    "modeling_agencies": "700-7200-0267",
    "motorcycle_service_and_maintenance": "700-7200-0268",
    "organizations_and_societies": "700-7200-0269",
    "entertainment_and_recreation": "700-7200-0270",
    "finance_and_insurance": "700-7200-0271",
    "healthcare_and_healthcare_support_services": "700-7200-0272",
    "rental_and_leasing": "700-7200-0274",
    "repair_and_maintenance_services": "700-7200-0275",
    "printing_and_publishing": "700-7200-0276",
    "specialty_trade_contractors": "700-7200-0277",
    "towing_service": "700-7200-0278",
    "translation_and_interpretation_services": "700-7200-0279",
    "apartment_rental_flat_rental": "700-7200-0324",
    "b2b_sales_and_services": "700-7200-0328",
    "b2b_restaurant_services": "700-7200-0329",
    "aviation": "700-7200-0330",
    "interior_and_exterior_design": "700-7200-0342",
    "property_management": "700-7200-0344",
    "financial_investment_firm": "700-7200-0345",
    "business_facility": "700-7250-0136",
    "police_box": "700-7300-0110",
    "police_station": "700-7300-0111",
    "police_services_security": "700-7300-0112",
    "fire_department": "700-7300-0113",
    "ambulance_services": "700-7300-0280",
    "consumer_services": "700-7400-0000",
    "travel_agent_ticketing": "700-7400-0133",
    "dry_cleaning_and_laundry": "700-7400-0137",
    "attorney": "700-7400-0138",
    "boating": "700-7400-0140",
    "business_service": "700-7400-0141",
    "funeral_director": "700-7400-0142",
    "mover": "700-7400-0143",
    "photography": "700-7400-0144",
    "real_estate_services": "700-7400-0145",
    "repair_service": "700-7400-0146",
    "social_service": "700-7400-0147",
    "storage": "700-7400-0148",
    "tailor_and_alteration": "700-7400-0149",
    "tax_service": "700-7400-0150",
    "utilities": "700-7400-0151",
    "waste_and_sanitary": "700-7400-0152",
    "bicycle_service_and_maintenance": "700-7400-0281",
    "bill_payment_service": "700-7400-0282",
    "body_piercing_and_tattoos": "700-7400-0283",
    "wedding_services_and_bridal_studio": "700-7400-0284",
    "internet_cafe": "700-7400-0285",
    "kindergarten_and_childcare": "700-7400-0286",
    "maid_services": "700-7400-0287",
    "marriage_and_match_making_services": "700-7400-0288",
    "public_administration": "700-7400-0289",
    "wellness_center_and_services": "700-7400-0292",
    "pet_care": "700-7400-0293",
    "legal_services": "700-7400-0327",
    "tanning_salon": "700-7400-0343",
    "recycling_center": "700-7400-0352",
    "electrical": "700-7400-0365",
    "plumbing": "700-7400-0366",
    "post_office": "700-7450-0114",
    "tourist_information": "700-7460-0115",
    "fueling_station": "700-7600-0000",
    "petrol_gasoline_station": "700-7600-0116",
    "ev_charging_station": "700-7600-0322",
    "ev_battery_swap_station": "700-7600-0325",
    "hydrogen_fuel_station": "700-7600-0444",
    "automobile_dealership_new_cars": "700-7800-0118",
    "automobile_dealership_used_cars": "700-7800-0119",
    "motorcycle_dealership": "700-7800-0120",
    "car_repair_service": "700-7850-0000",
    "car_wash_detailing": "700-7850-0121",
    "car_repair": "700-7850-0122",
    "auto_parts": "700-7850-0123",
    "emission_testing": "700-7850-0124",
    "tire_repair": "700-7850-0125",
    "truck_repair": "700-7850-0126",
    "van_repair": "700-7850-0127",
    "road_assistance": "700-7850-0128",
    "automobile_club": "700-7850-0129",
    "rental_car_agency": "700-7851-0117",
    "truck_semi_dealer_services": "700-7900-0000",
    "truck_dealership": "700-7900-0130",
    "truck_parking": "700-7900-0131",
    "truck_stop_plaza": "700-7900-0132",
    "truck_wash": "700-7900-0323",
    "hospital_or_health_care_facility": "800-8000-0000",
    "dentist_dental_office": "800-8000-0154",
    "family_general_practice_physicians": "800-8000-0155",
    "psychiatric_institute": "800-8000-0156",
    "nursing_home": "800-8000-0157",
    "medical_services-clinics": "800-8000-0158",
    "hospital": "800-8000-0159",
    "optical": "800-8000-0161",
    "veterinarian": "800-8000-0162",
    "hospital_emergency_room": "800-8000-0325",
    "therapist": "800-8000-0340",
    "chiropractor": "800-8000-0341",
    "blood_bank": "800-8000-0367",
    "covid_19_testing_site": "800-8000-0400",
    "government_or_community_facility": "800-8100-0000",
    "city_hall": "800-8100-0163",
    "embassy": "800-8100-0164",
    "military_base": "800-8100-0165",
    "county_council": "800-8100-0168",
    "civic_community_center": "800-8100-0169",
    "court_house": "800-8100-0170",
    "government_office": "800-8100-0171",
    "border_crossing": "800-8100-0172",
    "education_facility": "800-8200-0000",
    "higher_education": "800-8200-0173",
    "school": "800-8200-0174",
    "training_and_development": "800-8200-0295",
    "coaching_institute": "800-8200-0360",
    "fine_arts": "800-8200-0361",
    "language_studies": "800-8200-0362",
    "other_library": "800-8300-0000",
    "library": "800-8300-0175",
    "event_spaces": "800-8400-0000",
    "banquet_hall": "800-8400-0139",
    "convention_exhibition_center": "800-8400-0176",
    "parking": "800-8500-0000",
    "parking_garage-parking_house": "800-8500-0177",
    "parking_lot": "800-8500-0178",
    "park_and_ride": "800-8500-0179",
    "motorcycle,_moped_and_scooter_parking": "800-8500-0200",
    "cellphone_parking_lot": "800-8500-0315",
    "sports_facility_venue": "800-8600-0000",
    "sports_complex_stadium": "800-8600-0180",
    "ice_skating_rink": "800-8600-0181",
    "swimming_pool": "800-8600-0182",
    "tennis_court": "800-8600-0183",
    "bowling_center": "800-8600-0184",
    "indoor_ski": "800-8600-0185",
    "hockey": "800-8600-0186",
    "racquetball_court": "800-8600-0187",
    "shooting_range": "800-8600-0188",
    "soccer_club": "800-8600-0189",
    "squash_court": "800-8600-0190",
    "fitness_health_club": "800-8600-0191",
    "indoor_sports": "800-8600-0192",
    "golf_course": "800-8600-0193",
    "golf_practice_range": "800-8600-0194",
    "race_track": "800-8600-0195",
    "sporting_instruction_and_camps": "800-8600-0196",
    "sports_activities": "800-8600-0197",
    "basketball": "800-8600-0199",
    "badminton": "800-8600-0200",
    "rugby": "800-8600-0314",
    "diving_center": "800-8600-0316",
    "bike_park": "800-8600-0376",
    "bmx_track": "800-8600-0377",
    "running_track": "800-8600-0381",
    "facilities": "800-8700-0000",
    "cemetery": "800-8700-0166",
    "crematorium": "800-8700-0167",
    "public_restroom-toilets": "800-8700-0198",
    "clubhouse": "800-8700-0296",
    "registration_office": "800-8700-0298",
    "outdoor_area_complex": "900-9200-0000",
    "industrial_zone": "900-9200-0218",
    "marina": "900-9200-0219",
    "rv_parks": "900-9200-0220",
    "collective_community": "900-9200-0299",
    "island": "900-9200-0301",
    "meeting_point": "900-9200-0386",
    "building": "900-9300-0000",
    "residential_area-building": "900-9300-0221",
}
#: Use this config for categoreis of browse API.
#: Example: for ``restaurant`` category use ``PLACES_CATEGORIES.restaurant``.
PLACES_CATEGORIES = PlacesCategories(**places_categories)
