#!/usr/bin/python
"""
    Copyright (c) 2016-2019, Jaguar0625, gimre, BloodyRookie, Tech Bureau, Corp.
    Copyright (c) 2020-present, Jaguar0625, gimre, BloodyRookie.

    This file is part of Catapult.

    Catapult is free software: you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Catapult is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with Catapult. If not, see <http://www.gnu.org/licenses/>.
"""

# pylint: disable=W0622,W0612,C0301,R0904

from __future__ import annotations
from .GeneratorUtils import GeneratorUtils
from .AddressDto import AddressDto
from .AmountDto import AmountDto
from .BlockFeeMultiplierDto import BlockFeeMultiplierDto
from .BlockHeaderBuilder import BlockHeaderBuilder
from .DifficultyDto import DifficultyDto
from .EntityTypeDto import EntityTypeDto
from .Hash256Dto import Hash256Dto
from .HeightDto import HeightDto
from .ImportanceBlockFooterBuilder import ImportanceBlockFooterBuilder
from .KeyDto import KeyDto
from .NetworkTypeDto import NetworkTypeDto
from .SignatureDto import SignatureDto
from .TimestampDto import TimestampDto
from .VrfProofBuilder import VrfProofBuilder

class ImportanceBlockHeaderBuilder(BlockHeaderBuilder):
    """Binary layout for an importance block header.

    Attributes:
        importanceBlockFooter: Importance block footer.
    """

    def __init__(self, signature: SignatureDto, signerPublicKey: KeyDto, version: int, network: NetworkTypeDto, type: EntityTypeDto, height: HeightDto, timestamp: TimestampDto, difficulty: DifficultyDto, generationHashProof: VrfProofBuilder, previousBlockHash: Hash256Dto, transactionsHash: Hash256Dto, receiptsHash: Hash256Dto, stateHash: Hash256Dto, beneficiaryAddress: AddressDto, feeMultiplier: BlockFeeMultiplierDto, votingEligibleAccountsCount: int, harvestingEligibleAccountsCount: int, totalVotingBalance: AmountDto, previousImportanceBlockHash: Hash256Dto):
        """Constructor.
        Args:
            signature: Entity signature.
            signerPublicKey: Entity signer's public key.
            version: Entity version.
            network: Entity network.
            type: Entity type.
            height: Block height.
            timestamp: Number of milliseconds elapsed since creation of nemesis block.
            difficulty: Block difficulty.
            generationHashProof: Generation hash proof.
            previousBlockHash: Previous block hash.
            transactionsHash: Hash of the transactions in this block.
            receiptsHash: Hash of the receipts generated by this block.
            stateHash: Hash of the global chain state at this block.
            beneficiaryAddress: Beneficiary address designated by harvester.
            feeMultiplier: Fee multiplier applied to block transactions.
            votingEligibleAccountsCount: Number of voting eligible accounts.
            harvestingEligibleAccountsCount: Number of harvesting eligible accounts.
            totalVotingBalance: Total balance eligible for voting.
            previousImportanceBlockHash: Previous importance block hash.
        """
        super().__init__(signature, signerPublicKey, version, network, type, height, timestamp, difficulty, generationHashProof, previousBlockHash, transactionsHash, receiptsHash, stateHash, beneficiaryAddress, feeMultiplier)
        self.importanceBlockFooter = ImportanceBlockFooterBuilder(votingEligibleAccountsCount, harvestingEligibleAccountsCount, totalVotingBalance, previousImportanceBlockHash)


    @classmethod
    def loadFromBinary(cls, payload: bytes) -> ImportanceBlockHeaderBuilder:
        """Creates an instance of ImportanceBlockHeaderBuilder from binary payload.
        Args:
            payload: Byte payload to use to serialize the object.
        Returns:
            Instance of ImportanceBlockHeaderBuilder.
        """
        bytes_ = bytes(payload)
        superObject = BlockHeaderBuilder.loadFromBinary(bytes_)
        bytes_ = bytes_[superObject.getSize():]

        importanceBlockFooter = ImportanceBlockFooterBuilder.loadFromBinary(bytes_)  # kind:CUSTOM1
        bytes_ = bytes_[importanceBlockFooter.getSize():]
        return ImportanceBlockHeaderBuilder(superObject.signature, superObject.signerPublicKey, superObject.version, superObject.network, superObject.type, superObject.height, superObject.timestamp, superObject.difficulty, superObject.generationHashProof, superObject.previousBlockHash, superObject.transactionsHash, superObject.receiptsHash, superObject.stateHash, superObject.beneficiaryAddress, superObject.feeMultiplier, importanceBlockFooter.votingEligibleAccountsCount, importanceBlockFooter.harvestingEligibleAccountsCount, importanceBlockFooter.totalVotingBalance, importanceBlockFooter.previousImportanceBlockHash)

    def getVotingEligibleAccountsCount(self) -> int:
        """Gets number of voting eligible accounts.
        Returns:
            Number of voting eligible accounts.
        """
        return self.importanceBlockFooter.getVotingEligibleAccountsCount()

    def getHarvestingEligibleAccountsCount(self) -> int:
        """Gets number of harvesting eligible accounts.
        Returns:
            Number of harvesting eligible accounts.
        """
        return self.importanceBlockFooter.getHarvestingEligibleAccountsCount()

    def getTotalVotingBalance(self) -> AmountDto:
        """Gets total balance eligible for voting.
        Returns:
            Total balance eligible for voting.
        """
        return self.importanceBlockFooter.getTotalVotingBalance()

    def getPreviousImportanceBlockHash(self) -> Hash256Dto:
        """Gets previous importance block hash.
        Returns:
            Previous importance block hash.
        """
        return self.importanceBlockFooter.getPreviousImportanceBlockHash()

    def getSize(self) -> int:
        """Gets the size of the object.
        Returns:
            Size in bytes.
        """
        size = super().getSize()
        size += self.importanceBlockFooter.getSize()
        return size

    def serialize(self) -> bytes:
        """Serializes self to bytes.
        Returns:
            Serialized bytes.
        """
        bytes_ = bytes()
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, super().serialize())
        bytes_ = GeneratorUtils.concatTypedArrays(bytes_, self.importanceBlockFooter.serialize())  # kind:CUSTOM
        return bytes_
    # end of class
