"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.dispatchMetric = exports.metricPeriod = exports.metricKey = void 0;
const core_1 = require("@aws-cdk/core");
const METRICKEY_SYMBOL = Symbol('@aws-cdk/aws-cloudwatch.MetricKey');
/**
 * Return a unique string representation for this metric.
 *
 * Can be used to determine as a hash key to determine if 2 Metric objects
 * represent the same metric. Excludes rendering properties.
 */
function metricKey(metric) {
    // Cache on the object itself. This is safe because Metric objects are immutable.
    if (metric.hasOwnProperty(METRICKEY_SYMBOL)) {
        return metric[METRICKEY_SYMBOL];
    }
    const parts = new Array();
    const conf = metric.toMetricConfig();
    if (conf.mathExpression) {
        parts.push(conf.mathExpression.expression);
        for (const id of Object.keys(conf.mathExpression.usingMetrics).sort()) {
            parts.push(id);
            parts.push(metricKey(conf.mathExpression.usingMetrics[id]));
        }
        if (conf.mathExpression.searchRegion) {
            parts.push(conf.mathExpression.searchRegion);
        }
        if (conf.mathExpression.searchAccount) {
            parts.push(conf.mathExpression.searchAccount);
        }
    }
    if (conf.metricStat) {
        parts.push(conf.metricStat.namespace);
        parts.push(conf.metricStat.metricName);
        for (const dim of conf.metricStat.dimensions || []) {
            parts.push(dim.name);
            parts.push(dim.value);
        }
        if (conf.metricStat.statistic) {
            parts.push(conf.metricStat.statistic);
        }
        if (conf.metricStat.period) {
            parts.push(`${conf.metricStat.period.toSeconds()}`);
        }
        if (conf.metricStat.region) {
            parts.push(conf.metricStat.region);
        }
        if (conf.metricStat.account) {
            parts.push(conf.metricStat.account);
        }
    }
    const ret = parts.join('|');
    Object.defineProperty(metric, METRICKEY_SYMBOL, { value: ret });
    return ret;
}
exports.metricKey = metricKey;
/**
 * Return the period of a metric
 *
 * For a stat metric, return the immediate period.
 *
 * For an expression metric, all metrics used in it have been made to have the
 * same period, so we return the period of the first inner metric.
 */
function metricPeriod(metric) {
    return dispatchMetric(metric, {
        withStat(stat) {
            return stat.period;
        },
        withExpression() {
            return metric.period || core_1.Duration.minutes(5);
        },
    });
}
exports.metricPeriod = metricPeriod;
/**
 * Given a metric object, inspect it and call the correct function for the type of output
 *
 * In addition to the metric object itself, takes a callback object with two
 * methods, to be invoked for the particular type of metric.
 *
 * If the metric represent a metric query (nominally generated through an
 * instantiation of `Metric` but can be generated by any class that implements
 * `IMetric`) a particular field in its `toMetricConfig()` output will be set
 * (to wit, `metricStat`) and the `withStat()` callback will be called with
 * that object.
 *
 * If the metric represents an expression (usually by instantiating `MathExpression`
 * but users can implement `IMetric` arbitrarily) the `mathExpression` field
 * will be set in the object returned from `toMetricConfig` and the callback
 * called `withExpression` will be applied to that object.
 *
 * Will return the values returned by the callbacks.
 *
 * To be used as such:
 *
 * ```ts
 * const ret = dispatchMetric(someMetric, {
 *   withStat(stat) {
 *     // do something with stat
 *     return 1;
 *   },
 *   withExpression(expr) {
 *     // do something with expr
 *     return 2;
 *   },
 * });
 * ```
 *
 * This function encapsulates some type analysis that would otherwise have to be
 * repeated in all places where code needs to make a distinction on the type
 * of metric object that is being passed.
 */
// eslint-disable-next-line max-len
function dispatchMetric(metric, fns) {
    const conf = metric.toMetricConfig();
    if (conf.metricStat && conf.mathExpression) {
        throw new Error('Metric object must not produce both \'metricStat\' and \'mathExpression\'');
    }
    else if (conf.metricStat) {
        return fns.withStat(conf.metricStat, conf);
    }
    else if (conf.mathExpression) {
        return fns.withExpression(conf.mathExpression, conf);
    }
    else {
        throw new Error('Metric object must have either \'metricStat\' or \'mathExpression\'');
    }
}
exports.dispatchMetric = dispatchMetric;
//# sourceMappingURL=data:application/json;base64,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