import * as cdk from '@aws-cdk/core';
import { IAlarm } from './alarm-base';
import { IMetric } from './metric-types';
import { ConcreteWidget } from './widget';
/**
 * Basic properties for widgets that display metrics.
 *
 * @stability stable
 */
export interface MetricWidgetProps {
    /**
     * Title for the graph.
     *
     * @default - None
     * @stability stable
     */
    readonly title?: string;
    /**
     * The region the metrics of this graph should be taken from.
     *
     * @default - Current region
     * @stability stable
     */
    readonly region?: string;
    /**
     * Width of the widget, in a grid of 24 units wide.
     *
     * @default 6
     * @stability stable
     */
    readonly width?: number;
    /**
     * Height of the widget.
     *
     * @default - 6 for Alarm and Graph widgets.
     * 3 for single value widgets where most recent value of a metric is displayed.
     * @stability stable
     */
    readonly height?: number;
}
/**
 * Properties for a Y-Axis.
 *
 * @stability stable
 */
export interface YAxisProps {
    /**
     * The min value.
     *
     * @default 0
     * @stability stable
     */
    readonly min?: number;
    /**
     * The max value.
     *
     * @default - No maximum value
     * @stability stable
     */
    readonly max?: number;
    /**
     * The label.
     *
     * @default - No label
     * @stability stable
     */
    readonly label?: string;
    /**
     * Whether to show units.
     *
     * @default true
     * @stability stable
     */
    readonly showUnits?: boolean;
}
/**
 * Properties for an AlarmWidget.
 *
 * @stability stable
 */
export interface AlarmWidgetProps extends MetricWidgetProps {
    /**
     * The alarm to show.
     *
     * @stability stable
     */
    readonly alarm: IAlarm;
    /**
     * Left Y axis.
     *
     * @default - No minimum or maximum values for the left Y-axis
     * @stability stable
     */
    readonly leftYAxis?: YAxisProps;
}
/**
 * Display the metric associated with an alarm, including the alarm line.
 *
 * @stability stable
 */
export declare class AlarmWidget extends ConcreteWidget {
    private readonly props;
    /**
     * @stability stable
     */
    constructor(props: AlarmWidgetProps);
    /**
     * Return the widget JSON for use in the dashboard.
     *
     * @stability stable
     */
    toJson(): any[];
}
/**
 * Types of view.
 *
 * @stability stable
 */
export declare enum GraphWidgetView {
    /**
     * Display as a line graph.
     *
     * @stability stable
     */
    TIME_SERIES = "timeSeries",
    /**
     * Display as a bar graph.
     *
     * @stability stable
     */
    BAR = "bar",
    /**
     * Display as a pie graph.
     *
     * @stability stable
     */
    PIE = "pie"
}
/**
 * Properties for a GraphWidget.
 *
 * @stability stable
 */
export interface GraphWidgetProps extends MetricWidgetProps {
    /**
     * Metrics to display on left Y axis.
     *
     * @default - No metrics
     * @stability stable
     */
    readonly left?: IMetric[];
    /**
     * Metrics to display on right Y axis.
     *
     * @default - No metrics
     * @stability stable
     */
    readonly right?: IMetric[];
    /**
     * Annotations for the left Y axis.
     *
     * @default - No annotations
     * @stability stable
     */
    readonly leftAnnotations?: HorizontalAnnotation[];
    /**
     * Annotations for the right Y axis.
     *
     * @default - No annotations
     * @stability stable
     */
    readonly rightAnnotations?: HorizontalAnnotation[];
    /**
     * Whether the graph should be shown as stacked lines.
     *
     * @default false
     * @stability stable
     */
    readonly stacked?: boolean;
    /**
     * Left Y axis.
     *
     * @default - None
     * @stability stable
     */
    readonly leftYAxis?: YAxisProps;
    /**
     * Right Y axis.
     *
     * @default - None
     * @stability stable
     */
    readonly rightYAxis?: YAxisProps;
    /**
     * Position of the legend.
     *
     * @default - bottom
     * @stability stable
     */
    readonly legendPosition?: LegendPosition;
    /**
     * Whether the graph should show live data.
     *
     * @default false
     * @stability stable
     */
    readonly liveData?: boolean;
    /**
     * Display this metric.
     *
     * @default TimeSeries
     * @stability stable
     */
    readonly view?: GraphWidgetView;
    /**
     * Whether to show the value from the entire time range. Only applicable for Bar and Pie charts.
     *
     * If false, values will be from the most recent period of your chosen time range;
     * if true, shows the value from the entire time range.
     *
     * @default false
     * @stability stable
     */
    readonly setPeriodToTimeRange?: boolean;
    /**
     * The default period for all metrics in this widget.
     *
     * The period is the length of time represented by one data point on the graph.
     * This default can be overridden within each metric definition.
     *
     * @default cdk.Duration.seconds(300)
     * @stability stable
     */
    readonly period?: cdk.Duration;
    /**
     * The default statistic to be displayed for each metric.
     *
     * This default can be overridden within the definition of each individual metric
     *
     * @default - The statistic for each metric is used
     * @stability stable
     */
    readonly statistic?: string;
}
/**
 * A dashboard widget that displays metrics.
 *
 * @stability stable
 */
export declare class GraphWidget extends ConcreteWidget {
    private readonly props;
    private readonly leftMetrics;
    private readonly rightMetrics;
    /**
     * @stability stable
     */
    constructor(props: GraphWidgetProps);
    /**
     * Add another metric to the left Y axis of the GraphWidget.
     *
     * @param metric the metric to add.
     * @stability stable
     */
    addLeftMetric(metric: IMetric): void;
    /**
     * Add another metric to the right Y axis of the GraphWidget.
     *
     * @param metric the metric to add.
     * @stability stable
     */
    addRightMetric(metric: IMetric): void;
    /**
     * Return the widget JSON for use in the dashboard.
     *
     * @stability stable
     */
    toJson(): any[];
}
/**
 * Properties for a SingleValueWidget.
 *
 * @stability stable
 */
export interface SingleValueWidgetProps extends MetricWidgetProps {
    /**
     * Metrics to display.
     *
     * @stability stable
     */
    readonly metrics: IMetric[];
    /**
     * Whether to show the value from the entire time range.
     *
     * @default false
     * @stability stable
     */
    readonly setPeriodToTimeRange?: boolean;
    /**
     * Whether to show as many digits as can fit, before rounding.
     *
     * @default false
     * @stability stable
     */
    readonly fullPrecision?: boolean;
}
/**
 * A dashboard widget that displays the most recent value for every metric.
 *
 * @stability stable
 */
export declare class SingleValueWidget extends ConcreteWidget {
    private readonly props;
    /**
     * @stability stable
     */
    constructor(props: SingleValueWidgetProps);
    /**
     * Return the widget JSON for use in the dashboard.
     *
     * @stability stable
     */
    toJson(): any[];
}
/**
 * Horizontal annotation to be added to a graph.
 *
 * @stability stable
 */
export interface HorizontalAnnotation {
    /**
     * The value of the annotation.
     *
     * @stability stable
     */
    readonly value: number;
    /**
     * Label for the annotation.
     *
     * @default - No label
     * @stability stable
     */
    readonly label?: string;
    /**
     * The hex color code, prefixed with '#' (e.g. '#00ff00'), to be used for the annotation. The `Color` class has a set of standard colors that can be used here.
     *
     * @default - Automatic color
     * @stability stable
     */
    readonly color?: string;
    /**
     * Add shading above or below the annotation.
     *
     * @default No shading
     * @stability stable
     */
    readonly fill?: Shading;
    /**
     * Whether the annotation is visible.
     *
     * @default true
     * @stability stable
     */
    readonly visible?: boolean;
}
/**
 * Fill shading options that will be used with an annotation.
 *
 * @stability stable
 */
export declare enum Shading {
    /**
     * Don't add shading.
     *
     * @stability stable
     */
    NONE = "none",
    /**
     * Add shading above the annotation.
     *
     * @stability stable
     */
    ABOVE = "above",
    /**
     * Add shading below the annotation.
     *
     * @stability stable
     */
    BELOW = "below"
}
/**
 * A set of standard colours that can be used in annotations in a GraphWidget.
 *
 * @stability stable
 */
export declare class Color {
    /**
     * blue - hex #1f77b4.
     *
     * @stability stable
     */
    static readonly BLUE = "#1f77b4";
    /**
     * brown - hex #8c564b.
     *
     * @stability stable
     */
    static readonly BROWN = "#8c564b";
    /**
     * green - hex #2ca02c.
     *
     * @stability stable
     */
    static readonly GREEN = "#2ca02c";
    /**
     * grey - hex #7f7f7f.
     *
     * @stability stable
     */
    static readonly GREY = "#7f7f7f";
    /**
     * orange - hex #ff7f0e.
     *
     * @stability stable
     */
    static readonly ORANGE = "#ff7f0e";
    /**
     * pink - hex #e377c2.
     *
     * @stability stable
     */
    static readonly PINK = "#e377c2";
    /**
     * purple - hex #9467bd.
     *
     * @stability stable
     */
    static readonly PURPLE = "#9467bd";
    /**
     * red - hex #d62728.
     *
     * @stability stable
     */
    static readonly RED = "#d62728";
}
/**
 * The position of the legend on a GraphWidget.
 *
 * @stability stable
 */
export declare enum LegendPosition {
    /**
     * Legend appears below the graph (default).
     *
     * @stability stable
     */
    BOTTOM = "bottom",
    /**
     * Add shading above the annotation.
     *
     * @stability stable
     */
    RIGHT = "right",
    /**
     * Add shading below the annotation.
     *
     * @stability stable
     */
    HIDDEN = "hidden"
}
