# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['plotly_resampler',
 'plotly_resampler.aggregation',
 'plotly_resampler.aggregation.algorithms',
 'plotly_resampler.figure_resampler']

package_data = \
{'': ['*']}

install_requires = \
['Flask-Cors>=3.0.10,<4.0.0',
 'dash>=2.2.0,<3.0.0',
 'jupyter-dash>=0.4.2',
 'numpy>=1.14',
 'pandas>=1.3.5,<2.0.0',
 'plotly>=5.5.0,<6.0.0',
 'trace-updater>=0.0.8']

setup_kwargs = {
    'name': 'plotly-resampler',
    'version': '0.8.2rc1',
    'description': 'Visualizing large time series with plotly',
    'long_description': '<p align="center">\n    <a href="#readme">\n        <img alt="Plotly-Resampler logo" src="https://raw.githubusercontent.com/predict-idlab/plotly-resampler/main/docs/sphinx/_static/logo.svg" width=65%>\n    </a>\n</p>\n\n[![PyPI Latest Release](https://img.shields.io/pypi/v/plotly-resampler.svg)](https://pypi.org/project/plotly-resampler/)\n[![support-version](https://img.shields.io/pypi/pyversions/plotly-resampler)](https://img.shields.io/pypi/pyversions/plotly-resampler)\n[![codecov](https://img.shields.io/codecov/c/github/predict-idlab/plotly-resampler?logo=codecov)](https://codecov.io/gh/predict-idlab/plotly-resampler)\n[![Code quality](https://img.shields.io/lgtm/grade/python/github/predict-idlab/plotly-resampler?label=code%20quality&logo=lgtm)](https://lgtm.com/projects/g/predict-idlab/plotly-resampler/context:python)\n[![Downloads](https://pepy.tech/badge/plotly-resampler)](https://pepy.tech/project/plotly-resampler)\n[![PRs Welcome](https://img.shields.io/badge/PRs-welcome-brightgreen.svg?)](http://makeapullrequest.com)\n[![Documentation](https://github.com/predict-idlab/plotly-resampler/actions/workflows/deploy-docs.yml/badge.svg)](https://github.com/predict-idlab/plotly-resampler/actions/workflows/deploy-docs.yml)\n[![Testing](https://github.com/predict-idlab/plotly-resampler/actions/workflows/test.yml/badge.svg)](https://github.com/predict-idlab/plotly-resampler/actions/workflows/test.yml)\n\n\n<!-- [![Downloads](https://pepy.tech/badge/plotly-resampler)](https://pepy.tech/project/plotly-resampler) -->\n\n> `plotly_resampler`: visualize large sequential data by **adding resampling functionality to Plotly figures**\n\n[Plotly](https://github.com/plotly/plotly.py) is an awesome interactive visualization library, however it can get pretty slow when a lot of data points are visualized (100 000+ datapoints). This library solves this by downsampling (aggregating) the data respective to the view and then plotting the aggregated points. When you interact with the plot (panning, zooming, ...), callbacks are used to aggregate data and update the figure.\n\n![basic example gif](https://raw.githubusercontent.com/predict-idlab/plotly-resampler/main/docs/sphinx/_static/basic_example.gif)\n\n\nIn [this Plotly-Resampler demo](https://github.com/predict-idlab/plotly-resampler/blob/main/examples/basic_example.ipynb) over `110,000,000` data points are visualized!\n\n<!-- These dynamic aggregation callbacks are realized with: -->\n<!-- * [Dash](https://github.com/plotly/dash) when a `go.Figure` object is wrapped with dynamic aggregation functionality, see example ⬆️. -->\n<!-- * The [FigureWidget.layout.on_change](https://plotly.com/python-api-reference/generated/plotly.html?highlight=on_change#plotly.basedatatypes.BasePlotlyType.on_change) method, when a `go.FigureWidget` is used within a `.ipynb` environment. -->\n\n<!-- #### Useful links -->\n\n<!-- - [Documentation]() work in progress 🚧  -->\n<!-- - [Example notebooks](https://github.com/predict-idlab/plotly-resampler/tree/main/examples/) -->\n\n### Installation\n\n| [**pip**](https://pypi.org/project/plotly_resampler/) | `pip install plotly-resampler` |\n| ---| ----|\n<!-- | [**conda**](https://anaconda.org/conda-forge/plotly_resampler/) | `conda install -c conda-forge plotly_resampler` | -->\n\n<br>\n\n## Usage\n\n**Add dynamic aggregation** to your plotly Figure _(unfold your fitting use case)_\n* 🤖 <b>Automatically</b> _(minimal code overhead)_:\n  <details><summary>Use the <code>register_plotly_resampler</code> function</summary>\n    <br>\n\n    1. Import and call the `register_plotly_resampler` method\n    2. Just use your regular graph construction code\n\n    * **code example**:\n      ```python\n      import plotly.graph_objects as go; import numpy as np\n      from plotly_resampler import register_plotly_resampler\n\n      # Call the register function once and all Figures/FigureWidgets will be wrapped\n      # according to the register_plotly_resampler its `mode` argument\n      register_plotly_resampler(mode=\'auto\')\n\n      x = np.arange(1_000_000)\n      noisy_sin = (3 + np.sin(x / 200) + np.random.randn(len(x)) / 10) * x / 1_000\n\n\n      # auto mode: when working in an IPython environment, this will automatically be a \n      # FigureWidgetResampler else, this will be an FigureResampler\n      f = go.Figure()\n      f.add_trace({"y": noisy_sin + 2, "name": "yp2"})\n      f\n      ```\n\n    > **Note**: This wraps **all** plotly graph object figures with a \n    > `FigureResampler` | `FigureWidgetResampler`. This can thus also be \n    > used for the `plotly.express` interface. 🎉\n\n  </details>\n\n* 👷 <b>Manually</b> _(higher data aggregation configurability, more speedup possibilities)_:\n  <details>\n    <summary>Within a <b><i>jupyter</i></b> environment without creating a <i>web application</i></summary>\n    <br>\n\n    1. wrap the plotly Figure with `FigureWidgetResampler`\n    2. output the `FigureWidgetResampler` instance in a cell\n\n    * **code example**:\n      ```python\n      import plotly.graph_objects as go; import numpy as np\n      from plotly_resampler import FigureResampler, FigureWidgetResampler\n\n      x = np.arange(1_000_000)\n      noisy_sin = (3 + np.sin(x / 200) + np.random.randn(len(x)) / 10) * x / 1_000\n\n      # OPTION 1 - FigureWidgetResampler: dynamic aggregation via `FigureWidget.layout.on_change`\n      fig = FigureWidgetResampler(go.Figure())\n      fig.add_trace(go.Scattergl(name=\'noisy sine\', showlegend=True), hf_x=x, hf_y=noisy_sin)\n\n      fig\n      ```\n  </details>\n  <details>\n    <summary>Using a <b><i>web-application</i></b> with <b><a href="https://github.com/plotly/dash">dash</a></b> callbacks</summary>\n    <br>\n\n    1. wrap the plotly Figure with `FigureResampler`\n    2. call `.show_dash()` on the `Figure`\n\n    * **code example**:\n      ```python\n      import plotly.graph_objects as go; import numpy as np\n      from plotly_resampler import FigureResampler, FigureWidgetResampler\n\n      x = np.arange(1_000_000)\n      noisy_sin = (3 + np.sin(x / 200) + np.random.randn(len(x)) / 10) * x / 1_000\n\n      # OPTION 2 - FigureResampler: dynamic aggregation via a Dash web-app\n      fig = FigureResampler(go.Figure())\n      fig.add_trace(go.Scattergl(name=\'noisy sine\', showlegend=True), hf_x=x, hf_y=noisy_sin)\n\n      fig.show_dash(mode=\'inline\')\n      ```\n\n  </details>\n  <br>\n\n  > **Tip** 💡:\n   > For significant faster initial loading of the Figure, we advise to wrap the \n   > constructor of the plotly Figure and add the trace data as `hf_x` and `hf_y`\n\n<br>\n\n> **Note**:\n> Any plotly Figure can be wrapped with `FigureResampler` and `FigureWidgetResampler`! 🎉\n> But, (obviously) only the scatter traces will be resampled.\n\n\n\n\n<br>\n<details><summary>Features</summary>\n\n  * **Convenient** to use:\n    * just add either\n      * `register_plotly_resampler` function to your notebook with the best suited `mode` argument.\n      * `FigureResampler` decorator around a plotly Figure and call `.show_dash()`\n      * `FigureWidgetResampler` decorator around a plotly Figure and output the instance in a cell\n    * allows all other plotly figure construction flexibility to be used!\n  * **Environment-independent**\n    * can be used in Jupyter, vscode-notebooks, Pycharm-notebooks, Google Colab, and even as application (on a server)\n  * Interface for **various aggregation algorithms**:\n    * ability to develop or select your preferred sequence aggregation method\n</details>\n\n### Important considerations & tips\n\n* When running the code on a server, you should forward the port of the `FigureResampler.show_dash()` method to your local machine.<br>\n  **Note** that you can add dynamic aggregation to plotly figures with the `FigureWidgetResampler` wrapper without needing to forward a port!\n* The `FigureWidgetResampler` *uses the IPython main thread* for its data aggregation functionality, so when this main thread is occupied, no resampling logic can be executed. For example; if you perform long computations within your notebook, the kernel will be occupied during these computations, and will only execute the resampling operations that take place during these computations after finishing that computation.\n* In general, when using downsampling one should be aware of (possible) [aliasing](https://en.wikipedia.org/wiki/Aliasing) effects.\n  The <b style="color:orange">[R]</b> in the legend indicates when the corresponding trace is being resampled (and thus possibly distorted) or not. Additionally, the `~<range>` suffix represent the mean aggregation bin size in terms of the sequence index.\n* The plotly **autoscale** event (triggered by the autoscale button or a double-click within the graph), **does not reset the axes but autoscales the current graph-view** of plotly-resampler figures. This design choice was made as it seemed more intuitive for the developers to support this behavior with double-click than the default axes-reset behavior. The graph axes can ofcourse be resetted by using the `reset_axis` button.  If you want to give feedback and discuss this further with the developers, see issue [#49](https://github.com/predict-idlab/plotly-resampler/issues/49).\n\n## Cite\n\nPaper (preprint): https://arxiv.org/abs/2206.08703\n\n```latex\n@misc{https://doi.org/10.48550/arxiv.2206.08703,\n  author = {Van Der Donckt, Jonas and Van Der Donckt, Jeroen and Deprost, Emiel and Van Hoecke, Sofie},\n  title = {Plotly-Resampler: Effective Visual Analytics for Large Time Series},\n  year = {2022},\n  doi = {10.48550/ARXIV.2206.08703},\n  url = {https://arxiv.org/abs/2206.08703},\n  publisher = {arXiv},\n}\n```\n\n## Future work 🔨\n\n- [x] Support `.add_traces()` (currently only `.add_trace` is supported)\n- [ ] Support `hf_color` and `hf_markersize`, see [#50](https://github.com/predict-idlab/plotly-resampler/pull/50)\n- [ ] Create C bindings for our EfficientLTTB algorithm.\n\n<br>\n\n---\n\n<p align="center">\n👤 <i>Jonas Van Der Donckt, Jeroen Van Der Donckt, Emiel Deprost</i>\n</p>\n',
    'author': 'Jonas Van Der Donckt',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/predict-idlab/plotly-resampler',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.1,<3.11',
}
from build import *
build(setup_kwargs)

setup(**setup_kwargs)
