"use strict";
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'can create receipt rules with second after first'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.ReceiptRuleSet(stack, 'RuleSet', {
            rules: [
                {
                    receiptRuleName: 'FirstRule',
                },
                {
                    enabled: false,
                    receiptRuleName: 'SecondRule',
                    recipients: ['hello@aws.com'],
                    scanEnabled: true,
                    tlsPolicy: lib_1.TlsPolicy.REQUIRE,
                },
            ],
        });
        // THEN
        assert_1.expect(stack).toMatch({
            'Resources': {
                'RuleSetE30C6C48': {
                    'Type': 'AWS::SES::ReceiptRuleSet',
                },
                'RuleSetRule023C3B8E1': {
                    'Type': 'AWS::SES::ReceiptRule',
                    'Properties': {
                        'Rule': {
                            'Name': 'FirstRule',
                            'Enabled': true,
                        },
                        'RuleSetName': {
                            'Ref': 'RuleSetE30C6C48',
                        },
                    },
                },
                'RuleSetRule117041B57': {
                    'Type': 'AWS::SES::ReceiptRule',
                    'Properties': {
                        'Rule': {
                            'Enabled': false,
                            'Name': 'SecondRule',
                            'Recipients': [
                                'hello@aws.com',
                            ],
                            'ScanEnabled': true,
                            'TlsPolicy': 'Require',
                        },
                        'RuleSetName': {
                            'Ref': 'RuleSetE30C6C48',
                        },
                        'After': {
                            'Ref': 'RuleSetRule023C3B8E1',
                        },
                    },
                },
            },
        });
        test.done();
    },
    'import receipt rule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const receiptRule = lib_1.ReceiptRule.fromReceiptRuleName(stack, 'ImportedRule', 'MyRule');
        const receiptRuleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        receiptRuleSet.addRule('MyRule', {
            after: receiptRule,
        });
        // THEN
        assert_1.expect(stack).toMatch({
            'Resources': {
                'RuleSetE30C6C48': {
                    'Type': 'AWS::SES::ReceiptRuleSet',
                },
                'RuleSetMyRule60B1D107': {
                    'Type': 'AWS::SES::ReceiptRule',
                    'Properties': {
                        'Rule': {
                            'Enabled': true,
                        },
                        'RuleSetName': {
                            'Ref': 'RuleSetE30C6C48',
                        },
                        'After': 'MyRule',
                    },
                },
            },
        });
        test.done();
    },
    'can add actions in rule props'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const ruleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        ruleSet.addRule('Rule', {
            actions: [
                {
                    bind: () => ({
                        stopAction: {
                            scope: 'RuleSet',
                        },
                    }),
                },
            ],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            'Rule': {
                'Actions': [
                    {
                        'StopAction': {
                            'Scope': 'RuleSet',
                        },
                    },
                ],
                'Enabled': true,
            },
            'RuleSetName': {
                'Ref': 'RuleSetE30C6C48',
            },
        }));
        test.done();
    },
    'can add action with addAction'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const ruleSet = new lib_1.ReceiptRuleSet(stack, 'RuleSet');
        const rule = ruleSet.addRule('Rule');
        rule.addAction({
            bind: () => ({
                stopAction: {
                    scope: 'RuleSet',
                },
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SES::ReceiptRule', {
            'Rule': {
                'Actions': [
                    {
                        'StopAction': {
                            'Scope': 'RuleSet',
                        },
                    },
                ],
                'Enabled': true,
            },
            'RuleSetName': {
                'Ref': 'RuleSetE30C6C48',
            },
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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